/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: display.hh,v 1.22.2.1 2003/09/19 17:46:05 dheck Exp $
 */
#ifndef DISPLAY_HH
#define DISPLAY_HH

#include "fwd.hh"
#include "enigma.hh"
#include "player.hh"
#include "options.hh"
#include "px/px.hh"
#include <string>

//----------------------------------------
// Definition of models
//----------------------------------------
namespace display
{
    using std::string;

    class DisplayLayer;
    class ModelLayer;

    /* Animations can be prompted to invoke a callback on completion.
       Note that you may not delete or replace models other than the
       one that induced the callback from inside the callback
       -- use a timer or a flag to do this. */
    class ModelCallback {
    public:
        virtual ~ModelCallback() {}
        virtual void animcb() = 0;
    };

    class Animation {
    public:
        virtual ~Animation() {}
        virtual void set_callback (ModelCallback *) {}
        virtual void reverse() {}

	virtual bool is_garbage() const { return false; }
        virtual void tick(double /*dtime*/) {}
        virtual bool has_changed(px::Rect &/*changed_region*/) { return false; }
    };

    class Model : public Animation {
    public:

        virtual void draw(px::GC &/*gc*/, int /*x*/, int /*y*/) {}
        virtual void draw_shadow(px::GC &/*gc*/, int /*x*/, int /*y*/) {}

        virtual Model *get_shadow() const { return 0; }

        virtual void expose (ModelLayer */*ml*/, const px::V2 &/*pos*/) {}
        virtual void update (ModelLayer */*ml*/, double /*dtime*/) {}
        virtual void remove (ModelLayer */*ml*/) {}

        virtual bool is_garbage() const { return false; }
        virtual void tick(double /*dtime*/) {}
        virtual bool has_changed(px::Rect &/*changed_region*/) { return false; }

        virtual Model *clone() = 0;
    };

    void InitModels();
    void ShutdownModels();

    void DefineModel (const char *name, Model *m);
    Model * MakeModel (const std::string &name);

    void DefineImage (const char *name, const char *fname,
                     int xoff, int yoff);
    void DefineSubImage (const char *name, const char *fname,
                         int xoff, int yoff, px::Rect r);
    void DefineRandModel (const char *name, int n, const char **names);
    void DefineShadedModel (const char *name, const char *model, const char *shade);
    void DefineOverlayImage (const char *name, int n, const char **images);
    void DefineComposite (const char *name, const char *bgname, const char *fgname);
    void DefineAnim (const char *name, bool loop_p);
    void AddFrame (const char *name, const char *model, double time);
    void DefineAlias (const char *name, const char *othername);
}

//----------------------------------------
// Models for on the grid
//----------------------------------------
namespace display
{
    using enigma::GridPos;
    using enigma::GridLayer;
    using enigma::GridLoc;

    Model*  SetModel (const GridLoc & l, const string &modelname);
    Model*  SetModel (const GridLoc & l, Model *m);
    void    KillModel (const GridLoc & l);
    Model*  GetModel (const GridLoc & l);
    Model*  YieldModel (const GridLoc & l);
}

//----------------------------------------
// Sprites (movable models)
//----------------------------------------
namespace display
{
    enum SpriteLayer
    {
        SPRITE_ACTOR, SPRITE_EFFECT, SPRITE_DEBRIS
    };

    enum FollowMode
    {
        FOLLOW_NONE,            // Don't follow any sprite
        FOLLOW_SCROLLING,       // Scroll the screen
        FOLLOW_SCREEN           // Flip the screen region
    };

    typedef unsigned int SpriteId;

    class DL_Sprites;

    class SpriteHandle {
        DL_Sprites *layer;
        unsigned id;
    public:
        SpriteHandle (DL_Sprites *l, unsigned spriteid)
        : layer(l), id(spriteid)
        {}
        SpriteHandle();

        void kill();
        void move (const px::V2 &newpos) const;
        void replace_model (Model *m) const;
        Model *get_model () const;
        void set_callback (ModelCallback *cb) const;
    };

    void AddEffect (const px::V2 &pos, const char *modelname);

    /*! Create a new sprite.  If modelname==0, the sprite is considered
      invisible.  */
    SpriteHandle AddSprite (const px::V2 &pos, const char *modelname=0);


    void SetReferencePoint (const px::V2 &point);
    void SetFollowMode (FollowMode m);
    void FocusReferencePoint();
    void GetReferencePointCoordinates(int *x, int *y);
}

//----------------------------------------
// Rubber bands
//----------------------------------------
namespace  display
{
    class DL_Lines;

    class RubberHandle {
    public:
        RubberHandle (DL_Lines *layer=0, unsigned id=0);
        operator unsigned() { return id; }

        void update_first (const px::V2 &p1);
        void update_second (const px::V2 &p2);
        void kill();

        DL_Lines *line_layer;
        unsigned id;
    };

    RubberHandle AddRubber (const px::V2 &p1, const px::V2 &p2);
}

//----------------------------------------
// Status bar management
//----------------------------------------
namespace display
{
    enum TextMode {
        TEXT_SCROLLING,         // Scroll text from right to left
        TEXT_2SECONDS,          // Show text centered, for 2 seconds
        TEXT_5SECONDS,          // Show text centered, for 5 seconds
        TEXT_STATIC             // Show text centered, indefinitely
    };

    using player::Inventory;

    class StatusBar {
    public:
        virtual ~StatusBar() {}
        virtual void set_inventory (Inventory *inv)    = 0;
        virtual void update_inventory (Inventory *inv) = 0;

        virtual void show_text (const std::string &str,
                                TextMode m = TEXT_SCROLLING,
                                bool may_interrupt=false) = 0;
        virtual void hide_text()                                                     = 0;

        virtual void show_move_counter (bool active) = 0;
        virtual void show_odometer (bool active) = 0;
    };

    StatusBar *GetStatusBar();
}

//----------------------------------------
// Interface to the display engine
//----------------------------------------
namespace display
{
//     extern options::BoolOption FullScreen;

    enum ShadowType
    {
        SHADOW_NONE,
        SHADOW_STIPPLE,
        SHADOW_ALPHA
    };
    void SetShadowType(ShadowType st);


    enum DisplayFlags
    {
        SHOW_FLOOR     = 0x01,
        SHOW_STONES    = 0x02,
        SHOW_ITEMS     = 0x04,
        SHOW_SHADES    = 0x08,
        SHOW_SPRITES   = 0x10,
	SHOW_TIME      = 0x20,
	SHOW_INVENTORY = 0x40,
        SHOW_ALL       = 0x7f
    };

    void ToggleFlag(DisplayFlags flag);


    void Init();
    void Shutdown();

    void            NewWorld (int w, int h);
    void            ResizeGameArea (int w, int h);
    const px::Rect& GetGameArea ();

    void DrawAll(px::GC &gc);
    void RedrawAll(px::Screen *sfc);
    void Redraw(px::Screen *sfc);
    void Tick(double dtime);

    /* Reload all models.  This is similar to calling Init() and
       Shutdown(), but it restarts the current level.  [This is more
       or less a hack to make incremental changes to the models
       easier.] */
    void ReloadModels();
}

#endif
