/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: objects.cc,v 1.117.2.2 2003/10/06 19:32:08 dheck Exp $
 */
#include "game.hh"
#include "objects.hh"
#include "object_mixins.hh"
#include "display.hh"
#include "player.hh"
#include "lua.hh"
#include "sound.hh"
#include "actors.hh"

#include "px/tools.hh"
#include "px/dict.hh"

#include <algorithm>
#include <string>
#include <cstdlib>
#include <cstdarg>
#include <iostream>
#include <iomanip>
#include <cassert>

using namespace std;
using namespace world;
using namespace enigma;

// remove comment from define below to switch on verbose messaging
// note: VERBOSE_MESSAGES is defined in multiple source files!
// #define VERBOSE_MESSAGES

/* This function is used by all triggers, switches etc. that perform
   some particular action when activated (like opening doors or
   switching lasers on and off). */
void
world::PerformAction(Object *o, bool onoff)
{
    string action = "idle";
    string target;

    o->string_attrib("action", &action);
    o->string_attrib("target", &target);

#if defined(VERBOSE_MESSAGES)
    o->warning("PerformAction action=%s target=%s", action.c_str(), target.c_str());
#endif // VERBOSE_MESSAGES

    if (action == "callback") {
        lua::CallFunc(target.c_str(), Value(onoff));
    }
    else if (action == "signal") {
        world::EmitSignals (o, onoff);
    }
    else if (Object *t = GetNamedObject(target)) {
        SendMessage(t, action, Value(onoff));
    }
    else if (action != "idle") {
        fprintf(stderr, "Unknown target '%s' for action '%s'\n", target.c_str(), action.c_str());
    }
}

//----------------------------------------
// Object impl.
//----------------------------------------

Object::Object(const char *kind)
{
    set_attrib("kind", Value(kind));
}

void
Object::message(const string& /*msg*/, const Value &/*val*/)
{
}

const char *
Object::get_kind() const
{
    const Value *v = get_attrib("kind");
    assert(v && v->get_type()==Value::STRING);
    return v->get_string();
}

namespace {

    // string_match accepts simple wildcards
    // '?' means 'any character'
    // '*' means '0 or more characters'

    bool string_match(const char *str, const char *templ) {
        while (1) {
            char t = *templ++;
            char s = *str++;

            if (t == s) {
                if (!t) return true;
                continue;
            }
            else { // mismatch
                if (t == '?') continue;
                if (t != '*') break;

                t = *templ++;
                if (!t) return true; // '*' at EOS

                while (1) {
                    if (!s) break;
                    if (s == t) {
                        if (string_match(str, templ))
                            return true;
                    }
                    s = *str++;
                }
            }
        }
        return false;
    }
}

// check kind of object
// kind_templ may contain wildcards ( ? and * )
bool
Object::is_kind(const char *kind_templ) const {
    return string_match(get_kind(), kind_templ);
}
bool
Object::is_kind(const string& kind_templ) const {
    return string_match(get_kind(), kind_templ.c_str());
}

void
Object::set_attrib(const string& key, const Value& val)
{
    attribs[key] = val;//.insert (key, val);
}

const Value*
Object::get_attrib(const string& key) const
{
    AttribMap::const_iterator i = attribs.find(key);
    if (i == attribs.end())
        return 0;
    else
        return &i->second;
}

bool
Object::string_attrib(const string &name, string *val) const
{
    if (const Value *v = get_attrib(name))
    {
        const char *s = to_string(*v);
        if (s != 0) {
            *val = s;
            return true;
        }
    }
    return false;
}

int
Object::int_attrib(const string &name) const
{
    if (const Value *v = get_attrib(name))
        return to_int(*v);
    return 0;
}

bool
Object::int_attrib(const string &name, int *val) const
{
    if (const Value *v = get_attrib(name)) {
        *val = to_int(*v);
        return true;
    }
    return false;
}


bool
Object::double_attrib(const string &name, double *val) const
{
    if (const Value *v = get_attrib(name))
    {
        *val = to_double(*v);
        return true;
    }
    return false;
}

/* Send an impulse to position 'dest' into direction dir.
 * If 'dest' contains a stone, on_impulse() is called for that stone
 */

void
Object::send_impulse(const GridPos& dest, Direction dir)
{
    if (Stone *st = GetStone(dest)) {
        Impulse impulse(this, dest, dir);
        st->on_impulse(impulse);
    }
}

/* Like variant above, but the _result_ of the impulse is delayed.
 */

void
Object::send_impulse(const GridPos& dest, Direction dir, double delay)
{
    if (Stone *st = GetStone(dest)) {
        addDelayedImpulse(Impulse(this, dest, dir), delay, st);
    }
}

namespace  {
    const char *vstrf(const char *format, va_list argPtr) {
        static size_t  buf_size = 256;
        static char   *buffer   = new char[buf_size];
        // Note: buffer intentionally exists over the whole program-lifetime

        size_t length;
        while (1) {
            if (!buffer) {
                assert(buffer); // to stop when debugging
                throw std::bad_alloc();
            }

            length = vsnprintf(buffer, buf_size, format, argPtr);
            if (length < buf_size) break; // string fits into current buffer

            // otherwise resize buffer :
            buf_size += buf_size/2;
            delete [] buffer;
            buffer    = new char[buf_size];
        }

        return buffer;
    }
};

void
Object::warning(const char *format, ...) const {
    va_list arg_ptr;

    va_start(arg_ptr, format);

    fprintf(stderr, "%p non-grid-\"%s\": ", this, get_kind());
    vfprintf(stderr, format, arg_ptr);
    fputc('\n', stderr);

    va_end(arg_ptr);
}

//----------------------------------------
// GridObject
//----------------------------------------
void
GridObject::play_sound(const char *name)
{
    sound::PlaySound(name, get_pos().center());
}

void
GridObject::warning(const char *format, ...) const {
    va_list        arg_ptr;
    const GridPos& position = get_pos();

    va_start(arg_ptr, format);

    fprintf(stderr, "%p \"%s\" at %i/%i: ", this, get_kind(), position.x, position.y);
    vfprintf(stderr, format, arg_ptr);
    fputc('\n', stderr);

    va_end(arg_ptr);
}




//======================================================================
// FLOORS
//======================================================================

Floor::Floor(const char *kind, double friction_, double mfactor)
: TGridObject<GRID_FLOOR>(kind),
  traits (kind, friction_, mfactor, FLOOR_Normal)
{}

Floor::Floor (const FloorTraits &tr)
: TGridObject<GRID_FLOOR>(tr.name.c_str()), traits (tr)
{}

Floor *Floor::clone() {
    return this;
}

void Floor::dispose() {
    ; // do nothing
}

void
Floor::message(const string& msg, const Value &/*val*/)
{
}

px::V2 Floor::process_mouseforce (Actor *a, px::V2 force)
{
    if (a->int_attrib("controllers") & (1+player::CurrentPlayer()))
        return mousefactor() * force;
    else
        return V2();
}


//----------------------------------------
// Abyss
//----------------------------------------
namespace
{
    class Abyss : public Floor {
        CLONEOBJ(Abyss);
    public:
        Abyss() : Floor("fl-abyss", 2.0, 1) {}
    private:
//         void actor_enter(Actor* a) {SendMessage(a, "fall");}
        void actor_contact (Actor* a) {SendMessage(a, "fall");}
	bool is_destroyable() const {return false;} 
    };
}

//----------------------------------------
// Water
//----------------------------------------
namespace
{
    class Water : public Floor {
        CLONEOBJ(Water);
    public:
        Water() : Floor("fl-water", 5, 1) {}
    private:
//         void actor_enter(Actor *a) {SendMessage(a, "drown");}
        void actor_contact(Actor *a) {SendMessage(a, "drown");}

	bool is_destroyable() const {return false;} 
    };
}

//--------------
// Swamp
//--------------
namespace
{
    class Swamp : public Floor {
        CLONEOBJ(Swamp);
    public:
        Swamp() : Floor("fl-swamp", 13, 1.0) {}
    private:
//         void actor_enter(Actor *a) { SendMessage(a, "sink"); }
        void actor_contact(Actor *a) { SendMessage(a, "sink"); }
        void actor_leave(Actor *a) {
            Floor *new_floor = GetFloor(GridPos(a->get_pos()));

            if (!new_floor->is_kind("fl-swamp")) {
                SendMessage(a, "getout");
            }
        }
	bool is_destroyable() const {return false;} 
    };
}

// ------------------
//      FallenBox
// ------------------

namespace
{
    class FallenBox : public Floor {
        CLONEOBJ(FallenBox);
    public:
        FallenBox(const char *kind)
            :  Floor(modify_kind(kind), 6.4, 2.0) // uses same traits as fl-wood
        {}

    private:
        const char *modify_kind(const char *kind) {
            if (0 == strcmp(kind, "fl-stwood")) {
                return enigma::IntegerRand(0, 1) ? "fl-stwood1" :  "fl-stwood2";
            }
            return kind;
        }
    };
}

// -------------------
//      DummyFloor
// -------------------

namespace
{
    class DummyFloor : public Floor {
        CLONEOBJ(DummyFloor);
    public:
        DummyFloor() : Floor("fl-dummy", 4.0, 2.5) {}
    private:
        void actor_enter(Actor *) {
            static int lastCode = -1;
            int        code     = int_attrib("code");
            if (lastCode != code) {
                fprintf(stderr, "Entering floor 0x%x\n", code);
                lastCode = code;
            }
        }
    };
}



//----------------------------------------
// Gradient
//----------------------------------------

/** \page fl-gradient Gradient Floor

This is a sloped floor that accelerates in a particular direction.

\subsection gradienta Attributes

- \b type: number between 0 and 15 (see below)

\subsection gradiente Examples

*/
namespace
{
    class Gradient : public Floor {
        CLONEOBJ(Gradient);
    public:
        Gradient(int type=MINTYPE);
    private:
        int get_type() const;
        enum { MINTYPE=1, MAXTYPE=24 };

        // GridObject interface.
        virtual void init_model();

        // Floor interface
        virtual px::V2 get_force(Actor *a);
    };
}

Gradient::Gradient(int type)
: Floor("fl-gradient", 4, 2)
{
    set_attrib ("type", Value(type));
//    set_attrib ("force", Value(1.0));
}

int Gradient::get_type() const
{
    int t=int_attrib("type");
    if (t < MINTYPE || t>MAXTYPE) {
        enigma::Log << "Gradient: Illegal type="<< int(t) << endl;
        t=MINTYPE;
    }
    return t;
}

void Gradient::init_model()
{
    char mname[20];
    sprintf(mname, "fl-gradient%d", get_type());
    set_model(mname);
}

px::V2 Gradient::get_force(Actor */*a*/)
{
    V2 force;
    int t = get_type();

    static int xforce[MAXTYPE-MINTYPE+1] = {
         0,  0,  1, -1,
         1, -1,  1, -1,
         1,  1, -1, -1,
        -1, -1,  1,  1,
         1, -1,  1, -1,
         0,  0,  1, -1
    };
    static int yforce[MAXTYPE-MINTYPE+1] = {
         1, -1,  0,  0,
         1,  1, -1, -1,
         1, -1,  1, -1,
         1, -1,  1, -1,
        -1, -1,  1,  1,
         1, -1,  0,  0
    };
    force = V2(xforce[t-MINTYPE], yforce[t-MINTYPE]);
    force.normalize();

    // use enigma::SlopeForce if no "force" attribute is set
    double factor = enigma::SlopeForce;
    double_attrib("force", &factor);
    return factor*force;
}


//----------------------------------------
// Bridge
//----------------------------------------

/** \page fl-bridge Bridge Floor

This Floor can be opened and closed much like a bridge.
The actor can move over this floor if the bridge is closed,
and will fall into abyss when the bridge is open.

\subsection bridgea Attributes

- \b type	The type of the bridge, currently only 'a' is possible

\subsection bridgem Messages
- \b open	open the bridge so actors cannot pass it
- \b close	close the bridge so actors can pass it
- \b openclose	toggle the state of the bridge
- \b signal     same as \b openclose

*/
namespace
{
    class Bridge : public Floor {
        CLONEOBJ(Bridge);
    public:
        Bridge(bool open=true);
    private:
        enum State {
            OPEN, CLOSED, OPENING, CLOSING, // normal states
            CLOSING_BYSTONE, CLOSED_BYSTONE // when stones are moved onto the bridge
        } state;
        // the BYSTONE-states look like closed, but act like open

        char get_type() const {
            string type = "a";
            string_attrib("type", &type);
            return type[0];
        }

//         void actor_enter(Actor *);
        void actor_contact (Actor* a) {if (state!=CLOSED) SendMessage(a, "fall");}
        void message(const string &m, const Value &);
        void init_model();
        void stone_change(Stone *st);

        void change_state( State newstate);
        void animcb();
    };
}

Bridge::Bridge(bool open) : Floor("fl-bridge", 5, 1)
{
    set_attrib("type", "a");
    state=open ? OPEN : CLOSED;
}

void Bridge::stone_change(Stone *st) {
    if (st && !st->is_floating()) {
        if (state == OPEN || state == OPENING) {
            change_state(CLOSING_BYSTONE);
        }
    }
    else {
        if (state == CLOSED_BYSTONE || state == CLOSING_BYSTONE) {
            change_state(OPENING);
        }
    }
}

void Bridge::message(const string &m, const Value &)
{
    if (m == "open" && (state==CLOSED || state==CLOSING))
        change_state(OPENING);
    else if (m=="close")
        switch (state) {
        case OPEN:
        case OPENING:
        case CLOSING_BYSTONE:
            change_state(CLOSING);
            break;
        case CLOSED_BYSTONE:
            change_state(CLOSED);
            break;
        case CLOSED:
        case CLOSING:
            break; // already closed

        }
    else if (m=="openclose" || m=="signal")
        switch (state) {
        case OPEN:
        case OPENING:
        case CLOSING_BYSTONE:
            change_state(CLOSING);
            break;
        case CLOSED_BYSTONE:
            change_state(CLOSED);
            break;
        case CLOSED:
        case CLOSING:
            change_state(OPENING);
            break;
        }
}

// void Bridge::actor_enter(Actor *a)
// {
//     if (state != CLOSED)
//         SendMessage(a, "fall");
// }


void Bridge::init_model()
{
    char mname[25];
    sprintf(mname, "fl-bridge%c-%s", get_type(),
            (state==OPEN) ? "open" : "closed");
    set_model(mname);
}

void Bridge::change_state( State newstate)
{
    if (state != newstate) {
        string mname = string("fl-bridge")+get_type();

        switch( newstate) {
            case OPENING: {
                Stone *st = GetStone(get_pos());
                if (st && !st->is_floating()) {
                    if (state == CLOSED || state == CLOSED_BYSTONE)
                        newstate = CLOSED_BYSTONE;
                    else if (state == CLOSING || state == CLOSING_BYSTONE)
                        newstate = CLOSING_BYSTONE;
                    // here the model is already correct!
                }
                else { // no stone or floating stone :
                    if( state == CLOSING || state == CLOSING_BYSTONE)
                        get_model()->reverse();
                    else
                        set_anim(mname+"-opening");
                }
                break;
            }
            case CLOSING:
            case CLOSING_BYSTONE:
                if( state == OPENING)
                    get_model()->reverse();
                else if (state != CLOSING && state != CLOSING_BYSTONE)
                    set_anim(mname+"-closing");
                break;
            case OPEN:
            case CLOSED:
            case CLOSED_BYSTONE:
                state = newstate;
                init_model();
                break;
        }
        state = newstate;
    }
}

void Bridge::animcb()
{
    switch (state) {
        case OPENING: change_state(OPEN); break;
        case CLOSING: change_state(CLOSED); break;
        case CLOSING_BYSTONE: change_state(CLOSED_BYSTONE); break;
        default : assert(0); break;
    }
}


//----------------------------------------
// Black and white tiles
//----------------------------------------
namespace
{

    class BlackTile : public Floor {
    public:
        BlackTile() : Floor ("fl-acblack", 5.0, 2.0) {}

        px::V2 process_mouseforce (Actor *a, px::V2 force)
        {
            if (player::CurrentPlayer() == 0) // && player::IsCurrentPlayer(a))
                return mousefactor() * force;
            else
                return V2();
        }
    };

    class WhiteTile : public Floor {
    public:
        WhiteTile() : Floor ("fl-acwhite", 5.0, 2.0) {}

        px::V2 process_mouseforce (Actor *a, px::V2 force)
        {
            if (player::CurrentPlayer() == 1) // && player::IsCurrentPlayer(a))
                return mousefactor() * force;
            else
                return V2();
        }
    };
}

//======================================================================
// STONES
//======================================================================

/** \page lstones  Available Stones

Oxyd Stones:
- \ref st-oxyd
- \ref st-fakeoxyd
- \ref st-fart

Movable stones:
- \ref st-brownie
- \ref st-wood
- \ref st-block

Stones that can trigger actions:
- \ref st-switch
- \ref st-fourswitch
- \ref st-laserswitch
- \ref st-key
- \ref st-coinslot
- \ref st-timer

Stones that can change their behaviour:
- \ref st-brick_magic
- \ref st-stonebrush
- \ref st-invisible_magic
- \ref st-break_invisible

Lasers and Mirrors:
- \ref st-laser
- \ref st-pmirror
- \ref st-3mirror

Other stones:
- \ref st-death
- \ref st-swap
- \ref st-bolder
- \ref st-puzzle
- \ref st-stone_break
- \ref st-window
- \ref st-break_acwhite
- \ref st-break_acblack
- \ref st-oneway
- \ref st-oneway_black
- \ref st-oneway_white
- \ref st-chameleon
*/


Stone::Stone(const char * kind)
    : TGridObject<GRID_STONES>(kind)
{}

px::V2
Stone::actor_impulse(const StoneContact &sc)
{
    V2 normal = normalize(sc.normal);
    const V2 &v = sc.actor->get_vel();
    return -2*(v*normal)*normal;
}

StoneResponse
Stone::collision_response(const StoneContact &)
{
    return STONE_REBOUND;
}


void
Stone::actor_hit(const StoneContact &)
{
}

const char *
Stone::collision_sound()
{
    return "st-stone";
}

/* Move a stone (regardless whether it is_movable() or not) if
   the destination field is free.
   Returns: true if stone has been moved.

   Note: This should be used by on_impulse() to perform a move.
*/
bool
Stone::move_stone(Direction dir) {
    GridPos p      = get_pos();
    GridPos newPos = move(p, dir);

    if (!GetStone(newPos)) {
        play_sound("st-move");

        MoveStone(p, newPos);
        player::IncMoveCounter();

        on_move();
        if (Item *it = GetItem(newPos)) it->on_stonehit(this);

        return true;
    }
    return false;
}

/* Determine whether the actor hitting the stone can move stone
   and return either the direction the stone should move or NODIR. */
Direction
Stone::get_push_direction (const StoneContact &sc)
{
    ActorInfo *ai  = sc.actor->get_actorinfo();
    Direction  dir = contact_face(sc);

    // If the speed component towards the face of the stone is large
    // enough (and actually pointing towards the stone), moving the
    // stone is possible.
    if (dir!=enigma::NODIR && ai->vel * sc.normal < -4)
        return reverse(dir);
    return NODIR;
}

/* Move a stone (by sending an impulse)
   Called when actor hits a stone
 */
bool
Stone::maybe_push_stone (const StoneContact &sc)
{
    Direction dir = get_push_direction(sc);
    if (dir != enigma::NODIR) {
        sc.actor->send_impulse(sc.stonepos, dir);
        return GetStone(sc.stonepos) == 0; // return true only if stone vanished
    }
    return false;
}


//----------------------------------------------------------------------
// OBJECT REPOSITORY
//----------------------------------------------------------------------
namespace
{
    class ObjectRepos : public px::Nocopy {
    public:
        ObjectRepos();
        ~ObjectRepos();
        void add_templ(Object *o);
        void add_templ (const string &name, Object *o);
        bool has_templ(const string &name);
        Object *make(const string &name);
        Object *get_template(const string &name);
    private:
        typedef px::Dict<Object*> ObjectMap;
        ObjectMap objmap;           // repository of object templates
        int stonecount, floorcount, itemcount;
    };
}

ObjectRepos::ObjectRepos()
{
    stonecount=floorcount=itemcount=0;

    // Floors (most floors are defined in init.lua)
    add_templ(new Abyss);
    add_templ(new Water);
    add_templ(new Swamp);
    add_templ(new DummyFloor);
    add_templ(new FallenBox("fl-stwood"));
    add_templ(new FallenBox("fl-stwood1"));
    add_templ(new FallenBox("fl-stwood2"));
    add_templ(new Bridge);
    add_templ("fl-bridge-open", new Bridge(true));
    add_templ("fl-bridge-closed", new Bridge(false));
    add_templ(new WhiteTile);
    add_templ(new BlackTile);

    add_templ(new Gradient);
    add_templ("fl-gradient1", new Gradient(1));
    add_templ("fl-gradient2", new Gradient(2));
    add_templ("fl-gradient3", new Gradient(3));
    add_templ("fl-gradient4", new Gradient(4));
    add_templ("fl-gradient5", new Gradient(5));
    add_templ("fl-gradient6", new Gradient(6));
    add_templ("fl-gradient7", new Gradient(7));
    add_templ("fl-gradient8", new Gradient(8));
    add_templ("fl-gradient9", new Gradient(9));
    add_templ("fl-gradient10", new Gradient(10));
    add_templ("fl-gradient11", new Gradient(11));
    add_templ("fl-gradient12", new Gradient(12));
    add_templ("fl-gradient13", new Gradient(22));
    add_templ("fl-gradient14", new Gradient(21));
    add_templ("fl-gradient15", new Gradient(24));
    add_templ("fl-gradient16", new Gradient(23));

}

ObjectRepos::~ObjectRepos()
{
    px::delete_map(objmap.begin(), objmap.end());
}


void
ObjectRepos::add_templ (const string &kind, Object *o)
{
    if (has_templ(kind))
        enigma::Log << "add_templ: redefinition of object `" <<kind<< "'.\n";
    else
        objmap.insert(kind, o);
}

// Add an new Object template to `objmap'.
void
ObjectRepos::add_templ(Object *o)
{
    string kind = o->get_kind();
    if (has_templ(kind))
        enigma::Log << "add_templ: redefinition of object `" <<kind<< "'.\n";
    else
        objmap.insert(kind, o);
}

bool
ObjectRepos::has_templ(const string &name)
{
    return objmap.find(name) != objmap.end();
}

Object *
ObjectRepos::make(const string &name)
{
    ObjectMap::iterator i=objmap.find(name);
    if (i==objmap.end())
        return 0;
    else
        return i->second->clone();
}

Object *
ObjectRepos::get_template(const string &name)
{
    if (objmap.has_key(name))
        return objmap[name];
    else
        return 0;
}

//----------------------------------------------------------------------
// FUNCTIONS
//----------------------------------------------------------------------
namespace
{
    ObjectRepos *repos;
}

void world::Register (const string &kind, Object *obj)
{
    if (!repos)
        repos = new ObjectRepos;
    if (kind.empty())
        repos->add_templ(obj->get_kind(), obj);
    else
        repos->add_templ(kind, obj);
}


void world::Register (Object *obj)
{
    Register (obj->get_kind(), obj);
}

void world::Register (const string &kind, Floor *obj)
{
    Object *o = obj;
    Register(kind, o);
}

void world::Register (const string &kind, Item *obj)
{
    Object *o = obj;
    Register(kind, o);
}

void world::Register (const string &kind, Stone *obj)
{
    Object *o = obj;
    Register(kind, o);
}

void world::Register (const string &kind, Actor *obj)
{
    Object *o = obj;
    Register(kind, o);
}

void
world::Repos_Shutdown()
{
    delete repos;
}


Object *
world::MakeObject(const char *kind)
{
    static Object *last_templ = 0;
    static string last_kind;

    if (last_kind!=kind) {
        last_kind = kind;
        last_templ = repos->get_template(kind);
    }

    Object *o = 0;
    if (last_templ)
        o=last_templ->clone();
    if (!o)
        fprintf(stderr, "MakeObject: unkown object name `%s'\n",kind);
    return o;
}

Object *
world::GetObjectTemplate(const std::string &kind)
{
    if (!repos->has_templ(kind)) {
        cerr << "GetObjectTemplate: unkown object name `" <<kind<< "'.\n";
        return 0;
    } else
        return repos->get_template(kind);
}


Floor*
world::MakeFloor(const char *kind)
{
    return dynamic_cast<Floor*>(MakeObject(kind));
}

Item *
world::MakeItem(const char *kind)
{
    return dynamic_cast<Item*>(MakeObject(kind));
}

Stone *
world::MakeStone (const char *kind)
{
    return dynamic_cast<Stone*>(MakeObject(kind));
}

Actor  *
world::MakeActor (const char *kind)
{
    return dynamic_cast<Actor*>(MakeObject(kind));
}

void
world::DisposeObject(Object *o)
{
    if (o != 0) {
        UnnameObject(o);
        o->dispose();
    }
}

void
world::SendMessage(Object *o, const std::string &msg)
{
    if (o) {
#if defined(VERBOSE_MESSAGES)
        o->warning("will be sent message '%s'", msg.c_str());
#endif // VERBOSE_MESSAGES
        o->message(msg, Value());
    }
#if defined(VERBOSE_MESSAGES)
    else
        fprintf(stderr, "Sending message '%s' to NULL-object\n", msg.c_str());
#endif // VERBOSE_MESSAGES
}

void
world::SendMessage(Object *o, const std::string &msg, const Value& value)
{
    if (o) {
#if defined(VERBOSE_MESSAGES)
        o->warning("will be sent message '%s' (with Value)", msg.c_str());
#endif // VERBOSE_MESSAGES
        o->message(msg, value);
    }
#if defined(VERBOSE_MESSAGES)
    else
        fprintf(stderr, "Sending message '%s' to NULL-object\n", msg.c_str());
#endif // VERBOSE_MESSAGES
}

// explosion messages:
// -------------------
//
// "ignite"    : (sent by dynamite)
// "expl"      : (sent by bomb)
// "bombstone" : (sent by bombstone)

void
world::SendExplosionEffect(GridPos center, ExplosionType type) {
    const int AFFECTED_FIELDS       = 8;
    int       xoff[AFFECTED_FIELDS] = { 0, 0, 1,-1, 1,-1, 1,-1 };
    int       yoff[AFFECTED_FIELDS] = { 1,-1, 0, 0, 1, 1,-1,-1 };

    for (int a = 0; a<AFFECTED_FIELDS; ++a) {
	GridPos  dest(center.x+xoff[a], center.y+yoff[a]);
	Item    *item            = GetItem(dest);
	Stone   *stone           = GetStone(dest);
	bool     direct_neighbor = a<4;

	switch (type) {
	case DYNAMITE:
	    if (stone) SendMessage(stone, "ignite");
	    if (item) SendMessage(item, "ignite");
	    break;

	case BLACKBOMB:
	case WHITEBOMB:
	    if (direct_neighbor) {
		if (stone) SendMessage(stone, "expl");
		if (item) {
		    SendMessage(item, "expl");
		}
		else {
		    SetItem(dest, MakeItem(type == BLACKBOMB
					   ? "it-explosion1"
					   : "it-explosion3"));
		}
	    }
	    break;

	case BOMBSTONE:
	    if (direct_neighbor) {
		if (stone) SendMessage(stone, "bombstone");
		if (item) SendMessage(item, "bombstone");
	    }
	    break;
	}
    }
}


void
world::DefineSimpleFloor(const std::string &kind,
                         double friction, double mousefactor)
{
    Register(new Floor(kind.c_str(), friction, mousefactor));
}
