;#
;# Copyright (c) 1995-1998, Ikuo Nakagawa.
;# All rights reserved.
;#
;# $Id: ftputil.pl,v 1.14 1998/02/08 12:02:43 ikuo Exp $
;#
;# Description:
;#	ftputil.pl is a collection of FTP utilities.
;#	- Socket based routines.
;#	- Both of PASV and PORT are supported.
;#	- Round Robin (in name service) is supported.
;# Last update:
;#	1998/02/08 by Ikuo Nakagawa
;# History:
;#	1998/02/08 `print xx || die' must be `(print xx) || die'.
;#	1997/06/01 Fix a bug in &retr; move `open' before pasv/port.
;#	1996/10/24 Multiple servers in $server are supported.
;#	1996/10/24 DNS Round Robin was supported.
;#	1996/10/24 Siteidle routine was added.
;#	1996/09/22 Sitegroup routine was added.
;#	1996/09/22 Private routines were renamed as xx_*.
;#	1996/09/20 Add pack_sin/unpack_sin routines.
;#	1996/09/11 $bindaddress was added for multi-homed host.
;#	1996/09/10 umask/mkdir/rmdir/rename were added.
;#	1996/03/21 Now, PASV return code 125 is treated as normal.
;#	1996/03/12 Fix around pack_sockaddr_in for perl5.002.
;#	1995/12/04 Support pack_sockaddr_in in perl5.001n.
;#	1995/07/31 Use Socket rather than `require socket.ph'.
;#	1995/07/10 Rename internal routines as foo => _foo.
;#	1995/06/14 Add some comments.
;#	1995/05/07 Add timeout routines.
;#	1995/04/09 First edition.
;#
package ftp;
;#
;# Subroutines:
;#
;#  ftp::login()
;#	Log in to the host $server with USER as $user and PASS
;#	as $pass. You can also set $gateway as proxy gateway
;#	(ex. TIS ftp-gw). Login returns 1 on success, and returns
;#	undef otherwide.
;#
;#  ftp::quit()
;#	Terminate FTP session. If session is terminated nomally,
;#	quit returns 1. Otherwise undef is returned.
;#
;#  ftp::type(type-name)
;#	Set transfer type to <type-name>. <type-name> is one of
;#	'A' or 'I'. Type returns 1 on success, and return undef
;#	on any error.
;#
;#  ftp::cd(path-name)
;#	Change directory to <path-name>. 1 is returned on success,
;#	otherwise undef is returned.
;#
;#  ftp::pwd
;#	Get current working directory name and return it. If any
;#	error happens, undef is returned.
;#
;#  ftp::list(path-name)
;#	Get the list of <path-name>. If <path-name> is omitted,
;#	the list of current directory is obtained. List returns
;#	undef on any error.
;#
;#  ftp::stat(path-name)
;#	Get status of <path-name>. Any results are found in $buffer.
;#	Undef is returned on any error.
;#
;#  ftp::size(path-name)
;#	Get the size of <path-name>, or undef on failure.
;#
;#  ftp::mtime(path-name)
;#	Get the modified time of <path-name>. Result is a string
;#	'YYYYMMDDHHMMSS'. If error is happen, undef is returned.
;#
;#  ftp::retr(path-name, local-name)
;#	Retrieve <path-name> as <local-name> in local file system.
;#	If <local-name> is omitted, retr use <path-name> for new
;#	file name.
;#
;#  ftp::stor(local-name, remote-name)
;#	Send <local-name> in local file system to <remote-name> in
;#	remote file system. If <remote-name> is omitted, stor use
;#	<local-name> as remote file name.
;#
;#  ftp::umask(mask)
;#	Set umask to <mask> for uploads. <mask> is octet streams.
;#
;#  ftp::sitegroup(group, gpass)
;#	Set group name to <group>. <group> is group name. <gpass>
;#	is password for access group.
;#
;#  ftp::siteidle(idle)
;#	Get or set idle timer for ftp transmission. Idle is optional.
;#	If idle is undef, siteidle returns current idle timer, and
;#	otherwise, siteidle sets idle timer.
;#
;#  ftp::unlink(path-name)
;#	Delete <path-name> in remote file system.
;#
;#  ftp::mkdir(path-name)
;#	Make directory <path-name> in remote file system.
;#
;#  ftp::rmdir(path-name)
;#	Remove directory <path-name> in remote file system.
;#
;#  ftp::rename(old-name, new-name)
;#	Rename <old-name> to <new-name> in remote file system.
;#
;# Internal routines:
;#
;#  xx_init(host-name, port);
;#	Initialize tcp socket connecting to <host-name> with port
;#	<port>. Init returns 1 if socket is normally connected,
;#	or otherwise undef is returned.
;#
;#  xx_send(string)
;#	Send a string through HANDLE. String is stripped trailing
;#	spaces and appended CR-LF. Xfer returns 1 on success, or
;#	undef on failure.
;#
;#  xx_recv()
;#	Recieve a message from remote server. Recv read a message
;#	until `code' included line (i.e. beginning with 3 numbers
;#	ans one space charactor) was found. Recv store the message
;#	to $buffer and returns the `code'. Undef is returned if
;#	any error happens.
;#
;#  xx_port()
;#	Send PORT command to remote server. This must be used with
;#	`xx_accept' subroutine. Undef is retruned if it fails.
;#
;#  xx_accept()
;#	Accept connection from remote server. This routine must be
;#	called after `xx_port' subroutine. Undef is returned if any
;#	error happenes.
;#
;#  xx_pasv()
;#	Send PASV command to remote server, and make connection to
;#	the server. A new connection between local-host and remote
;#	server will be used for data transfer.
;#
;#  xx_tick(timeout)
;#	Set timeout callback. Timeout value is defined in `timeout'.
;#	If the first aegument is 0, xx_tick let timeout be off.
;#
;#  xx_timeup()
;#	Registered as signal handler by `xx_tick' subroutine and called
;#	if timeout is occered in data transmit. xx_timeup terminate any
;#	session and exit.
;#
;#  xx_debug(strings)
;#	Print any string to STDOUT as debug output, if $debug is true.
;#	Do nothing if $debug is not true.
;#
;# Variables:
;#
;#  SOCK
;#	File handle for socket which is connected to remote ftp
;#	server. SOCK is aviable between login and quit.
;#
;#  DATA
;#	File handle for real data transfer
;#
;#  $server
;#	Server hostname you want to login. No default value is defined.
;#
;#  $gateway
;#	Set this value if you want to use proxy gateway(ex. TIS ftp-gw).
;#	If $gateway is defined, `login' will try to connect $gateway and
;#	send ``USER user@server'' command to login. By default, this
;#	not defined.
;#
;#  $user
;#	User name used when you login to the server. Default value is
;#	`anonymous'.
;#
;#  $pass
;#	Password used when you login to the server. By default, it is
;#	your mail address, like ``ikuo@isl.intec.co.jp''.
;#
;#  $buffer
;#	Messages retrieved by last ftp commend. You should not
;#	use $buffer in upper session.
;#
;#  $debug
;#	Debug level for print debugging information. Currently,
;#	0, 1, 2 is meeningful.
;#
;#  $timeout
;#	Interval time to disconnect ftp session. If unit transfer
;#	costs too much time, ftp::xx_timeup will called.
;#
;# we use Socket extensions
;#
use Socket;
use strict 'subs';
;#
;# and we use logging routines
;#
require "log.pl";
;#
;# prototypes for internal subroutines
;#
;# sub xx_recv(;$);
;# sub xx_send(@);
;# sub xx_init($$);
;# sub xx_login($$$);
;# sub xx_port();
;# sub xx_pasv();
;# sub xx_accept();
;# sub xx_debug(@);
;# sub xx_tick($);
;# sub xx_timeup($);
;#
;# prototypes for external subroutines
;#
;# sub pack_sin($$$$$);
;# sub unpack_sin($);
;# sub login();
;# sub quit();
;# sub cd($);
;#
;# default parameters
;#
chomp($hostname = `hostname`);
$server = '';
$gateway = '';
$user = 'anonymous';
$pass = getpwuid($<).'@'.$hostname;
$debug = 0;
$timeout = 30;
$passive = 0; # use Passive mode to make a data connection
$tcp = (getprotobyname('tcp'))[2];
$bindaddress = '';
$lastmesg = '';
$lastcode = 0;
$sockaddr_in = 'S n C4 x8';
undef $dumpfile;		# $dumpfile = "/tmp/ftp.dump";
$idle = 0;
$max_idle = undef;

;# for pack_sockaddr_in, unpack_sockaddr_in
$version = $1 * 1000 if $] =~ /^(\d+\.\d+)/;

;#
;# $sockaddr_in = pack_sin($port, $a, $b, $c, $d)
;# where
;#	$port is service name of TCP, or port #
;#	$host is a hostname or nn.nn.nn.nn
;#
sub pack_sin {
	my($port, @quad) = @_;

	if (@quad != 4) {
		log::putl("WARNING", "address contains 4 octets.");
		return undef;
	}
	if ($port !~ /^\d+$/) {
		defined($port = (getservbyname($port, 'tcp'))[2])
			|| return undef;
	}
	if (defined(&pack_sockaddr_in) && $version >= 5002) {
		return pack_sockaddr_in($port, pack('C4', @quad));
	}
	if (defined(&pack_sockaddr_in)) {
		return pack_sockaddr_in(AF_INET, $port, pack('C4', @quad));
	}
	pack($sockaddr_in, AF_INET, $port, @quad);
}

;#
;# ($port, $a, $b, $c, $d) = unpack_sin($sockaddr_in);
;#
sub unpack_sin {
	my($sock) = @_;
	my($port, $addr, $f, $a, $b, $c, $d);

	if (defined(&pack_sockaddr_in) && $version >= 5002) {
		($port, $addr) = unpack_sockaddr_in($sock);
		($a, $b, $c, $d) = unpack('C4', $addr);
	} elsif (defined(&pack_sockaddr_in)) {
		($f, $port, $addr) = unpack_sockaddr_in($sock);
		($a, $b, $c, $d) = unpack('C4', $addr);
	} else {
		($f, $port, $a, $b, $c, $d) = unpack($sockaddr_in, $sock);
	}
	($port, $a, $b, $c, $d);
}

;#
sub xx_debug {
	my(@x) = @_;
	local($_);

	grep((s/\s+$//, log::putl("DEBUG", "+ $_")), @x) if $debug;
}

;#
sub xx_timeup {
	my($sig) = @_;

	die("!CATCH SIG$sig, stopped");
}

;#
sub xx_tick {
	my($val) = @_;

	$SIG{'ALRM'} = $val > 0 ? \&xx_timeup : 'DEFAULT';
	alarm($val);
	1;
}

;# Initalize tcp session. Create a socket and connect to remote host.
;# We try to connect all servers if multiple addresses were found by
;# gethostbyname, but...
;# Hmmm, is this perl BUG? When we try to connect more than one time,
;# connect returns "Invalid argument".
;# This becase we create a socket for each time.
sub xx_init {
	my($remote, $port) = @_;
	local($[, @addr, $this, $that);

	;# set timeout
	xx_tick($timeout);

	;# if we have $bindaddress...
	if ($bindaddress ne '') {
		$this = pack_sin(0, unpack('C4', inet_aton($bindaddress)));
	}

	;# make sockaddr_in structure
	if ($remote =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/) {
		@addr = (pack('C4', $1, $2, $3, $4));
	} else {
		@addr = gethostbyname($remote);
		splice(@addr, 0, 4);
		if (!defined($addr[$[])) {
			log::putl("WARNING", "gethostbyname: $!");
			return undef;
		}
	}

	;# clear timeout once.
	xx_tick(0);

	;# try to connect server, in order...
	for ($i = $[; $i <= $#addr; $i++) {
		$that = pack_sin($port, unpack('C4', $addr[$i]));

		;#
		my $b = inet_ntoa($addr[$i]);
		log::putl("DEBUG", "try to connect $b");

		;# set timeout (in future release)
		# xx_tick($timeout);

		;# creating socket for control port
		if (!socket(SOCK, PF_INET, SOCK_STREAM, $tcp)) {
			log::putl("WARNING", "socket: $!");
			return undef;
		}

		;# bint to this socket only if $bindaddress is defined.
		if (defined($this) && !bind(SOCK, $this)) {
			log::putl("WARNING", "bind: $!");
			close(SOCK);
			return undef;
		}

		;# connect to the server
		last if connect(SOCK, $that);

		;# clear timeout (in future release)
		# xx_tick(0);

		;# log
		log::putl("NOTICE", "connect to $b: $!");
		close(SOCK);
	}

	;# check success...
	if ($i > $#addr) {
		log::putl("WARNING", "all connect failed.");
		return undef;
	} else {
		my ($a, $b, $c, $d) = unpack('C4', $addr[$i]);
		log::putl("DEBUG", "connect to $a.$b.$c.$d ok.");
	}

	;# set this socket non-blocking
	my $a = select(SOCK); $| = 1; select($a);

	;# success
	1;
}

;#
;# Usage: xx_recv();
;# Recieve a response from control socket. A response is end
;# with a line which matches to /^d\d\d /.
;# After xx_recv was called,
;#	$buffer is full response words.
;#	$lastcode is the result code of response.
;#	$lastmesg is the result message of response.
;#
sub xx_recv {
	my($ret) = @_;
	local($_);

	;# clear input buffer
	$buffer = '';

	;# setup timeout
	xx_tick($timeout);

	;# main loop of reading from socket
	while (<SOCK>) {
		print DUMP $_ if defined($dumpfile);
		xx_debug($_);

		;# clear timer
		xx_tick(0);

		;# append this response to the input buffer.
		$buffer .= $_;

		;# check end-of-response
		if (/^(\d\d\d) /) {
			$lastcode = $1;
			($lastmesg = $') =~ s/\s+$//;
			return $lastcode;
		}

		;# reset timeout
		xx_tick($timeout);
	}

	;# we found end-of-file. terminate all.
	$ret ? undef : die("xx_recv, stopped");
}

;#
;# Usage: xx_send(string, ...);
;# Send strings to the control socket. For each string, xx_send
;# appends CRLF at the end of words.
;# When send was failed, we should return undef rather than die.
;#
sub xx_send {
	for $x (@_) {
		local($_) = $x;
		print DUMP $_ if defined($dumpfile);
		s/\s*$/\r\n/;	# append CRLF
		xx_debug($_);
		xx_tick($timeout);
		(print SOCK $_) || die("xx_send, stopped");
		xx_tick(0);
	}
	1;
}

;#
;# send QUIT to the control socket, and close the control socket.
;#
sub quit {
	my $code = 0;	# default is error

	if (xx_send("QUIT")) {	# only if we can send QUIT
		$code = (xx_recv(1) == 221);
	}
	shutdown(SOCK, 2);
	close(SOCK);
	close(DUMP) if defined($dumpfile);
	$code;
}

;#
;# xx_login try to log on a single server.
;#
sub xx_login {
	my($xhost, $xuser, $xpass) = @_;

	if (defined($dumpfile)) {
		undef $dumpfile if !open(DUMP, ">$dumpfile");
	}

	if (!xx_init($xhost, 'ftp')) {
		log::putl("WARNING", "xx_init failed");
		return undef;
	}

	if (!(xx_recv(1) == 220
	   && xx_send("USER $xuser")
	   && xx_recv(1) == 331
	   && xx_send("PASS $xpass")
	   && xx_recv(1) == 230)) {
		log::putl("WARNING", "fail to login $xhost/$xuser.");
		quit;
		return undef;
	}

        ;# send group/gpass
	if ($group ne '' && $gpass ne '') {
		if (!sitegroup($group, $gpass)) {
			log::putl("WARNING", "site group/gpass failed.");
			quit;
			return undef;
		}
	}

	;# get/set idle timer
	if ($idle > 0 || $max_idle) {
		my($i, $max) = siteidle();
		log::putl("DEBUG", "IDLE ($i, $max)");
		if (defined($max)) {
			$idle = $max if $max_idle;
			siteidle($idle); # ignore result
		}
	}

	;# success to login
	log::putl("NOTICE", "success to login $xhost/$xuser.");
	1;
}

;#
;# login - call xx_login for each FTP server, until connection
;# would be established. currently, multiple FTP servers can be
;# defined in $server. Please set $server as:
;#
;#  $server = 'ring.etl.go.jp ring.crl.go.jp ring.asahi-net.or.jp';
;#
sub login {
	if ($server eq '') {
		log::putl("WARNING", "\$ftp::server is not defined");
		return undef;
	}
	if ($user eq '') {
		log::putl("WARNING", "\$ftp::user is not defined");
		return undef;
	}

	;# we support multiple server.
	my @servers = split(/\s+/, $server);

	;# is $gateway defined?
	if ($gateway eq '') {
		for $s (@servers) {
			log::putl("DEBUG", "login: try $s.");
			return 1 if xx_login($s, $user, $pass);
		}
	} else {
		for $s (@servers) {
			log::putl("DEBUG", "login: try $s.");
			return 1 if xx_login($gateway, $user.'@'.$s, $pass);
		}
	}

	;# all tries failed.
	undef;
}

;#
;# Send PORT command to the control socket.
;#
sub xx_port {

	;# get address of control socket
	my($port, $a, $b, $c, $d) = unpack_sin(getsockname(SOCK));
	my $this = pack_sin(0, $a, $b, $c, $d);

	;# PORT command need new socket
	if (!socket(S, PF_INET, SOCK_STREAM, $tcp)) {
		log::putl("WARNING", "socket: $!");
		return undef;
	}

	;# bind the socket an address which same as control socket
	if (!bind(S, $this)) {
		log::putl("WARNING", "bind: $!");
		close(S);
		return undef;
	}

	;# get address/port of accepting socket
	($port, $a, $b, $c, $d) = unpack_sin(getsockname(S));
	my($hi, $lo) = (($port >> 8) & 0x00ff, $port & 0x00ff);

	;# listen on this socket
	if (!listen(S, 5)) {
		log::putl("WARNING", "listen: $!");
		close(S);
		return undef;
	}

	;# send PORT command
	if (!xx_send("PORT $a,$b,$c,$d,$hi,$lo")) {
		log::putl("WARNING", "xx_send failed");
		close(S);
		return undef;
	}

	;# wait for a response
	if (xx_recv() !~ /^2/) {
		log::putl("NOTICE", "PORT failed: $lastmesg");
		close(S);
		return undef;
	}

	;# success to PORT
	1;
}

;#
;#
;#
sub xx_accept {

	;# set timeout
	xx_tick($timeout);

	;# accept new connection from the server
	if (!accept(DATA, S)) {
		xx_tick(0);	# clear timeout
		close(S);
		log::putl("WARNING", "accept: $!");
		return undef;
	}

	;# clear timeout
	xx_tick(0);

	;# close listening socket
	close(S);

	;# set data socket non-blocking
	my $a = select(DATA); $| = 1; select($a);

	;#
	1;
}

;#
sub xx_pasv {

	if (!xx_send("PASV")) {
		log::putl("WARNING", "xx_send failed");
		return undef;
	}
	if (xx_recv() !~ /^2/) {
		log::putl("WARNING", "PASV failed: $lastmesg");
		return undef;
	}
	if ($buffer !~ /\((\d+),(\d+),(\d+),(\d+),(\d+),(\d+)\)/) {
		log::putl("WARNING", "PASV can't parse address");
		return undef;
	}
	my @quad = ($1, $2, $3, $4);
	my $port = $5 * 256 + $6;
	my $that = pack_sin($port, @quad);
	my($p, $a, $b, $c, $d) = unpack_sin(getsockname(SOCK));
	my $this = pack_sin(0, $a, $b, $c, $d);

	;# setup timeout
	xx_tick($timeout);

	if (!socket(DATA, PF_INET, SOCK_STREAM, $tcp)) {
		xx_tick(0);
		log::putl("WARNING", "socket: $!");
		return undef;
	}
	if (!bind(DATA, $this)) {
		xx_tick(0);
		close(DATA);
		log::putl("WARNING", "bind: $!");
		return undef;
	}
	if (!connect(DATA, $that)) {
		xx_tick(0);
		close(DATA);
		log::putl("WARNING", "connect: $!");
		return undef;
	}

	;# clear timeout
	xx_tick(0);

	;# set this socket non-blocking
	my $x = select(DATA); $| = 1; select($x);

	;# success
	1;
}

;#
sub list {
	my($path) = @_;
	my(@list, $addr, $port, $buff);
	local($_);

	# we use PORT or PASV?
	($passive ? xx_pasv : xx_port) || return undef;

	# send LIST command
	xx_send("LIST $path") &&
	xx_recv() == 150 || return undef;

	#
	$passive || xx_accept || return undef;

	# get data of LIST from another socket
	xx_tick($timeout * 5);
	while (<DATA>) {
		xx_tick($timeout * 5);
		s/\s+$//;
		push(@list, $_);
	}
	xx_tick(0);
	close(DATA);

	# check code
	xx_recv() =~ /^2/ || return undef;

	# LIST is success but empty directry.
	@list = ('total 0') if @list == 0;

	# return list as result.
	@list;
}

;#
;# ftp::retr(remote-path, local-path)
;# where `remote-path' is remote-side path name to retrieve,
;# and `local-path' is local-side path name to write.
;#
sub retr {
	my($path, $file) = @_;
	my($temp, $length);
	local($_, *FILE);

	$file = $path if !defined($file);
	($temp = $file) =~ s%[^/]+$%.in.$&%;
	$_ = ''; # to aboid warning when we use -w option

	;# open should be before PASV/PORT command.
	if (!open(FILE, ">$temp")) {
		log::putl("WARNING", "open: $!"), return undef;
	}

	;# send PASV/PORT command.
	($passive ? xx_pasv : xx_port) || do { unlink($temp), return undef };

	xx_send("RETR $path") &&
	xx_recv() =~ /^1/ || do { unlink($temp), return undef };

	$passive || xx_accept || do { unlink($temp), return undef };

	xx_tick($timeout * 10);
	while (($length = read(DATA, $_, 2048)) > 0) {
		(print FILE $_) || die("xx_retr, stopped");
		xx_tick($timeout * 10);
	}
	xx_tick(0);
	close(FILE);
	close(DATA);

	# check termination value
	if (xx_recv() =~ /^2/) {
		# unlink($file); # ignore result
		return rename($temp, $file);
	}
	undef;
}

;#
;# ftp::stor(local-path, remote-path)
;# where `local-path' is the local-side file name to be read,
;# and `remote-path' is the remote-side file name to store.
;#
sub stor {
	my($lx, $rx) = @_;
	my($length);
	local($_, *FILE);

	if (!open(FILE, $lx)) {
		log::putl("WARNING", "open: $!"), return undef;
	}

	$rx = $lx if !defined($rx);

	($passive ? xx_pasv : xx_port) || return undef;

	xx_send("STOR $rx") &&
	xx_recv() == 150 || return undef;

	$passive || xx_accept || return undef;

	xx_tick($timeout * 10);
	while (($length = read(FILE, $_, 2048)) > 0) {
		(print DATA $_) || die("xx_store, stopped");
		xx_tick($timeout * 10);
	}
	xx_tick(0);
	close(FILE);
	close(DATA);

	# check termination value
	xx_recv() =~ /^2/ ? 1 : undef;
}

;#
sub stat {
	my($path) = @_;

	xx_send("STAT $path")		&&
	xx_recv() =~ /^2/		? $buffer : undef;
}

;#
sub cd {
	my($dir) = @_;

	xx_send("CWD $dir")		&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub pwd {
	xx_send("PWD")			&&
	xx_recv() =~ /^2/		&&
	$lastmesg =~ /^"(.*)\" /	? $1 : undef;
}

;#
sub type {
	my($type) = @_;

	xx_send("TYPE $type")		&&
	xx_recv() =~ /^2/		? 1: undef;
}

;#
sub size {
	my($path) = @_;

	xx_send("SIZE $path")		&&
	xx_recv() =~ /^2/		&&
	$lastmesg =~ /^(\d+)$/		? $1 : undef;
}

;#
sub mtime {
	my($path) = @_;

	xx_send("MDTM $path")		&&
	xx_recv() =~ /^2/ 		&&
	$lastmesg =~ /^(\d+)$/		? $1 : undef;
}

;#
sub umask {
	my($mask) = @_;

	xx_send("SITE UMASK $mask")	&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub sitegroup {
	my($group, $gpass) = @_;

	xx_send("SITE group $group")	&&
	xx_recv() =~ /^2/		&&
	xx_send("SITE gpass $gpass")	&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub siteidle {
	my($idle) = @_;

	if (defined($idle) && $idle > 0) {
		xx_send("SITE IDLE $idle")	&&
		xx_recv() =~ /^2/		&&
		$lastmesg =~ /\d+/		? $& : undef;
	} else {
		xx_send("SITE IDLE")		&&
		xx_recv() =~ /^2/		&&
		$lastmesg =~ /(\d+)\D+(\d+)/	?
		wantarray ? ($1, $2) : $1 : undef;
	}
}

;#
sub unlink {
	my($path) = @_;

	xx_send("DELE $path")		&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub mkdir {
	my($path) = @_;

	xx_send("MKD $path")		&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub rmdir {
	my($path) = @_;

	xx_send("RMD $path")		&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;#
sub rename {
	my($old, $new) = @_;

	xx_send("RNFR $old")		&&
	xx_recv() =~ /^3/		&&
	xx_send("RNTO $new")		&&
	xx_recv() =~ /^2/		? 1 : undef;
}

;# success for this package
1;
