/* Copyright (C) 2004 Damir Zucic */

/*=============================================================================

			    hyphob_function3.c

Purpose:
	Draw the hydrophobicity function F3.  The sequence stored to the
	main sequence buffer  is used  to calculate  the function value.

Input:
	(1) Pointer to the storage where the minimal function value will
	    be stored.
	(2) Pointer to the storage where the maximal function value will
	    be stored.
	(3) Pointer to RuntimeS structure.

Output:
	(1) Function F3 calculated and stored.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Negative on failure.

Notes:
	(1) The function  F3 may be modified and used for many purposes.
	    Originally, it was introduced while searching for the method
	    which will be suitable for prediction of the porin secondary
	    structure.

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"


/*======function prototypes:=================================================*/

int		IsAromatic_ (char *);

/*======calculate the hydrophobicity function F3:============================*/

int HyphobFunction3_ (double *min_functionP, double *max_functionP,
		      RuntimeS *runtimeSP)
{
int             residuesN, residueI;   /* Do not use size_t instead of int ! */
int		max_length;
int		windowI, combinedI;
int		used_residuesN;
char		*residue_nameP;
char		residue_nameA[RESNAMESIZE];
int		aromaticF;
int		aromatic29, aromatic28, aromatic27, aromatic26,
		aromatic25, aromatic24, aromatic23, aromatic22,
		aromatic21, aromatic20, aromatic19, aromatic18;
int		function_int_value;
double		function_double_value;

static int	flag29A[29] =  {1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1};

static int	flag28A[29] =  {0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1};

static int	flag27A[29] =  {0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0};

static int	flag26A[29] =  {0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0};

static int	flag25A[29] =  {0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0};

static int	flag24A[29] =  {0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0};

static int	flag23A[29] =  {0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0};

static int	flag22A[29] =  {0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0};

static int	flag21A[29] =  {0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0};

static int	flag20A[29] =  {0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0};

static int	flag19A[29] =  {0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0};

static int	flag18A[29] =  {0, 0, 0, 0, 0, 0,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				1, 0, 1, 0, 1, 0, 1, 0, 1,
				0, 0, 0, 0, 0};

/*------prepare some parameters:---------------------------------------------*/

/* The number of residues in sequence buffer: */
residuesN = (int) runtimeSP->residuesN;
if (residuesN == 0) return -1;

/* The maximal residue name length: */
max_length = RESNAMESIZE - 1;

/*------calculate the function F3:-------------------------------------------*/

/* Initialize the extreme values: */
*min_functionP = +999999.0;
*max_functionP = -999999.0;

/* 20040910.0924: */
/* A window of 29 residues is scanned.  The total number of */
/* aromatic residues on the same side is counted for twelve */
/* subwindows, ranging from 18 to 29 residues.  The largest */
/* number of  aromatic residues  is used as function value. */

/* Initialize (reset) the function value: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function3P + residueI) = 0.0;
	}

/* Scan the sequence, ignoring 14 residues at each end: */
for (residueI = 14; residueI < residuesN - 14; residueI++)
	{
	/* Reset the number of residues used to prepare the raw values: */
	used_residuesN = 0;

        /* Reset the flaged hydrophobicities for all subwindows: */
	aromatic29 = 0;
	aromatic28 = 0;
	aromatic27 = 0;
	aromatic26 = 0;
	aromatic25 = 0;
	aromatic24 = 0;
	aromatic23 = 0;
	aromatic22 = 0;
	aromatic21 = 0;
	aromatic20 = 0;
	aromatic19 = 0;
	aromatic18 = 0;

	/* Scan the sliding window: */
	for (windowI = 0; windowI < 29; windowI++)
		{
		/* Prepare and check the combined index: */
		combinedI = residueI + windowI - 14;
		if (combinedI < 0) continue;
		if (combinedI >= residuesN) continue;

		/* Pointer  to  the residue  name of */
		/* the residue defined by combinedI: */
		residue_nameP = runtimeSP->sequenceP + combinedI * max_length;

		/* Copy the residue name: */
		strncpy (residue_nameA, residue_nameP, max_length);
		residue_nameA[max_length] = '\0';

		/* Now check the residue name. */
		aromaticF = IsAromatic_ (residue_nameA);

		/* Update all counts of aromatic residues: */
		aromatic29 += aromaticF * flag29A[windowI];
		aromatic28 += aromaticF * flag28A[windowI];
		aromatic27 += aromaticF * flag27A[windowI];
		aromatic26 += aromaticF * flag26A[windowI];
		aromatic25 += aromaticF * flag25A[windowI];
		aromatic24 += aromaticF * flag24A[windowI];
		aromatic23 += aromaticF * flag23A[windowI];
		aromatic22 += aromaticF * flag22A[windowI];
		aromatic21 += aromaticF * flag21A[windowI];
		aromatic20 += aromaticF * flag20A[windowI];
		aromatic19 += aromaticF * flag19A[windowI];
		aromatic18 += aromaticF * flag18A[windowI];

		/* Update the number of used residues: */
		used_residuesN++;
		}

	/* Check the number of used residues: */
	if (used_residuesN == 0) continue;

	/* The function value should be equal to */
	/* the highest  flaged hydrophobicity: */
	function_int_value = aromatic29;
	if (aromatic28 > function_int_value) function_int_value = aromatic28;
	if (aromatic27 > function_int_value) function_int_value = aromatic27;
	if (aromatic26 > function_int_value) function_int_value = aromatic26;
	if (aromatic25 > function_int_value) function_int_value = aromatic25;
	if (aromatic24 > function_int_value) function_int_value = aromatic24;
	if (aromatic23 > function_int_value) function_int_value = aromatic23;
	if (aromatic22 > function_int_value) function_int_value = aromatic22;
	if (aromatic21 > function_int_value) function_int_value = aromatic21;
	if (aromatic20 > function_int_value) function_int_value = aromatic20;
	if (aromatic19 > function_int_value) function_int_value = aromatic19;
	if (aromatic18 > function_int_value) function_int_value = aromatic18;

	/* Scale and store the function value: */
	function_double_value = (double) function_int_value;
	function_double_value /= 1.0;
	*(runtimeSP->function3P + residueI) = function_double_value;

	/* Dummy values should not be used to determine extreme values: */
	if (used_residuesN < 29) continue;

	/* Find the extreme values: */
	if (function_double_value < *min_functionP)
		{
		*min_functionP = function_double_value;
		}
	if (function_double_value > *max_functionP)
		{
		*max_functionP = function_double_value;
		}

	/* End of residueI loop: */
	}

/*---------------------------------------------------------------------------*/

return 1;
}

/*===========================================================================*/


