from TargetWrapper import TargetWrapper
from ControlWrapper import ControlWrapper
from config.StateSaver import StateSaver
from utils import dialog
from utils import Unit

import sys

# The name of the namespace where all XML elements are accessible.
_ROOT = "Dsp"


#
# Class for inline scripts together with their environment.
#
class Script:

    def __init__(self, dsp_id):

        # ID of the display
        self.__dsp_id = dsp_id

        # the state saver
        self.__state_saver = StateSaver(dsp_id)
        
        # the environment for this script
        self.__environment = {}

        # the list of loaded controls
        self.__loaded_controls = []

        # flag indicating whether the display has been stopped
        self.__is_stopped = False

        # setup a sandbox environment
        class Dsp: pass

        self.__environment["__builtins__"] = None
        self.__environment["__name__"] = "inline"  # required for classes
        self.__environment[_ROOT] = Dsp()
        # unit constructor and units
        self.__environment["Unit"] = Unit.Unit
        self.__environment["PX"] = Unit.UNIT_PX
        self.__environment["CM"] = Unit.UNIT_CM
        self.__environment["IN"] = Unit.UNIT_IN
        self.__environment["PT"] = Unit.UNIT_PT
        self.__environment["PERCENT"] = Unit.UNIT_PERCENT

        self.__environment["add_timer"] = self.__script_add_timer
        self.__environment["get_config"] = self.__script_get_config
        self.__environment["get_control"] = self.__script_get_control
        self.__environment["set_config"] = self.__script_set_config
        
        self.__environment["Null"] = Null

        # see end of file
        # WTF do we need True/False there ?
        self.__environment["False"] = False
        self.__environment["True"] = True
        self.__environment["abs"] = abs
        self.__environment["bool"] = bool
        self.__environment["callable"] = callable
        self.__environment["chr"] = chr
        self.__environment["classmethod"] = classmethod
        self.__environment["cmp"] = cmp
        self.__environment["complex"] = complex
        self.__environment["delattr"] = delattr
        self.__environment["dict"] = dict
        self.__environment["divmod"] = divmod
        self.__environment["enumerate"] = enumerate
        self.__environment["float"] = float
        self.__environment["getattr"] = getattr
        self.__environment["hasattr"] = hasattr
        self.__environment["hash"] = hash
        self.__environment["hex"] = hex
        self.__environment["id"] = id
        self.__environment["int"] = int
        self.__environment["isinstance"] = isinstance
        self.__environment["issubclass"] = issubclass
        self.__environment["iter"] = iter
        self.__environment["len"] = len
        self.__environment["list"] = list
        self.__environment["locals"] = locals
        self.__environment["long"] = long
        self.__environment["max"] = max
        self.__environment["min"] = min
        self.__environment["object"] = object
        self.__environment["oct"] = oct
        self.__environment["ord"] = ord
        self.__environment["property"] = property
        self.__environment["range"] = range
        self.__environment["reduce"] = reduce
        self.__environment["repr"] = repr
        self.__environment["round"] = round
        self.__environment["setattr"] = setattr
        self.__environment["staticmethod"] = staticmethod
        self.__environment["str"] = str
        self.__environment["sum"] = sum
        self.__environment["super"] = super
        self.__environment["tuple"] = tuple
        self.__environment["type"] = type
        self.__environment["unichr"] = unichr
        self.__environment["unicode"] = unicode
        self.__environment["vars"] = vars
        self.__environment["xrange"] = xrange
        self.__environment["zip"] = zip


    #
    # Runs a timer in the sandbox. The timer stops when the script is being
    # stopped.
    #
    def __script_add_timer(self, interval, callback, *args):

        def f():
            try:
                if (self.__is_stopped):
                    return False
                else:
                    ret = callback(*args)
                    return ret

            except:
                import traceback; traceback.print_exc()
                print >>sys.stderr, \
                  "Error in script. An error occured while executing a script."
                dialog.warning(_("Error in script"),
                               _("An error occured while executing a script."))
                return False


        import gtk
        gtk.timeout_add(interval, f)


    #
    # Retrieves a configuration value.
    #
    def __script_get_config(self, key, default = None):

        return self.__state_saver.get_key(key, default)


    #
    # Stores a configuration value.
    #
    def __script_set_config(self, key, value):

        self.__state_saver.set_key(key, value)


    #
    # Returns a control readily wrapped for the sandbox.
    #
    def __script_get_control(self, interface):

        # FIXME: ensure that this does not break the sandbox
        from factory.ControlFactory import ControlFactory
        factory = ControlFactory()
        ctrl = factory.get_control(interface)
        self.__loaded_controls.append(ctrl)
        wrapped = ControlWrapper(ctrl)
        return wrapped



    #
    # Stops this scripting object.
    #
    def stop(self):

        self.__is_stopped = True
        for c in self.__loaded_controls:
            try:
                c.stop()
            except StandardError, exc:
                import traceback; traceback.print_exc()
                print >>sys.stderr, "Could not stop control %s" % (c,)


    #
    # Removes this scripting object and its state.
    #
    def remove(self):

        self.__state_saver.remove()
        

    #
    # Adds the given target to the environment.
    #
    def add_target(self, name, target, direct = False):

        if (direct):
            self.add_element(name, TargetWrapper(target), True)
            return

        index_path = target.get_index_path()
        length = len(index_path)

        if ("#" in name):
            name = name[:name.find("#")]

        if (not length):
            self.add_element(name, TargetWrapper(target))

        else:
            try:
                lst = getattr(self.__environment[_ROOT], name)
            except:
                lst = []
            self.add_element(name, lst)

            for index in index_path[:-1]:
                while (len(lst) - 1 < index): lst.append([])
                lst = lst[index]
            index = index_path[-1]
            while (len(lst) - 1 < index): lst.append([])
            lst[index] = TargetWrapper(target)



    #
    # Adds the given element to the environment. Unless the "direct" flag is
    # set, the element becomes a member of the XML objects namespace.
    # If the flag is set, the element becomes a member of the global namespace.
    #
    def add_element(self, name, elem, direct = False):

        if (direct):
            self.__environment[name] = elem
        else:
            setattr(self.__environment[_ROOT], name, elem)



    #
    # Fixes the indentation of Python code.
    #
    def __fix_indentation(self, code):

        lines = code.splitlines()
        min_indent = len(code)
        # find the minimal indentation
        for l in lines:
            if (not l.strip()): continue
            this_indent = len(l) - len(l.lstrip())
            min_indent = min(min_indent, this_indent)

        # apply the minimal indentation
        out = ""
        for l in lines:
            out += l[min_indent:] + "\n"

        return out



    #
    # Executes a block of script.
    #
    def execute(self, script):

        # get the block into shape
        script = self.__fix_indentation(script)

        # compile
        try:
            pycode = compile(script, '<inline script>', 'exec')
            
        except SyntaxError, exc:
            import traceback; traceback.print_exc()
            print >>sys.stderr
            print >>sys.stderr, "SyntaxError in script:\n"
            print >>sys.stderr, "> %s" % (str(exc).replace("\n", "\n> "))
            print >>sys.stderr
            dialog.warning(_("Error in script"),
                           _("An error occured while loading a script."))
            return

        # run
        try:
            exec pycode in self.__environment

        except StandardError, exc:
            import traceback; traceback.print_exc()
            print >>sys.stderr
            print >>sys.stderr, "Error in script:\n"
            print >>sys.stderr, "> %s" % (str(exc).replace("\n", "\n> "))
            print >>sys.stderr
            dialog.warning(_("Error in script"),
                           _("An error occured while executing a script."))



    #
    # Retrieves the value of the given object from the scripting environment.
    #
    def get_value(self, name):

        cmd = "__retrieve__ = %s" % (name)
        self.execute(cmd)
        try:
            return self.__environment.pop("__retrieve__")
        except:
            return

        return value


    #
    # Sets the value of the given object in the scripting environment.
    #
    def set_value(self, name, value):

        cmd = "%s = %s" % (name, value)
        self.execute(cmd)


    #
    # Calls the given function in the sandbox.
    #
    def call_function(self, name, *args):

        func = self.__environment.get(name, Null)
        try:
            func(*args)
        except:
            import traceback; traceback.print_exc()
            print >>sys.stderr, "A function call in the inline script failed."


    #BEGIN
    # + 2.3.3 http://python.org/doc/2.3.3/lib/built-in-funcs.html
    #                        vs.
    # - 2.2.3 http://python.org/doc/2.2.3/lib/built-in-funcs.html
    # * dangerous

    # * __import__
    # abs
    # - apply
    # + basestring
    # bool
    # - buffer
    # callable
    # chr
    # classmethod
    # cmp
    # - coerce
    # * compile
    # complex
    # delattr
    # dict
    # dir # ?
    # divmod
    # + enumerate
    # * eval
    # * execfile
    # * file
    # filter # may be should print a warning and a link to list-comprehension
    # float
    # getattr
    # globals # ?
    # hasattr
    # hash
    # help # not very useful
    # hex
    # id
    # - intern
    # input # not very useful
    # int
    # isinstance
    # issubclass
    # iter
    # len
    # list
    # locals
    # long
    # map # maybe should print a warning and a link to list-comprehension
    # max
    # min
    # + object
    # oct
    # * open
    # ord
    # property
    # range
    # raw_input # not very usefull
    # reduce
    # * reload
    # repr
    # round
    # setattr
    # - slice
    # staticmethod
    # str
    # + sum
    # super
    # tuple
    # type
    # unichr
    # unicode
    # vars
    # xrange
    # zip

    # 2.3 Deprecated/Non-essentials functions
    # http://python.org/doc/2.3.3/lib/non-essential-built-in-funcs.html
    # - apply
    # - buffer
    # - coerce
    # - intern
    #END
