// QM1MDL.CPP

// Copyright (C) 2000 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "qm1mdl.h"	// config.h is here -> we get ENABLE-macros here...

#include "mm1mdl.h"	// for import...

#include "qm1e_mopac.h"
#include "qm1e_mpqc.h"

#include <fstream>
#include <strstream>
using namespace std;

/*################################################################################################*/

qm1_atom::qm1_atom(void)
{
	selected = false;
}

qm1_atom::qm1_atom(element p1, fGL * p2, i32u p3)
{
	el = p1;
	
	fGL_a3 data = { p2[0], p2[1], p2[2] };
	for (i32u n1 = 0;n1 < p3;n1++) crd_vector.push_back(data);
	
	selected = false;
}

qm1_atom::qm1_atom(const qm1_atom & p1)
{
	el = p1.el;
	crd_vector = p1.crd_vector;
	
	selected = p1.selected;
}

qm1_atom::~qm1_atom(void)
{
}

// for qm1_atoms, equality is tested using pointers -> you can find the
// iterator (using STL's find-function) if the pointer is known.

bool qm1_atom::operator==(const qm1_atom & p1) const
{
	return (this == (& p1));
}

/*################################################################################################*/

qm1_bond::qm1_bond(void)
{
	atmr[0] = atmr[1] = NULL;
}

qm1_bond::qm1_bond(qm1_atom * p1, qm1_atom * p2, bondtype p3)
{
	atmr[0] = p1; atmr[1] = p2; bt = p3;
}

qm1_bond::qm1_bond(const qm1_bond & p1)
{
	atmr[0] = p1.atmr[0]; atmr[1] = p1.atmr[1]; bt = p1.bt;
}

qm1_bond::~qm1_bond(void)
{
}

// for bonds, equality is tested using atom pointers. if you need to find a certain bond,
// just make a temporary mm1_bond and use that to find the original one (using STL's find-function).

bool qm1_bond::operator==(const qm1_bond & p1) const
{
	if (atmr[0] == p1.atmr[0] && atmr[1] == p1.atmr[1]) return true;
	if (atmr[0] == p1.atmr[1] && atmr[1] == p1.atmr[0]) return true;
	
	return false;
}

/*################################################################################################*/

qm1_mdl::qm1_mdl(ostream * p1, class_factory & p2) :
	model_extended(p1, p2), model_simple(p1, p2), all_atoms_interface()
{
	current_eng = NULL;
	
	total_charge = 0;
	current_orbital = 0;
	
	default_eng = 0;
}

qm1_mdl::~qm1_mdl(void)
{
	if (current_eng != NULL) delete current_eng;
}

const char * qm1_mdl::GetProjectFileNameExtension(void)
{
	static const char ext[] = "gpr";
	return ext;
}

bool qm1_mdl::CheckEngSettings(void)
{
	i32s electron_count = 0;
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		electron_count += (* it1).el.GetAtomicNumber();
	}
	
	// calculate the total number of electrons in the system.
	// calculate the total number of electrons in the system.
	// calculate the total number of electrons in the system.
	
	i32s total_electron_count = electron_count - total_charge;
	
	// checks start now...
	// checks start now...
	// checks start now...
	
	if (total_electron_count < 1)
	{
		err->ErrorMessage("Less than one electron in the system!\nPlease check the \"total charge\" setting.");
		return false;
	}
	
	if (total_electron_count % 2)
	{
		err->ErrorMessage("Odd number of electrons in the system!\nOnly singlet states with an even number\nof electrons are supported at the moment.\nPlease check the \"total charge\" setting.");
		return false;
	}
	
	// passed...
	// passed...
	// passed...
	
	return true;
}

/*##############################################*/
/*##############################################*/

const char * qm1_mdl::engtab1[] =
{
	"qm1_eng_mopac : MOPAC7 / MNDO",
	"qm1_eng_mopac : MOPAC7 / MINDO/3",
	"qm1_eng_mopac : MOPAC7 / AM1",
	"qm1_eng_mopac : MOPAC7 / PM3",
	
#ifdef ENABLE_MPQC

	"qm1_eng_mpqc : MPQC / STO-3G",
	"qm1_eng_mpqc : MPQC / STO-6G",
	"qm1_eng_mpqc : MPQC / 3-21G",
	"qm1_eng_mpqc : MPQC / 3-21G*",
	"qm1_eng_mpqc : MPQC / 4-31G",
	"qm1_eng_mpqc : MPQC / 4-31G*",
	"qm1_eng_mpqc : MPQC / 4-31G**",
	"qm1_eng_mpqc : MPQC / 6-31G",
	"qm1_eng_mpqc : MPQC / 6-31G*",
	"qm1_eng_mpqc : MPQC / 6-31G**",
	
#endif	// ENABLE_MPQC

	NULL
};

const i32s qm1_mdl::engtab2[] =
{
	(ENG_QM1_MOPAC | MOPAC_MNDO),
	(ENG_QM1_MOPAC | MOPAC_MINDO3),
	(ENG_QM1_MOPAC | MOPAC_AM1),
	(ENG_QM1_MOPAC | MOPAC_PM3),
	
#ifdef ENABLE_MPQC

	(ENG_QM1_MPQC | MPQC_STO3G),
	(ENG_QM1_MPQC | MPQC_STO6G),
	(ENG_QM1_MPQC | MPQC_3_21G),
	(ENG_QM1_MPQC | MPQC_3_21GS),
	(ENG_QM1_MPQC | MPQC_4_31G),
	(ENG_QM1_MPQC | MPQC_4_31GS),
	(ENG_QM1_MPQC | MPQC_4_31GSS),
	(ENG_QM1_MPQC | MPQC_6_31G),
	(ENG_QM1_MPQC | MPQC_6_31GS),
	(ENG_QM1_MPQC | MPQC_6_31GSS),
	
#endif	// ENABLE_MPQC

	NOT_DEFINED
};

qm1_eng * qm1_mdl::CreateDefaultEngine(void)
{
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	// perform a sanity check of settings, and return NULL if failed. the user will be notified...
	
	if (CheckEngSettings() == false) return NULL;
	
	i32s engtype1 = (engtab2[default_eng] & 0xff00);
	i32s engtype2 = (engtab2[default_eng] & 0x00ff);
	
	switch (engtype1)
	{

#ifdef ENABLE_MPQC

		case ENG_QM1_MPQC:
		return new qm1_eng_mpqc(* this, engtype2);
		
#endif	// ENABLE_MPQC

		default:	// ENG_QM1_MOPAC
		if (qm1_eng_mopac::GetLock() != NULL)
		{
			err->ErrorMessage("MOPAC lock failed!!!\nCan't run multiple MOPAC calculations.");
			return NULL;
		}
		else return new qm1_eng_mopac(* this, engtype2);
	}
}

void qm1_mdl::DiscardCurrentEng(void)
{
	if (current_eng != NULL)
	{
		delete current_eng;
		current_eng = NULL;
	}
}

void qm1_mdl::SetupPlotting(void)
{
	if (current_eng != NULL) current_eng->SetupPlotting();
}

/*##############################################*/
/*##############################################*/

// this stuff is pretty much stolen from the mm1_mdl...
// this stuff is pretty much stolen from the mm1_mdl...
// this stuff is pretty much stolen from the mm1_mdl...

void qm1_mdl::PushCRDSets(i32u p1)
{
	for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back(new crd_set());
	
	fGL_a3 newcrd = { 0.0, 0.0, 0.0 };
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32u n1 = 0;n1 < p1;n1++)
		{
			(* it1).crd_vector.push_back(newcrd);
		}
	}
}

void qm1_mdl::PopCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::CopyCRDSet(i32u p1, i32u p2)
{
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		(* it1).crd_vector[p2][0] = (* it1).crd_vector[p1][0];
		(* it1).crd_vector[p2][1] = (* it1).crd_vector[p1][1];
		(* it1).crd_vector[p2][2] = (* it1).crd_vector[p1][2];
	}
}

void qm1_mdl::SwapCRDSets(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void qm1_mdl::CenterCRDSet(i32u p1)
{
	fGL sum[3] = { 0.0, 0.0, 0.0 };
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++) sum[n1] += (* it1).crd_vector[p1][n1];
	}
	
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			fGL tmp1 = sum[n1] / (f64) atom_list.size();
			(* it1).crd_vector[p1][n1] -= tmp1;
		}
	}
}

void qm1_mdl::ReserveCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

/*##############################################*/
/*##############################################*/

void qm1_mdl::AddAtom(qm1_atom & p1)
{
	DiscardCurrentEng();
	atom_list.push_back(p1);
}

void qm1_mdl::RemoveAtom(iter_qm1al it1)
{
	DiscardCurrentEng();
	
	// before removing an atom, also remove all bonds that relate to this atom.
	// if a bond is removed, the search is restarted; not very efficient but should not be a bottleneck...
	
	iter_qm1bl it2 = bond_list.begin();
	while (it2 != bond_list.end())
	{
		bool must_be_removed = false;
		if ((* it2).atmr[0] == & (* it1)) must_be_removed = true;
		if ((* it2).atmr[1] == & (* it1)) must_be_removed = true;
		
		if (must_be_removed)
		{
			bond_list.erase(it2);
			it2 = bond_list.begin();	// restart!!!
		}
		else it2++;
	}
	
	atom_list.erase(it1);
}

void qm1_mdl::AddBond(qm1_bond & p1)
{
	bond_list.push_back(p1);
}

void qm1_mdl::RemoveBond(iter_qm1bl it1)
{
	bond_list.erase(it1);
}

/*##############################################*/
/*##############################################*/

bool qm1_mdl::ReadStream(istream & ifile, bool selected, bool)
{
	mm1_mdl * mdl = new mm1_mdl(NULL, (* factory));
	bool retval = mdl->ReadStream(ifile, selected);
	
// the crd-set handling here might have problems if QM model has more sets than MM model...
// the crd-set handling here might have problems if QM model has more sets than MM model...
// the crd-set handling here might have problems if QM model has more sets than MM model...
	
	i32s new_crd_sets = mdl->cs_vector.size() - cs_vector.size();
	if (new_crd_sets > 0)
	{
		PushCRDSets(new_crd_sets);
		for (i32u n1 = 1;n1 < cs_vector.size();n1++) SetCRDSetVisible(n1, true);
	}
	
	vector<mm1_atom *> mmtab;	// store the pointers here...
	vector<qm1_atom *> qmtab;	// ...to replicate the bonds!
		
	iter_mm1al it1 = mdl->atom_list.begin();
	while (it1 != mdl->atom_list.end())
	{
		element tmp1 = (* it1).el;
		fGL * tmp2 = (* it1).crd_vector[0].data;
		
		qm1_atom newatom(tmp1, tmp2, (* it1).crd_vector.size());
		for (i32u n1 = 0;n1 < (* it1).crd_vector.size();n1++)		// copy also the other crd-sets, if needed...
		{
			newatom.crd_vector[n1].data[0] = (* it1).crd_vector[n1].data[0];
			newatom.crd_vector[n1].data[1] = (* it1).crd_vector[n1].data[1];
			newatom.crd_vector[n1].data[2] = (* it1).crd_vector[n1].data[2];
		}
		
		AddAtom(newatom);
		
		mmtab.push_back(& (* it1));
		qmtab.push_back(& atom_list.back());
		
		it1++;
	}
	
	iter_mm1bl it2 = mdl->bond_list.begin();
	while (it2 != mdl->bond_list.end())
	{
		i32u index1 = 0; while (index1 < mmtab.size()) if ((* it2).atmr[0] == mmtab[index1]) break; else index1++;
		if (index1 == mmtab.size()) { cout << "FATAL ERROR : could not find atom #1." << endl; exit(EXIT_FAILURE); }
		
		i32u index2 = 0; while (index2 < mmtab.size()) if ((* it2).atmr[1] == mmtab[index2]) break; else index2++;
		if (index2 == mmtab.size()) { cout << "FATAL ERROR : could not find atom #2." << endl; exit(EXIT_FAILURE); }
		
		bondtype tmp1 = (* it2).bt;
		
		qm1_bond newbond(qmtab[index1], qmtab[index2], tmp1);
		bond_list.push_back(newbond);
		
		it2++;
	}
	
	delete mdl;
	
	return retval;
}

void qm1_mdl::WriteStream(ostream & ostr)
{
	mm1_mdl * mdl = new mm1_mdl(NULL, (* factory));
	
	mdl->aai_MakeCopy(this);
	mdl->WriteStream(ostr);
	
	delete mdl;
}

/*##############################################*/
/*##############################################*/

iter_qm1al qm1_mdl::aai_FindA(i32s atmi) //const
{
	if (atmi < 0) return GetAtomsEnd();				// out of valid range...
	if (atmi >= (i32s) aai_GetAtomCount()) return GetAtomsEnd();	// out of valid range...
	
	i32s counter = 0;
	iter_qm1al it1 = GetAtomsBegin();
	
	while (counter < atmi)
	{
		counter++;
		it1++;
	}
	
	return it1;
}

iter_qm1bl qm1_mdl::aai_FindB(i32s bndi) //const
{
	if (bndi < 0) return GetBondsEnd();				// out of valid range...
	if (bndi >= (i32s) aai_GetBondCount()) return GetBondsEnd();	// out of valid range...
	
	i32s counter = 0;
	iter_qm1bl it1 = GetBondsBegin();
	
	while (counter < bndi)
	{
		counter++;
		it1++;
	}
	
	return it1;
}

i32u qm1_mdl::aai_GetAtomCount(void) //const
{
	return atom_list.size();
}

i32u qm1_mdl::aai_GetBondCount(void) //const
{
	return bond_list.size();
}

void * qm1_mdl::aai_FindAtomByInd(i32s atmi) //const
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return NULL;
	else return (void *) & (* it1);
}

i32s qm1_mdl::aai_FindAtomByPtr(void * atmr) //const
{
	i32s counter = 0;
	iter_qm1al it1 = GetAtomsBegin();
	
	while (it1 != GetAtomsEnd())
	{
		void * tmpr = & (* it1);
		if (atmr == tmpr) return counter;
		
		counter++;
		it1++;
	}
	
	return -1;	// not found...
}

i32s qm1_mdl::aai_FindBond(i32s atmi1, i32s atmi2) //const
{
	void * atmr1 = aai_FindAtomByInd(atmi1);
	if (atmr1 == NULL) return -1;
	
	void * atmr2 = aai_FindAtomByInd(atmi2);
	if (atmr2 == NULL) return -1;
	
	i32s counter = 0;
	iter_qm1bl it1 = GetBondsBegin();
	
	while (it1 != GetBondsEnd())
	{
		void * tmpr1 = (* it1).atmr[0];
		void * tmpr2 = (* it1).atmr[1];
		
		if (atmr1 == tmpr1 && atmr2 == tmpr2) return counter;
		if (atmr1 == tmpr2 && atmr2 == tmpr1) return counter;
		
		counter++;
		it1++;
	}
	
	return -1;	// not found...
}

void * qm1_mdl::aai_AddAtom(element el, fGL * xyz)
{
	qm1_atom newatom(el, xyz, cs_vector.size());
	AddAtom(newatom);
	
	return (void *) & atom_list.back();
}

bool qm1_mdl::aai_RemoveAtom(i32s atmi)
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	RemoveAtom(it1);
	
	return true;
}

bool qm1_mdl::aai_AddBond(i32s atmi1, i32s atmi2, bondtype bt)
{
	void * atmr1 = aai_FindAtomByInd(atmi1);
	if (atmr1 == NULL) return false;
	
	void * atmr2 = aai_FindAtomByInd(atmi2);
	if (atmr2 == NULL) return false;
	
	qm1_bond newbond((qm1_atom *) atmr1, (qm1_atom *) atmr2, bt);
	AddBond(newbond);
	
	return true;
}

bool qm1_mdl::aai_RemoveBond(i32s bndi)
{
	iter_qm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	RemoveBond(it1);
	
	return true;
}

bool qm1_mdl::aai_AtomGetElement(i32s atmi, element & el) //const
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	el = (* it1).el;
	
	return true;
}

bool qm1_mdl::aai_AtomSetElement(i32s atmi, element el)
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	(* it1).el = el;
	
	return true;
}

bool qm1_mdl::aai_AtomGetXYZ(i32s atmi, fGL * xyz) //const
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	xyz[0] = (* it1).crd_vector[0].data[0];
	xyz[1] = (* it1).crd_vector[0].data[1];
	xyz[2] = (* it1).crd_vector[0].data[2];
	
	return true;
}

bool qm1_mdl::aai_AtomSetXYZ(i32s atmi, const fGL * xyz)
{
	iter_qm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
//	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	i32s n1 = 0;	// how to handle this properly?!?!?!
	{
		(* it1).crd_vector[n1].data[0] = xyz[0];
		(* it1).crd_vector[n1].data[1] = xyz[1];
		(* it1).crd_vector[n1].data[2] = xyz[2];
	}
	
	return true;
}

bool qm1_mdl::aai_BondGetBondType(i32s bndi, bondtype & bt) //const
{
	iter_qm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	bt = (* it1).bt;
	
	return true;
}

bool qm1_mdl::aai_BondSetBondType(i32s bndi, bondtype bt)
{
	iter_qm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	(* it1).bt = bt;
	
	return true;
}

bool qm1_mdl::aai_BondGetAtomIndices(i32s bndi, i32s * iii) //const
{
	iter_qm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	iii[0] = aai_FindAtomByPtr((void *) (* it1).atmr[0]);
	iii[1] = aai_FindAtomByPtr((void *) (* it1).atmr[1]);
	
	return true;
}

/*##############################################*/
/*##############################################*/

void qm1_mdl::DoEnergy(void)
{
	if (!current_eng) current_eng = CreateDefaultEngine();
	if (!current_eng) return;	// sanity check failed...
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Calculating Energy ";
	str1 << "(quantum mechanics";
	str1 << ", engine = " << engtab1[default_eng];
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, current_eng, 0); current_eng->Compute(0);
	
	char buffer[128];
	ostrstream str(buffer, sizeof(buffer)); str.setf(ios::fixed); str.precision(8);
	
	str << "Energy = " << current_eng->energy << " kJ/mol" << endl << ends;
	
	PrintToLog(buffer);
	
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	
	SetupPlotting();
}

void qm1_mdl::DoGeomOpt(qm1_geomopt_param & param)
{
	GeomOptGetParam(param);
	if (!param.confirm) return;
	
	if (!current_eng) current_eng = CreateDefaultEngine();
	if (!current_eng) return;	// sanity check failed...
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Starting Geometry Optimization ";
	str1 << "(quantum mechanics";
	str1 << ", engine = " << engtab1[default_eng];
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, current_eng, 0);
	
	qm1_geomopt * opt = new qm1_geomopt(current_eng, 20, 0.0125);	// optimal settings?!?!?
	
	char buffer[1024];
	f64  last_energy = 0.0;		// this is for output and delta_e test...
	
	PrintToLog("Cycle    Energy    Gradient       Step        Delta E\n");
	
	i32s n1 = 0;	// n1 counts the number of steps...
	while (true)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		
// problem: the gradient information is in fact not precise in this stage. the current gradient
// is the one that was last calculated in the search, and it is not necessarily the best one.
// to update the gradient, we need to Compute(1) here...
	current_eng->Compute(1);	// this is not vital, but will update the gradient vector length...
// at MM side the geomopt is able to bring the gradients to level ~1.0e-7, but here at QM side the final
// gradients seem to be ~1.0e+0 that are *very* much farther away from zero. why is that?!?!?!
		
		if (n1 != 0)
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e %10.4e \n", n1,
			    opt->optval, current_eng->GetGradientVectorLength(),
			    opt->optstp, last_energy - opt->optval);
		  }
		else
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e ********** \n", n1,
			    opt->optval, current_eng->GetGradientVectorLength(),
			    opt->optstp);
		  }
		PrintToLog(buffer);
		
		bool terminate = false;
		
		if (param.enable_nsteps)	// the nsteps test...
		{
			if (n1 >= param.treshold_nsteps)
			{
				terminate = true;
				PrintToLog("the nsteps termination test was passed.\n");
			}
		}
		
		if (param.enable_grad)		// the grad test...
		{
			if (current_eng->GetGradientVectorLength() < param.treshold_grad)
			{
				terminate = true;
				PrintToLog("the grad termination test was passed.\n");
			}
		}
		
		if (param.enable_delta_e)	// the delta_e test...
		{
			bool flag = false; const f64 treshold_step = 1.0e-12;		// can we keep this as a constant???
			if (n1 != 0 && (last_energy - opt->optval) != 0.0 && fabs(last_energy - opt->optval) < param.treshold_delta_e) flag = true;
			if ((opt->optstp != 0.0) && (opt->optstp < treshold_step)) flag = true;
			
			if (flag)
			{
				terminate = true;
				PrintToLog("the delta_e termination test was passed.\n");
			}
		}
		
		last_energy = opt->optval;
		
		if (!(n1 % 10) || terminate)
		{
			CopyCRD(current_eng, this, 0); CenterCRDSet(0);
			UpdateAllGraphicsViews(true);
		}
		
		if (terminate) break;		// exit the loop here!!!
		
		n1++;	// update the number of steps...
	}
	
	delete opt;
	
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	// we will not delete current_eng here, so that we can draw plots using it...
	
	SetupPlotting();
}

void qm1_mdl::GeomOptGetParam(qm1_geomopt_param & param)
{
	param.confirm = true;
}

/*################################################################################################*/

// eof
