// QM1GUI.CPP

// Copyright (C) 2000 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "qm1docv.h"	// config.h is here -> we get ENABLE-macros here...

#ifdef ENABLE_GRAPHICS
#include "qm1e_mopac.h"
#include "qm1e_mpqc.h"

#include "plane.h"
#include "surface.h"

//#include "Vector.h" // OELib utils for torsion, angle measure
//using namespace std;
//using namespace OpenEye;

#include "color.h"
#include "views.h"

#include <stdlib.h>

#include <strstream>
using namespace std;

/*################################################################################################*/

qm1_cm_element qm1_docv::cm_element = qm1_cm_element();

qm1_docv::qm1_docv(ostream * p1, graphics_class_factory & p2) :
	docview(p1, p2), qm1_mdl(p1, p2), model_simple(p1, p2)
{
}

qm1_docv::~qm1_docv(void)
{
}

void qm1_docv::DiscardCurrentEng(void)
{
	i32u n1 = 0;
	while (n1 < object_vector.size())
	{
		bool flag = false;
		
		color_plane_object * cpo_ref = dynamic_cast<color_plane_object *>(object_vector[n1]);
		if (cpo_ref != NULL && cpo_ref->GetRef() == current_eng) flag = true;
		
		volume_rendering_object * vro_ref = dynamic_cast<volume_rendering_object *>(object_vector[n1]);
		if (vro_ref != NULL && vro_ref->GetRef() == current_eng) flag = true;
		
		color_surface_object * cso_ref = dynamic_cast<color_surface_object *>(object_vector[n1]);
		if (cso_ref != NULL && cso_ref->GetRef() == current_eng) flag = true;
		
		if (flag)
		{
			if (selected_object == object_vector[n1]) selected_object = NULL;
			RemoveObject(object_vector[n1]);
		}
		else n1++;
	}
	
	qm1_mdl::DiscardCurrentEng();
	
	// we also call UpdatePlottigViews() here, to update the energy-level diagram views we might have...
	// we also call UpdatePlottigViews() here, to update the energy-level diagram views we might have...
	// we also call UpdatePlottigViews() here, to update the energy-level diagram views we might have...
	
	UpdatePlottingViews();
}

void qm1_docv::SetupPlotting(void)
{
	qm1_mdl::SetupPlotting();
	
	// here we also send a notification to the energy-level diagrams we might have...
	// here we also send a notification to the energy-level diagrams we might have...
	// here we also send a notification to the energy-level diagrams we might have...
	
	UpdatePlottingViews();
}

fGL qm1_docv::GetDefaultFocus(void)
{
	return 2.0;
}

const char * qm1_docv::GetType(void)
{
	return "QM, molecular calculations";
}

color_mode * qm1_docv::GetDefaultColorMode(void)
{
	return & qm1_docv::cm_element;
}

void qm1_docv::SelectAll(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_qm1al it1 = atom_list.begin();
	while (it1 != atom_list.end()) (* it1++).selected = true;
	
	UpdateAllGraphicsViews();
}

void qm1_docv::InvertSelection(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_qm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		bool flag = (* it1).selected;
		(* it1++).selected = !flag;
	}
	
	UpdateAllGraphicsViews();
}

//void qm1_docv::SelectNone(void)
//{
//	if (selected_object != NULL)
//	{
//		selected_object = NULL;
//		event_SelectedObjectChanged();
//	}
//	
//	iter_qm1al it1 = atom_list.begin();
//	while (it1 != atom_list.end()) (* it1++).selected = false;
//	
//	UpdateAllGraphicsViews();
//}

bool qm1_docv::TestAtom(qm1_atom * ref, rmode rm)
{
	// what about the "visible"-flag???
	
	if (rm == Transform1 && ref->selected) return false;
	if (rm == Transform2 && !ref->selected) return false;
	
	return true;
}

void qm1_docv::SetColor(color_mode * cm, qm1_atom * ref)
{
	if (ref->selected) glColor3f(1.0, 0.0, 1.0);
	else
	{
		fGL_a4 color;
		cm->GetColor(ref, color, model_prefs);
		glColor3fv(color);
	}
}

void qm1_docv::Render(graphics_view * gv, rmode rm)
{
bool accum = gv->accumulate; if (rm != Normal) accum = false;
//if (accum) { glClear(GL_ACCUM_BUFFER_BIT); UpdateAccumValues(); }
//else if (rm != Transform2) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	if (gv->enable_fog) glEnable(GL_FOG);
	
	/*################*/
	/*################*/
	
	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	{
		if (!GetCRDSetVisible(n1)) continue;
if (accum) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);		// FIXME!!!

		if (gv->render == RENDER_WIREFRAME)
		{
			glPointSize(3.0); glLineWidth(1.0);
			for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
			{
				if (!TestAtom(& (* it1), rm)) continue;
				
				glPushName(GLNAME_MD_TYPE1);
				glPushName((i32u) & (* it1));
				
				glBegin(GL_POINTS);
				SetColor(gv->colormode, & (* it1));
				glVertex3fv((* it1).crd_vector[n1].data);
				glEnd();
				
				glPopName();
				glPopName();
			}
		}
		
		if (gv->render != RENDER_WIREFRAME && gv->render != RENDER_NOTHING)
		{
			glEnable(GL_LIGHTING);
			
			for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
			{
				if (!TestAtom(& (* it1), rm)) continue;
				
				SetColor(gv->colormode, & (* it1));
				
					float rad = model_prefs->Double("QM1Graphics/BallSize", 0.035);
					if (model_prefs->Boolean("QM1Graphics/BallVdWScale", true))
					    rad *= (* it1).el.GetVDWRadius() * 4.0;
					int res = model_prefs->Value("QM1Graphics/BallResolution", 12);
				
				glPushName(GLNAME_MD_TYPE1); glPushName((i32u) & (* it1));
				
				GLUquadricObj * qo = gluNewQuadric();
				gluQuadricDrawStyle(qo, (GLenum) GLU_FILL);
				glPushMatrix();
				glTranslated((* it1).crd_vector[n1][0], (* it1).crd_vector[n1][1], (* it1).crd_vector[n1][2]);
				gluSphere(qo, rad, res, res / 2);
				glPopMatrix();
				gluDeleteQuadric(qo);
				
				glPopName(); glPopName();
			}
			
			glDisable(GL_LIGHTING);
		}
		
		if (gv->render != RENDER_NOTHING)
		{
			glEnable(GL_LINE_STIPPLE);
			
			for (iter_qm1bl it2 = bond_list.begin();it2 != bond_list.end();it2++)
			{
		//		switch ((* it2).bt.GetSymbol1())
		//		{
		//			case 'S': glLineStipple(1, 0xFFFF); break;
		//			case 'C': glLineStipple(1, 0x3FFF); break;
		//			case 'D': glLineStipple(1, 0x3F3F); break;
		//			case 'T': glLineStipple(1, 0x3333); break;
		//		}
				
				glLineStipple(1, 0x3FFF);	// like in 'C' case...
				
				glBegin(GL_LINES);
				SetColor(gv->colormode, (* it2).atmr[0]);
				glVertex3fv((* it2).atmr[0]->crd_vector[n1].data);
				SetColor(gv->colormode, (* it2).atmr[1]);
				glVertex3fv((* it2).atmr[1]->crd_vector[n1].data);
				glEnd();
			}
			
			glDisable(GL_LINE_STIPPLE);
		}
		
		if (accum)
		{
			gv->cam->RenderObjects(gv);
			glAccum(GL_ACCUM, cs_vector[n1]->accum_value);
		}
	}
	
	/*################*/
	/*################*/
	
	if (accum) glAccum(GL_RETURN, 1.0);
	else if (rm != Transform2) gv->cam->RenderObjects(gv);
	
	if (gv->label == LABEL_ELEMENT)
	{
		glColor3f(0.0, 1.0, 1.0); char string[32];
		for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << (* it1).el.GetSymbol() << ends;
			
			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	
	if (gv->enable_fog) glDisable(GL_FOG);
	
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	
	RenderAllTPs(gv, rm);
}

void qm1_docv::Center(transformer * p1)
{
	i32s sum = 0;
	p1->GetLocDataRW()->crd[0] = 0.0;
	p1->GetLocDataRW()->crd[1] = 0.0;
	p1->GetLocDataRW()->crd[2] = 0.0;
	
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			sum++;
			p1->GetLocDataRW()->crd[0] += (* it1).crd_vector[n1][0];
			p1->GetLocDataRW()->crd[1] += (* it1).crd_vector[n1][1];
			p1->GetLocDataRW()->crd[2] += (* it1).crd_vector[n1][2];
		}
	}
	
	if (!sum) return;
	
	p1->GetLocDataRW()->crd[0] /= (fGL) sum;
	p1->GetLocDataRW()->crd[1] /= (fGL) sum;
	p1->GetLocDataRW()->crd[2] /= (fGL) sum;
	
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			(* it1).crd_vector[n1][0] -= p1->GetLocData()->crd[0];
			(* it1).crd_vector[n1][1] -= p1->GetLocData()->crd[1];
			(* it1).crd_vector[n1][2] -= p1->GetLocData()->crd[2];
		}
	}
}

void qm1_docv::Transform(transformer * p1)
{
	fGL matrix[16]; p1->GetMatrix(matrix);
	
	for (iter_qm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			v3d<fGL> posv = v3d<fGL>((* it1).crd_vector[n1].data);
			TransformVector(posv, matrix);
			
			(* it1).crd_vector[n1][0] = posv[0];
			(* it1).crd_vector[n1][1] = posv[1];
			(* it1).crd_vector[n1][2] = posv[2];
		}
	}
}

// this comes directly from mm1_mdl, the bonds are just disabled...
// this comes directly from mm1_mdl, the bonds are just disabled...
// this comes directly from mm1_mdl, the bonds are just disabled...

void qm1_docv::DrawEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	i32s mouse[2] =
	{
		gv->current_tool->latest_x,
		gv->current_tool->latest_y
	};
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (qm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			qm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			
			AddAtom(newatom);	// DiscardCurrentEng() is called here...
			
			draw_data[0] = & atom_list.back();
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (qm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			qm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			
			AddAtom(newatom);	// DiscardCurrentEng() is called here...
			
			draw_data[1] = & atom_list.back();
		}
		
		// if different: update bondtype or add a new bond.
		// if not different: change atom to different element.
		
		if (draw_data[0] != draw_data[1])
		{
	//		mm1_bond newbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
	//		iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), newbond);
	//		if (it1 != bond_list.end()) (* it1).bt = bondtype::current_bondtype;
	//		else AddBond(newbond);
		}
		else
		{
			draw_data[0]->el = element::current_element;
			DiscardCurrentEng();	// here we must call this explicitly...
		}
		
		UpdateAllGraphicsViews();
	}
}

// this comes directly from mm1_mdl, the bonds are just disabled...
// this comes directly from mm1_mdl, the bonds are just disabled...
// this comes directly from mm1_mdl, the bonds are just disabled...

void qm1_docv::EraseEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (qm1_atom *) names[1];
		}
		else
		{
			draw_data[0] = NULL;
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (qm1_atom *) names[1];
		}
		else
		{
			draw_data[1] = NULL;
		}
		
		if (!draw_data[0] || !draw_data[1]) return;
		
		// if different: try to find and remove a bond.
		// if not different: remove atom.
		
		if (draw_data[0] != draw_data[1])
		{
//			mm1_bond tmpbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
//			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), tmpbond);
//			if (it1 != bond_list.end()) RemoveBond(it1); else return;
		}
		else
		{
			iter_qm1al it1 = find(atom_list.begin(), atom_list.end(), (* draw_data[0]));
			if (it1 != atom_list.end()) RemoveAtom(it1); else exit(EXIT_FAILURE);
		}
		
		UpdateAllGraphicsViews();
	}
}

void qm1_docv::SelectEvent(graphics_view *, vector<iGLu> & names)
{
	if (names[0] == GLNAME_MD_TYPE1)
	{
		qm1_atom * ref = (qm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
	}
}

void qm1_docv::MeasureEvent(graphics_view *, vector<iGLu> & names)
{
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	
        static qm1_atom *a1 = NULL;
	static qm1_atom *a2 = NULL;
	static qm1_atom *a3 = NULL;
// this function is quite similar in mm1_docv and qm1_docv. could this be shorter as a loop???

	if (names[0] == GLNAME_MD_TYPE1)
	{
		qm1_atom * ref = (qm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
		
		if (a1 == NULL)
		  {
		    a1 = ref;
	//	    cout << "charge: " << ref->charge << endl;
		  }
		else if (a1 != NULL && a2 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = NULL; return; }
		    
		    a2 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    
		    float len = measure_len(p1, p2);
		    str1 << "distance: " << len << " nm" << endl << ends;
		    PrintToLog(mbuff1);
		  }
		else if (a1 != NULL && a2 != NULL && a3 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = NULL; return; }
		    
		    a3 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    
		    float ang = measure_ang(p1, p2, p3);
		    str1 << "angle: " << ang << " deg" << endl << ends;
		    PrintToLog(mbuff1);
		  }
		else
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = a3; a3 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = a3; a3 = NULL; return; }
		    else if (a3 == ref) { a3->selected = false; a3 = NULL; return; }
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    float * p4 = ref->crd_vector[0].data;
		    
		    float tor = measure_tor(p1, p2, p3, p4);
		    str1 << "torsion: " << tor << " deg " << endl << ends;
		    PrintToLog(mbuff1);
		    
		    a1->selected = false; a1 = NULL;
		    a2->selected = false; a2 = NULL;
		    a3->selected = false; a3 = NULL;
		    ref->selected = false;
		    
		    UpdateAllGraphicsViews();
		  }
	}
}

enlevdiag_view * qm1_docv::AddEnLevDiagView(void)
{
	enlevdiag_view * eldv = graphics_factory->ProduceEnLevDiagView(this);
	plotting_view_vector.push_back(eldv);
	
	return eldv;
}

void qm1_docv::ProcessCommandString(graphics_view * gv, const char * command)
{
	char mbuff1[512];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Processing Command : " << command << endl << ends;
	PrintToLog(mbuff1);
	
	bool retval = ProcessCommonCommands(gv, command);
	if (retval == true) return;
	
	istrstream istr(command);
	char kw1[32]; istr >> kw1;
	
	// is it "help"??? if is, then append...
	
	if (!strcmp("help", kw1))
	{
		PrintToLog("> add plane <vf> <cf> <cscale> <dim> <res> <tp> <alpha> -- add a plane object.\n");
		PrintToLog(">   where: <vf> = value function : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add volrend <vf> <cf> <cscale> <dim> <res> <alpha> -- add a volume-rendering object.\n");
		PrintToLog(">   where: <vf> = value function : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add surf1 <vf1> <vf2> <cf> <sscale> <cscale> <dim> <res> <solid> <tp> <alpha> -- add a single surface object.\n");
		PrintToLog(">   where: <vf1> = value function for calculating the surface : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <vf2> = value function for calculating the colours : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <cf> = colour function : red green blue rb1 rb2\n");
		PrintToLog(">          <sscale> = scaling value for calculating the surface\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <solid> = 0 or 1 telling if the object is solid\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> add surf2 <vf1> <vf2> <cf1> <cf2> <sscale1> <sscale2> <cscale> <dim> <res> <solid> <tp> <alpha> -- add a pair of surface objects.\n");
		PrintToLog(">   where: <vf1> = value function for calculating the surface : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <vf2> = value function for calculating the colours : esp vdws eldens mo mod dist unity\n");
		PrintToLog(">          <cf1> = colour function for 1st surface : red green blue rb1 rb2\n");
		PrintToLog(">          <cf2> = colour function for 2nd surface : red green blue rb1 rb2\n");
		PrintToLog(">          <sscale1> = scaling value for calculating the surface for 1st surface\n");
		PrintToLog(">          <sscale2> = scaling value for calculating the surface for 2nd surface\n");
		PrintToLog(">          <cscale> = scaling value for calculating the colours\n");
		PrintToLog(">          <dim> = dimension of the plane object (in nm units)\n");
		PrintToLog(">          <res> = resolution of the plane object\n");
		PrintToLog(">          <solid> = 0 or 1 telling if the object is solid\n");
		PrintToLog(">          <tp> = 0 or 1 telling if the object is transparent\n");
		PrintToLog(">          <alpha> = transparency alpha value\n");
		
		PrintToLog("> set_current_orbital <orbital_index> -- set the current orbtal index for plotting the orbitals.\n");
		
		return;
	}
	
	// process other commands.
	// process other commands.
	// process other commands.
	
	if (!strcmp("add", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; type of the object to add.
		char object_name[128];
		
		if (!strcmp("plane", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf
			char kw4[32]; istr >> kw4;	// cf
			char kw5[32]; istr >> kw5;	// cscale
			char kw6[32]; istr >> kw6;	// dim
			char kw7[32]; istr >> kw7;	// res
			char kw8[32]; istr >> kw8;	// tp
			char kw9[32]; istr >> kw9;	// alpha
			char ** endptr = NULL;
			
			cp_param cpp;
			cpp.docv = this; cpp.ref = current_eng;
			
			if (!strcmp(kw3, "esp")) cpp.vf = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw3, "vdws")) cpp.vf = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw3, "eldens")) cpp.vf = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw3, "mo")) cpp.vf = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw3, "mod")) cpp.vf = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw3, "dist")) cpp.vf = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) cpp.vf = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add plane : unknown value function.\n"); return; }
			
			if (!strcmp(kw4, "red")) cpp.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw4, "green")) cpp.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw4, "blue")) cpp.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw4, "rb1")) cpp.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw4, "rb2")) cpp.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add plane : unknown colour function.\n"); return; }
			
			f64 cscale = strtod(kw5, endptr);
			
			f64 dim = strtod(kw6, endptr);
			
			i32s res = strtol(kw7, endptr, 10);
			if (res < 2) res = 2;
			
			i32s tp = strtol(kw8, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kw9, endptr);
			
			cpp.value = cscale; cpp.dim = dim; cpp.np = res;
			cpp.transparent = tp; cpp.alpha = alpha;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << ends;
			
			AddObject(new color_plane_object(ol_static(), cpp, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : plane (" << kw3 << " " << kw4 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("volrend", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf
			char kw4[32]; istr >> kw4;	// cf
			char kw5[32]; istr >> kw5;	// cscale
			char kw6[32]; istr >> kw6;	// dim
			char kw7[32]; istr >> kw7;	// res
			char kw8[32]; istr >> kw8;	// alpha
			char ** endptr = NULL;
			
			cp_param cpp;
			cpp.docv = this; cpp.ref = current_eng;
			
			if (!strcmp(kw3, "esp")) cpp.vf = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw3, "vdws")) cpp.vf = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw3, "eldens")) cpp.vf = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw3, "mo")) cpp.vf = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw3, "mod")) cpp.vf = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw3, "dist")) cpp.vf = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) cpp.vf = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add volrend : unknown value function.\n"); return; }
			
			if (!strcmp(kw4, "red")) cpp.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw4, "green")) cpp.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw4, "blue")) cpp.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw4, "rb1")) cpp.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw4, "rb2")) cpp.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add volrend : unknown colour function.\n"); return; }
			
			f64 cscale = strtod(kw5, endptr);
			
			f64 dim = strtod(kw6, endptr);
			
			i32s res = strtol(kw7, endptr, 10);
			if (res < 4) res = 4;
			
			f64 alpha = strtod(kw8, endptr);
			
			cpp.value = cscale; cpp.dim = dim; cpp.np = res;
			cpp.transparent = true; cpp.alpha = alpha;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << ends;
			
			AddObject(new volume_rendering_object(ol_static(), cpp, res / 2, dim / 2.0, (* gv->cam), object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : volrend (" << kw3 << " " << kw4 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("surf1", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf1
			char kw4[32]; istr >> kw4;	// vf2
			char kw5[32]; istr >> kw5;	// cf
			char kw6[32]; istr >> kw6;	// sscale
			char kw7[32]; istr >> kw7;	// cscale
			char kw8[32]; istr >> kw8;	// dim
			char kw9[32]; istr >> kw9;	// res
			char kwA[32]; istr >> kwA;	// solid
			char kwB[32]; istr >> kwB;	// tp
			char kwC[32]; istr >> kwC;	// alpha
			char ** endptr = NULL;
			
			cs_param csp1;
			csp1.docv = this; csp1.ref = current_eng; csp1.next = NULL;
			
			if (!strcmp(kw3, "esp")) csp1.vf1 = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw3, "vdws")) csp1.vf1 = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw3, "eldens")) csp1.vf1 = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw3, "mo")) csp1.vf1 = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw3, "mod")) csp1.vf1 = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw3, "dist")) csp1.vf1 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) csp1.vf1 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf1 : unknown value function 1.\n"); return; }
			
			if (!strcmp(kw4, "esp")) csp1.vf2 = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw4, "vdws")) csp1.vf2 = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw4, "eldens")) csp1.vf2 = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw4, "mo")) csp1.vf2 = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw4, "mod")) csp1.vf2 = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw4, "dist")) csp1.vf2 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw4, "unity")) csp1.vf2 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf1 : unknown value function 2.\n"); return; }
			
			if (!strcmp(kw5, "red")) csp1.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw5, "green")) csp1.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw5, "blue")) csp1.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw5, "rb1")) csp1.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw5, "rb2")) csp1.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf1 : unknown colour function.\n"); return; }
			
			f64 sscale = strtod(kw6, endptr);
			f64 cscale = strtod(kw7, endptr);
			
			f64 dim = strtod(kw8, endptr);
			
			i32s res = strtol(kw9, endptr, 10);
			if (res < 4) res = 4;
			
			i32s solid = strtol(kwA, endptr, 10);
			if (solid < 0) solid = 0; if (solid > 1) solid = 1;

			i32s tp = strtol(kwB, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kwC, endptr);
			
			static fGL dim_arr[3];
			dim_arr[0] = dim_arr[1] = dim_arr[2] = dim;
			
			static i32s res_arr[3];
			res_arr[0] = res_arr[1] = res_arr[2] = res;
			
			csp1.svalue = sscale; csp1.cvalue = cscale;
			csp1.dim = dim_arr; csp1.np = res_arr; csp1.wireframe = !solid;
			csp1.transparent = tp; csp1.alpha = alpha; csp1.toler = fabs(1.0e-6 * sscale); csp1.maxc = 250;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << kw4 << "-" << ends;
			
			AddObject(new color_surface_object(ol_static(), csp1, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : surf1 (" << kw3 << " " << kw4 << " " << kw5 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		if (!strcmp("surf2", kw2))
		{
			char kw3[32]; istr >> kw3;	// vf1
			char kw4[32]; istr >> kw4;	// vf2
			char kw5[32]; istr >> kw5;	// cf1
			char kw6[32]; istr >> kw6;	// cf2
			char kw7[32]; istr >> kw7;	// sscale1
			char kw8[32]; istr >> kw8;	// sscale2
			char kw9[32]; istr >> kw9;	// cscale
			char kwA[32]; istr >> kwA;	// dim
			char kwB[32]; istr >> kwB;	// res
			char kwC[32]; istr >> kwC;	// solid
			char kwD[32]; istr >> kwD;	// tp
			char kwE[32]; istr >> kwE;	// alpha
			char ** endptr = NULL;
			
			cs_param csp2a; cs_param csp2b;
			
			csp2a.docv = this; csp2a.ref = current_eng; csp2a.next = & csp2b;

			csp2b.docv = this; csp2b.ref = current_eng; csp2b.next = NULL;
			
			if (!strcmp(kw3, "esp")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw3, "vdws")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw3, "eldens")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw3, "mo")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw3, "mod")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw3, "dist")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw3, "unity")) csp2a.vf1 = csp2b.vf1 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf2 : unknown value function 1.\n"); return; }
			
			if (!strcmp(kw4, "esp")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) qm1_eng::getESP;
			else if (!strcmp(kw4, "vdws")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) qm1_eng::getVDWSurface;
			else if (!strcmp(kw4, "eldens")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) qm1_eng::getElDens;
			else if (!strcmp(kw4, "mo")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) qm1_eng::getOrbital;
			else if (!strcmp(kw4, "mod")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) qm1_eng::getOrbDens;
			else if (!strcmp(kw4, "dist")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) GetDistance;
			else if (!strcmp(kw4, "unity")) csp2a.vf2 = csp2b.vf2 = (ValueFunction *) GetUnity;
			else { PrintToLog("ERROR : add surf2 : unknown value function 2.\n"); return; }
			
			if (!strcmp(kw5, "red")) csp2a.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw5, "green")) csp2a.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw5, "blue")) csp2a.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw5, "rb1")) csp2a.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw5, "rb2")) csp2a.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf2 : unknown colour function 1.\n"); return; }
			
			if (!strcmp(kw6, "red")) csp2b.cf = (ColorFunction *) GetRedColor;
			else if (!strcmp(kw6, "green")) csp2b.cf = (ColorFunction *) GetGreenColor;
			else if (!strcmp(kw6, "blue")) csp2b.cf = (ColorFunction *) GetBlueColor;
			else if (!strcmp(kw6, "rb1")) csp2b.cf = (ColorFunction *) GetRBRange1;
			else if (!strcmp(kw6, "rb2")) csp2b.cf = (ColorFunction *) GetRBRange2;
			else { PrintToLog("ERROR : add surf2 : unknown colour function 2.\n"); return; }
			
			f64 sscale1 = strtod(kw7, endptr);
			f64 sscale2 = strtod(kw8, endptr);
			f64 cscale = strtod(kw9, endptr);
			
			f64 dim = strtod(kwA, endptr);
			
			i32s res = strtol(kwB, endptr, 10);
			if (res < 4) res = 4;
			
			i32s solid = strtol(kwC, endptr, 10);
			if (solid < 0) solid = 0; if (solid > 1) solid = 1;

			i32s tp = strtol(kwD, endptr, 10);
			if (tp < 0) tp = 0; if (tp > 1) tp = 1;
			
			f64 alpha = strtod(kwE, endptr);
			
			static fGL dim_arr[3];
			dim_arr[0] = dim_arr[1] = dim_arr[2] = dim;
			
			static i32s res_arr[3];
			res_arr[0] = res_arr[1] = res_arr[2] = res;
			
			csp2a.svalue = sscale1; csp2a.cvalue = cscale;
			csp2a.dim = dim_arr; csp2a.np = res_arr; csp2a.wireframe = !solid;
			csp2a.transparent = tp; csp2a.alpha = alpha; csp2a.toler = fabs(1.0e-6 * sscale1); csp2a.maxc = 250;
			
			csp2b.svalue = sscale2; csp2b.cvalue = cscale;
			csp2b.dim = dim_arr; csp2b.np = res_arr; csp2b.wireframe = !solid;
			csp2b.transparent = tp; csp2b.alpha = alpha; csp2b.toler = fabs(1.0e-6 * sscale2); csp2b.maxc = 250;
			
			ostrstream strN(object_name, sizeof(object_name));
			strN << kw3 << "-" << kw4 << "-" << ends;
			
			AddObject(new color_surface_object(ol_static(), csp2a, object_name));
			UpdateAllGraphicsViews();
			
			ostrstream strR(mbuff1, sizeof(mbuff1));
			strR << "Added a new object : surf2 (" << kw3 << " " << kw4 << " " << kw5 << " " << kw6 << ")." << endl << ends;
			PrintToLog(mbuff1);
			return;
		}
		
		PrintToLog("ERROR : could not process a command : \"add\".\n");
	}
	
	if (!strcmp("set_current_orbital", kw1))
	{
		char kw2[32]; istr >> kw2;	// the 2nd keyword; the orbital index.
		
		char ** endptr = NULL;
		int index = strtol(kw2, endptr, 10);
		if (index < 0) index = 0;
		
		current_orbital = index;
		
		ostrstream strR(mbuff1, sizeof(mbuff1));
		strR << "The current orbital is now " << current_orbital << "." << endl << ends;
		PrintToLog(mbuff1);
		return;
	}
	
	// error message...
	ostrstream strE(mbuff1, sizeof(mbuff1)); strE << "ERROR : Unknown command : " << command << endl << ends;
	PrintToLog(mbuff1); PrintToLog("The \"help\" command will give more information about command strings.\n");
}

/*################################################################################################*/

void qm1_cm_element::GetColor(const void * p1, fGL_a4 & p2, prefs *p3)
{
	qm1_atom * ref = (qm1_atom *) p1;
	const fGL * color = ref->el.GetColor(p3);
	for (i32s n1 = 0;n1 < 3;n1++) p2[n1] = color[n1];
}

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS

// eof
