/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include "gnome-cmd-includes.h"
#include "dir_pool.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-clist.h"
#include "gnome-cmd-data.h"
#include "utils.h"

static GList *queue = NULL;
static GHashTable *map = NULL;


static void
check_cache_maxsize ()
{
	// remove the last dir if maxsize is exceeded
	while (g_list_length (queue) > gnome_cmd_data_get_dir_cache_size ()) {
		GnomeCmdDir *dir = (GnomeCmdDir*)g_list_last (queue)->data;
		g_hash_table_remove (map, gnome_cmd_dir_get_uri_str (dir));
		queue = g_list_remove (queue, dir);
		DEBUG ('p', "removing %s 0x%x from the pool\n", dir->path, dir);
		gnome_cmd_dir_unref (dir);

		DEBUG('p', "queue size: %d  map size: %d\n",
			  g_list_length (queue),
			  g_hash_table_size (map));
	}
}


void
dir_pool_init (void)
{
	map = g_hash_table_new (g_str_hash, g_str_equal);
}


GnomeCmdDir *
dir_pool_get (const gchar *dir_uri_str)
{
	GnomeCmdDir *dir = g_hash_table_lookup (map, dir_uri_str);
	if (dir) {
		// move it first in the queue
		queue = g_list_remove (queue, dir);
		queue = g_list_prepend (queue, dir);
		return dir;
	}
	
	return gnome_cmd_dir_new_from_uri_str (dir_uri_str, TRUE);
}


void
dir_pool_add (GnomeCmdDir *dir)
{
	g_return_if_fail (dir != NULL);
	
	DEBUG ('p', "adding %s 0x%x to the pool\n", dir->path, dir);
	gnome_cmd_dir_ref (dir);
	queue = g_list_prepend (queue, dir);
	g_hash_table_insert (map, (gpointer)gnome_cmd_dir_get_uri_str(dir), dir);

	check_cache_maxsize ();
}


void
dir_pool_remove (GnomeCmdDir *dir)
{
	g_return_if_fail (dir != NULL);
}


gboolean
dir_pool_exists (GnomeCmdDir *dir)
{
	return (g_list_index (queue, dir) != -1);
}


static void
add_dir_to_list (GnomeCmdDir *dir, GtkCList *list)
{
	gchar *text[3];

	g_return_if_fail (dir != NULL);
	g_return_if_fail (list != NULL);

	text[0] = (gchar*)gnome_cmd_dir_get_uri_str (dir);
	text[1] = g_strdup_printf ("%d", dir->ref_cnt);
	text[2] = NULL;

	gtk_clist_append (list, text);

	g_free (text[1]);
}

void
dir_pool_show_state (void)
{
	gchar *titles[] = {"uri", "ref_cnt", NULL};
	GtkWidget *win;
	GtkWidget *list;

	win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_usize (win, 400, 300);
	gtk_window_set_title (GTK_WINDOW (win), "The current directory pool");

	list = gnome_cmd_clist_new_with_titles (2, titles);
	gtk_clist_set_column_width (GTK_CLIST (list), 0, 300);
	gtk_container_add (GTK_CONTAINER (win), list);

	g_list_foreach (queue, (GFunc)add_dir_to_list, list);
	
	gtk_widget_show (list);	
	gtk_widget_show (win);
}
