/*
 *  PGSchema.m
 *  graphviz
 *
 *  Created by Glen Low on Tue Dec 30 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/macstl/
 *  macstl@pixelglow.com
 *
 *  You may redistribute or use this software in source and binary forms, with or without modification,
 *  provided you meet the following conditions:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * All modified files must carry prominent notices stating the nature and date of the modification.
 *
 *  The copyright holders make no representations about the suitability of this software for any purpose.
 *  It is provided "as is" without express or implied warranty.
 */

#include <QuickTime/QuickTime.h>

#import "PGSchema.h"
#import "PGDescriptor.h"
#import "PGAttribute.h"
#import "PGOption.h"

@implementation PGSchema

+ (PGSchema*) mainSchema
	{
		static PGSchema* schema = nil;
		if (!schema)
			schema = [[PGSchema alloc] init];
		return schema;
	}

- (id) init
	{
		if ((self = [super init]))
			{
				rawChoices_ = [[NSDictionary alloc] initWithContentsOfFile: [[NSBundle mainBundle] pathForResource: @"Choices" ofType: @"plist"]];
				choices_ = [[NSMutableDictionary alloc] init];
			
				NSNumberFormatter* intFormatter = [[NSNumberFormatter alloc] init];
				[intFormatter setFormat: @"#,##0;0;-#,##0"];
				NSNumberFormatter* doubleFormatter = [[NSNumberFormatter alloc] init];
				[doubleFormatter setFormat: @"#,##0.00;0.00;-#,##0.00"];
				
				formatters_ = [[NSDictionary alloc] initWithObjectsAndKeys:
					doubleFormatter, @"double",
					intFormatter, @"int",
					nil];
				
				[intFormatter release];
				[doubleFormatter release];
				
				attributes_ = nil;
				
			}
		return self;
	}
	
- (NSArray*) choicesForType: (NSString*) type
	{
		NSMutableArray* choices = [choices_ objectForKey: type];
		
		// not previously saved, so need to generate it from raw choices and runtime info
		if (!choices)
			{
				choices = [[NSMutableArray alloc] init];
						
				// add descriptors from raw choices i.e. Choices.plist
				NSArray* rawChoices = [rawChoices_ objectForKey: type];
				if (rawChoices)
					{
						NSEnumerator* eachRawChoice = [rawChoices objectEnumerator];
						NSString* nextRawChoice;
						while ((nextRawChoice = [eachRawChoice nextObject]))
							{
								PGDescriptor* nextChoice = [[PGDescriptor alloc] initWithScope: type name: nextRawChoice];
								[choices addObject: nextChoice];
								[nextChoice release];
							}
					}
					
				// add any choices that can only be determined at runtime e.g. addFonts
				SEL customAdder = NSSelectorFromString ([NSString stringWithFormat: @"%@s:", type]);
				if ([self respondsToSelector: customAdder])
					[self performSelector: customAdder withObject: choices];
					
				// sort by description
				if ([choices count])
					{
						[choices sortUsingSelector: @selector (compare:)];
					
						[choices_ setObject: choices forKey: type];
						[choices release];
					}
				else
					{
						[choices release];
						choices = nil;
					}
			}
		return choices;
	}

- (NSFormatter*) formatterForType: (NSString*) type
	{
		return [formatters_ objectForKey: type];
	}

- (NSArray*) optionsForComponent: (NSString*) component
	{
		if (!attributes_)
			{
				NSArray* rawAttributes = [[NSArray alloc] initWithContentsOfFile: [[NSBundle mainBundle] pathForResource: @"Attributes" ofType: @"plist"]];
				attributes_ = [[NSMutableArray alloc] initWithCapacity: [rawAttributes count]];
				
				NSEnumerator* eachRawAttribute = [rawAttributes objectEnumerator];
				NSDictionary* nextRawAttribute;
				while ((nextRawAttribute = [eachRawAttribute nextObject]))
					[attributes_ addObject: [PGAttribute attributeWithDictionary: nextRawAttribute]];
			}
			
		NSMutableArray* options = [[NSMutableArray alloc] init];
				
		NSEnumerator* eachAttribute = [attributes_ objectEnumerator];
		PGAttribute* nextAttribute;
		while ((nextAttribute = [eachAttribute nextObject]))
			if ([[nextAttribute usedBy] containsObject: component])
				{
					PGOption* nextOption = [[PGOption alloc] initWithPrefix:
						[NSString stringWithFormat: @"-%@", [[component substringWithRange: NSMakeRange (0, 1)] uppercaseString]]
						attribute: nextAttribute];
					[options addObject: nextOption];
					[nextOption release];
				}
						
		[options sortUsingSelector: @selector (compare:)];
		return options;
	}

- (void) fonts: (NSMutableArray*) choices
	{
		// load Postscript font names
		[choices addObjectsFromArray: [[NSFontManager sharedFontManager] availableFonts]];
	}
	
- (void) formats: (NSMutableArray*) choices
	{
		// load bitmap formats from Quicktime
		Component nextComponent = 0;

		ComponentDescription criteria;
		criteria.componentType = GraphicsExporterComponentType;
		criteria.componentSubType = 0;
		criteria.componentManufacturer = 0;
		criteria.componentFlagsMask = graphicsExporterIsBaseExporter;

		Handle componentName = NewHandle (0);
		while ((nextComponent = FindNextComponent (nextComponent, &criteria)))
			{
				ComponentDescription componentDescription;
				GetComponentInfo (nextComponent, &componentDescription, componentName, 0, 0);
				HLock (componentName);
				CFStringRef nameString = CFStringCreateWithPascalString (kCFAllocatorDefault, *componentName, kCFStringEncodingUTF8);
				HUnlock (componentName);

				OSType extension;
				GraphicsExportGetDefaultFileNameExtension ((GraphicsExportComponent) nextComponent, &extension);
				NSString* extensionString =
					[[[NSString stringWithCString: (char*) &extension length: sizeof (extension)]
						stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]]
						lowercaseString];
				
				PGDescriptor* nextFormat = [[PGDescriptor alloc] initWithName: extensionString tip: nil description: (NSString*) nameString];
				
				[choices addObject: nextFormat];
				[nextFormat release];
					
				CFRelease (nameString);
			}
		DisposeHandle (componentName);	
	}
	
- (void) dealloc
	{
		[rawChoices_ release];
		[choices_ release];
		
		[formatters_ release];
		
		[attributes_ release];
	}
	
/*
+ (NSFormatter*) colorFormat
	{
		static ChoiceFormatter* formatter = nil;
		
		if (!formatter)
			{
				formatter = [[ChoiceFormatter alloc] init];
				[formatter setChoices: [NSArray arrayWithContentsOfFile: [[NSBundle mainBundle] pathForResource: @"Colors" ofType: @"plist"]]];
			}
		return formatter;
	}
*/
	
@end
