/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// def-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "def-proc.h"

#include "defs.h"
#include "htp-files.h"
#include "macro.h"
#include "option.h"

/*
// Block macro destructor callback ... used whenever a block macro is
// destroyed with a RemoveVariable() or ClearVariableList()
*/
void BlockDestructor(const char *name, void *value, uint type, uint flags,
    void *param)
{
    STREAM *stream = (STREAM*) value;
    UNREF_PARAM(name);
    UNREF_PARAM(type);
    UNREF_PARAM(flags);

    CloseStream(stream);
    FreeMemory(stream);

    /* DEF macros use param */
    if(param != NULL)
    {
        FreeMemory(param);
    }
}

BOOL ReadBlockToFile(TASK *task, HTML_MARKUP *htmlMarkup, 
                     STREAM *blockFile) {
    char closeTag[MAX_VARNAME_LEN + 1];
    char *openTag = closeTag + 1;
    char *plaintext;
    char ch;
    HTML_MARKUP newHtml;
    BOOL result;
    static uint blockDepth = 0;
    uint markupType;
    STREAM *oldOutFile;
    BOOL expand = IsAttributeInMarkup(htmlMarkup, "EXPAND") == TRUE;
    uint markupResult;

    closeTag[0] = '/';
    StringCopy(openTag, htmlMarkup->tag, MAX_VARNAME_LEN);

    HtpMsg(MSG_INFO, task->infile, "reading %s block to memory",
           expand ? "expanded" : "raw");

    oldOutFile = task->outfile;
    task->outfile = blockFile;

    /* Skip EOL, but put it back if it's another character */
    if (GetStreamChar(task->infile, &ch) && ch != '\n')
    {
        UnreadStreamChar(task->infile, ch);
    }

    task->outfile->name = DuplicateString(task->infile->name);
    task->outfile->lineNumber = task->infile->lineNumber;

    /* start copying the file into the temporary file, looking for the */
    /* BLOCK or /BLOCK tag if block macro, DEF or /DEF otherwise ... */
    /* just squirrel away all other tags and text */
    for(;;)
    {
        result = ReadHtmlFile(task->infile, blockFile, &plaintext, &markupType);
        if(result == ERROR)
            return FALSE;
        else if(result == FALSE)
        {
            /* end-of-file encountered before end-of-block */
            HtpMsg(MSG_ERROR, task->infile,
                   "EOF encountered before %s in line %d was closed",
                   openTag, task->outfile->lineNumber);
            return FALSE;
        }

        /* turn the plain text into HTML structure, but don't destroy */
        /* the plaintext, this will be used to copy into output file if */
        /* necessary */
        if(PlaintextToMarkup(plaintext, &newHtml) == FALSE)
        {
            /* memory alloc failed, most likely */
            HtpMsg(MSG_ERROR, task->infile, "could not process markup (out of memory?)");
            FreeMemory(plaintext);
            return FALSE;
        }

        if((markupType & MARKUP_TYPE_HTP) && blockDepth == 0
           && IsMarkupTag(&newHtml, closeTag) == TRUE)
        {
            /* found the end of the macro block */
            DestroyMarkupStruct(&newHtml);
            FreeMemory(plaintext);
            break;
        }

        if (expand) {
            /* destroy the ORIGINAL plain text, not needed again */
            FreeMemory(plaintext);
            plaintext = NULL;

            /* give the default processor a chance to expand macros, etc. */
            markupResult = ExpandAll(task, &newHtml, 
                                     &plaintext, markupType);

        } else {

            if(markupType & MARKUP_TYPE_HTP)
            {
                /* check for embedded block declarations */
                if(IsMarkupTag(&newHtml, openTag) == TRUE)
                {
                    /* add to the block macro depth and continue */
                    blockDepth++;
                }
                else if(IsMarkupTag(&newHtml, closeTag) == TRUE)
                {
                    /* depth has decreased one */
                    blockDepth--;
                }
            }

            /* if continuing, then the plaintext is put into the
             * output stream as-is ... there is no case where the
             * processor continues scanning but discards a markup
             */
            markupResult = MARKUP_OKAY;
        }
        
        switch(markupResult)
        {
            case MARKUP_OKAY:
            {
                /* add the markup to the output file as it should appear */
                
                StreamPrintF(blockFile, "%c%s%c", 
			      MARKUP_OPEN_DELIM(markupType),
                              plaintext, MARKUP_CLOSE_DELIM(markupType));
            }
            break;

            case NEW_MARKUP:
            case MARKUP_REPLACED:
            {
                /* the markup has been replaced by a normal string */
                StreamPrintF(blockFile, "%s", plaintext);
            }
            break;

            case DISCARD_MARKUP:
            {
                /* markup will not be included in final output */
            }
            break;

            case MARKUP_ERROR:
            {
                /* (need to destroy plaintext buffer before exiting) */
                FreeMemory(plaintext);
                return FALSE;
            }

            default:
            {
                FreeMemory(plaintext);
                printf("%s: serious internal error\n", PROGRAM_NAME);
                exit(1);
            }
        }

        /* destroy the HTML markup, not needed any longer */
        DestroyMarkupStruct(&newHtml);

        /* destroy the plaintext buffer */
        if (plaintext)
            FreeMemory(plaintext);
        plaintext = NULL;
    }

    task->outfile = oldOutFile;
    return TRUE;
}

BOOL ReadinBlock(TASK *task, HTML_MARKUP *htmlMarkup, STREAM *blockStream) {
    BOOL result;

    /* try and create the temporary file */
    if(CreateBufferWriter(blockStream, htmlMarkup->tag) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile,
	       "unable to create stream for %s macro", htmlMarkup->tag);
        return FALSE;
    }

    result = ReadBlockToFile(task, htmlMarkup, blockStream);
    FlushBufferWriter(blockStream);
    return result;
}

uint BlockProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *defName, *param;
    char *defNameCopy, *paramCopy;
    STREAM *stream;
    BOOL blockMacro;
    uint macroType;
    VARSTORE *varstore;

    UNREF_PARAM(newPlaintext);

    /* first: is this a BLOCK macro or a DEF macro?  This function is */
    /* overloaded to handle both types, and must internally change its */
    /* functionality for each type */
    /* if this is false, this is a DEF macro */
    if(stricmp(htmlMarkup->tag, "BLOCK") == 0)
    {
        blockMacro = TRUE;
        macroType = VAR_TYPE_BLOCK_MACRO;
        /* no extra varstore parameter for a block macro */
        param = NULL;
    }
    else
    {
        blockMacro = FALSE;
        if (stricmp(htmlMarkup->tag, "BLOCKDEF") == 0)
        {
            macroType = VAR_TYPE_BLOCKDEF_MACRO;
        }
        else 
        {
            assert(stricmp(htmlMarkup->tag, "DEF") == 0);
            macroType = VAR_TYPE_DEF_MACRO;
        }

        /* get the OPTION parameter and duplicate it */
        param = MarkupAttributeValue(htmlMarkup, "OPTION");
    }
    
    /* check if name is present */
    if (IsAttributeInMarkup(htmlMarkup, "NAME")) {
        /* new syntax */
        
        defName = MarkupAttributeValue(htmlMarkup, "NAME");
        if(defName == NULL)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "%s requires a NAME attribute to have a value",
                   htmlMarkup->tag);
            return MARKUP_ERROR;
        }
    } else {

        /* old syntax only allowed for BLOCK tag */
        if(!blockMacro)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "%s requires NAME attribute", htmlMarkup->tag);
            return MARKUP_ERROR;
        }

        /* is a name specified? (only one name can be specified) */
        if(htmlMarkup->attribCount != 1)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "bad %s markup specified", htmlMarkup->tag);
            return MARKUP_ERROR;
        }
        defName = htmlMarkup->attrib[0].name;
    }

    if ((defNameCopy = DuplicateString(defName)) == NULL) {
        HtpMsg(MSG_ERROR, task->infile, 
               "unable to create NAME copy for %s macro", 
               htmlMarkup->tag);
        return MARKUP_ERROR;
    }

    if (param == NULL) {
        paramCopy = NULL;
    } else if ((paramCopy = DuplicateString(param)) == NULL) {
        HtpMsg(MSG_ERROR, task->infile, 
               "unable to create OPTION copy for %s macro", 
               htmlMarkup->tag);
        FreeMemory(defNameCopy);
        return MARKUP_ERROR;
    }

    stream = (STREAM*) AllocMemory(sizeof(STREAM));
    if (!ReadinBlock(task, htmlMarkup, stream))
    {
        FreeMemory(defNameCopy);
        if (paramCopy != NULL)
            FreeMemory(paramCopy);
        return MARKUP_ERROR;
    }

    /* store the block file name and the block macro name as a variable */
    varstore = task->varstore;
    if (IsAttributeInMarkup(htmlMarkup, "GLOBAL")) {
        while (!varstore->isGlobal)
            varstore = varstore->parent;
    }

    if(StoreVariable(varstore, defNameCopy, stream, macroType, 
                     VAR_FLAG_DEALLOC_NAME, paramCopy, BlockDestructor) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile, 
               "unable to store macro information (out of memory?)");
        FreeMemory(defNameCopy);
        if (paramCopy != NULL)
            FreeMemory(paramCopy);
        return MARKUP_ERROR;
    }

    return DISCARD_MARKUP;
}   


uint UndefProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    uint ctr;
    const char *name;

    UNREF_PARAM(newPlaintext);

    /* need at least one attribute to undef */
    if(htmlMarkup->attribCount == 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "UNDEF requires at least one name");
        return MARKUP_ERROR;
    }

    /* walk the list of attributes, deleting them as found */
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        name = htmlMarkup->attrib[ctr].name;

        /* is it in the store? */
        if(VariableExists(task->varstore, name) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "No metatag \"%s\" to undefine",
                name);
            return MARKUP_ERROR;
        }

        /* only remove it if a DEF macro */
        if(GetVariableType(task->varstore, name) == VAR_TYPE_DEF_MACRO)
        {
            RemoveVariable(task->varstore, name);
        }

        HtpMsg(MSG_INFO, task->infile, "metatag \"%s\" removed", name);
    }

    return DISCARD_MARKUP;
}
