/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// set-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "set-proc.h"

#include "defs.h"

uint SetProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    char *name;
    char *value;
    uint ctr;
    uint flag;
    HTML_ATTRIBUTE *attrib;
    VARSTORE *varstore;

    UNREF_PARAM(newPlaintext);

    varstore = task->varstore;
    if (IsAttributeInMarkup(htmlMarkup, "GLOBAL")) {
        while (varstore->parent != NULL)
            varstore = varstore->parent;
    }

    /* have to declare at least one macro, but more are acceptable */
    if(htmlMarkup->attribCount == 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "incomplete macro declaration");
        return MARKUP_ERROR;
    }

    attrib = &htmlMarkup->attrib[0];

    /* walk the list and add each macro to the variable store */
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        if(stricmp(attrib->name, "GLOBAL") == 0)
            continue;
        name = attrib->name;
        value = attrib->value;
        if (!value) {
            value = "";
        }
        flag = (attrib->quoted) ? VAR_FLAG_QUOTED : VAR_FLAG_NONE;

        /* put the new variable into the store */
        if(StoreVariable(varstore, 
                         DuplicateString(name), DuplicateString(value), 
                         VAR_TYPE_SET_MACRO, 
                         flag | VAR_FLAG_DEALLOC_NAME | VAR_FLAG_DEALLOC_VALUE,
                         NULL, NULL) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to store macro \"%s\" (out of memory?)",
                name);
            return MARKUP_ERROR;
        }

        HtpMsg(MSG_INFO, task->infile, "macro \"%s\" assigned value \"%s\"",
               name, value);
        attrib++;
    }

    return DISCARD_MARKUP;
}   



uint UnsetProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    uint ctr;
    uint type;
    const char *name;

    UNREF_PARAM(newPlaintext);

    /* have to specify at least one macro to remove, but more are acceptable */
    if(htmlMarkup->attribCount == 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "UNSET tag improperly specified");
    }

    /* walk the attributes list */
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        name = htmlMarkup->attrib[ctr].name;

        /* verify that the variable exists */
        if(VariableExists(task->varstore, name) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "bad macro name \"%s\"", name);
            return MARKUP_ERROR;
        }

        /* remove it from the variable store if its not a DEF macro */
        type = GetVariableType(task->varstore, name);

        if((type == VAR_TYPE_SET_MACRO) || (type == VAR_TYPE_BLOCK_MACRO))
        {
            RemoveVariable(task->varstore, name);
        }

        HtpMsg(MSG_INFO, task->infile, "variable \"%s\" removed", name);
    }

    return DISCARD_MARKUP;
}



uint IncProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *expansion;
    char valueStr[MAX_INC_VALUE_LENGTH];
    int value;
    uint ctr;
    HTML_ATTRIBUTE *attrib;

    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        attrib = &htmlMarkup->attrib[ctr];
        /* make sure variable exists in store */
        if(VariableExists(task->varstore, attrib->name) != TRUE)
        {
            /* only a warning */
            HtpMsg(MSG_WARNING, task->infile, "unrecognized macro name \"%s\"",
                attrib->name);
            continue;
        }

        /* block macros cannot be increased */
        if(GetVariableType(task->varstore, attrib->name) != VAR_TYPE_SET_MACRO)
        {
            HtpMsg(MSG_ERROR, task->infile, "only set macros can be increased, not \"%s\"",
                attrib->name);
            continue;
        }

        /* get the macros value and replace the attribute value */
        expansion = GetVariableValue(task->varstore, attrib->name);

        if(expansion == NULL)
        {
            HtpMsg(MSG_WARNING, task->infile, "no value for macro \"%s\"",
                attrib->name);
            continue;
        }
        value = atoi(expansion);
        if (attrib->value != NULL) {
            value += atoi(attrib->value);
        } else {
            value++;
        }
        sprintf(valueStr, "%i", value);
        if(UpdateVariableValue(task->varstore, attrib->name, 
                               DuplicateString(valueStr)) == FALSE) {
            HtpMsg(MSG_WARNING, task->infile, "new value %s for macro \"%s\" could not be stored",
                   valueStr, attrib->name);
        }
    }
    return DISCARD_MARKUP;
}   
