/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// $Id: streams.c,v 1.7 2002/05/11 19:34:50 hoenicke Exp $
//
// file/buffer stream functions
//
// Copyright (c) 2001 Jochen Hoenicke under Artistic License.
//
*/

#include "htp.h"
#include "snprintf.h"

BOOL CreateNullWriter (STREAM* stream) {
    stream->sflags = STREAM_FLAG_NULL_FILE;
    stream->name = "NULL";
    stream->lineNumber = 1;
    return TRUE;
}

BOOL CreateBufferWriter (STREAM* stream, const char *name) {
    stream->sflags = STREAM_FLAG_BUFFER;
    stream->u.buffer.length = 128;
    stream->u.buffer.buffer = AllocMemory(stream->u.buffer.length);
    stream->u.buffer.offset = 0;

    if (stream->u.buffer.buffer == NULL)
    {
        HtpMsg(MSG_ERROR, NULL, "unable to allocate stream buffer");
        return FALSE;
    }
    return TRUE;
}

BOOL CreateBufferReader (STREAM* stream, STREAM *writeStream)
{
    if (writeStream->sflags != STREAM_FLAG_BUFFER)
    {
        HtpMsg(MSG_ERROR, NULL, "CreateBufferReader: wrong buffer type");
        return FALSE;
    }
    stream->sflags = STREAM_FLAG_BUFFER | STREAM_FLAG_READER;
    stream->name = writeStream->name;
    stream->lineNumber = writeStream->lineNumber;
    stream->u.buffer.length = writeStream->u.buffer.offset;
    stream->u.buffer.buffer = writeStream->u.buffer.buffer;
    stream->u.buffer.offset = 0;
    stream->hasUnread = FALSE;
    return TRUE;
}

BOOL FlushBufferWriter (STREAM *stream)
{
    char *str;
    if (stream->sflags != STREAM_FLAG_BUFFER)
    {
        HtpMsg(MSG_ERROR, NULL, "CreateBufferReader: wrong buffer type");
        return FALSE;
    }
    str = AllocMemory(stream->u.buffer.offset + 1);
    memcpy(str, stream->u.buffer.buffer, stream->u.buffer.offset);
    str[stream->u.buffer.offset] = 0;
    FreeMemory(stream->u.buffer.buffer);
    stream->u.buffer.buffer = str;
    return TRUE;
}

BOOL CreateFileReader (STREAM *stream, const char *filename)
{
    stream->sflags = STREAM_FLAG_READER;
    stream->name = filename;
    stream->lineNumber = 1;
    stream->hasUnread = FALSE;
    return OpenFile(filename, "r", &stream->u.textfile);
}

BOOL CreateFDReader (STREAM *stream, const char *filename, int filedes)
{
    stream->sflags = STREAM_FLAG_READER;
    stream->name = filename;
    stream->lineNumber = 1;
    stream->hasUnread = FALSE;
    return OpenFD(filedes, "r", &stream->u.textfile);
}

BOOL CreateFileWriter (STREAM *stream, const char *filename, BOOL append)
{
    stream->sflags = 0;
    stream->name = filename;
    stream->lineNumber = 1;
    return OpenFile(filename, append ? "a" : "w", &stream->u.textfile);
}

void CloseStream (STREAM *stream)
{
    if ((stream->sflags & STREAM_FLAG_BUFFER)) {
        if (!(stream->sflags & STREAM_FLAG_READER)) {
            FreeMemory(stream->u.buffer.buffer);
            FreeMemory((void *)stream->name);
        }
    } else if ((stream->sflags & STREAM_FLAG_NULL_FILE)) {
        /* nothing */
    } else {
        CloseFile(&stream->u.textfile);
    }
}


BOOL GetStreamChar(STREAM *stream, char *c)
{
    assert(stream != NULL);
    assert((stream->sflags & STREAM_FLAG_READER));
    assert(stream->u.buffer.buffer != NULL);

    if (stream->hasUnread)
    {
        *c = stream->unread;
        stream->hasUnread = FALSE;
        return TRUE;
    }

    if ((stream->sflags & STREAM_FLAG_BUFFER))
    {
        /* Check for EOF */
        if (stream->u.buffer.length <= stream->u.buffer.offset)
            return FALSE;

        *c = stream->u.buffer.buffer[stream->u.buffer.offset++];
        if (*c == '\n')
            stream->lineNumber++;
        return TRUE;

    } else {
        BOOL result = GetFileChar(&stream->u.textfile, c);
        if (*c == '\n')
            stream->lineNumber++;
        return result;
    }
}

BOOL UnreadStreamChar(STREAM *stream, char c)
{
    assert(stream != NULL);
    assert((stream->sflags & STREAM_FLAG_READER));
    assert(stream->u.buffer.buffer != NULL);
    assert(!stream->hasUnread);

    stream->hasUnread = TRUE;
    stream->unread = c;
    return TRUE;
}

uint GetStreamBlock(STREAM *stream, char *buffer, uint size, bitmap delims)
{
    char *ptr, ch;
    int read = 0;

    assert(stream != NULL);
    assert((stream->sflags & STREAM_FLAG_READER));
    assert(buffer != NULL);

    if (stream->hasUnread)
    {
        *buffer++ = ch = stream->unread;
        size--;
        stream->hasUnread = FALSE;
        if (BITMAP_GET(delims, ch))
            return 1;
        read = 1;
    }

    if ((stream->sflags & STREAM_FLAG_BUFFER))
    {
        char * end = stream->u.buffer.buffer + stream->u.buffer.length;
        char * start = stream->u.buffer.buffer + stream->u.buffer.offset;

        /* Check for EOF */
        if (start >= end)
            return FALSE;


        /* leave space for trailing NUL. */
        size--;
        if (start + size < end)
            end = start + size;

        /* Search for new line. */
        ptr = start;
        while (ptr < end)
        {
            ch = *ptr++;
            if (ch == '\n')
                stream->lineNumber++;
            *buffer++ = ch;
            if (BITMAP_GET(delims, ch))
                break;
        }
        *buffer = 0;
        stream->u.buffer.offset += ptr - start;
        return read + ptr - start;

    } else {

        if(feof(stream->u.textfile.file))
            return FALSE;

        ptr = buffer;
        
        /* compare size to 1 rather than 0 to leave room for trailing NUL */
        while(size-- > 1)
        {
            if(GetFileChar(&stream->u.textfile, ptr) == FALSE)
            {
                break;
            }
            ch = *ptr;

            if(ch == '\n')
            {
                stream->lineNumber++;
            }

            ptr++;
            if (BITMAP_GET(delims, ch))
                break;
        }
        *ptr = 0;
        return read + ptr - buffer;
    }
}

BOOL PutStreamString(STREAM *stream, const char *str)
{
    /* check the arguments */
    assert(stream != NULL);
    assert(!(stream->sflags & STREAM_FLAG_READER));
    if ((stream->sflags & STREAM_FLAG_BUFFER))
    {
        ulong len = strlen(str);
        ulong free = stream->u.buffer.length - stream->u.buffer.offset;

        if (len > free) {
            char * newbuffer;
            while (len > free) {
                free += stream->u.buffer.length;
                stream->u.buffer.length *= 2;
            }
            newbuffer = ResizeMemory(stream->u.buffer.buffer, 
                                     stream->u.buffer.length);
            if (newbuffer == NULL) {
                /* unable to enlarge buffer area */
                HtpMsg(MSG_ERROR, NULL, "unable to grow stream buffers");
                return FALSE;
            }
            stream->u.buffer.buffer = newbuffer;
        }
        memcpy (stream->u.buffer.buffer + stream->u.buffer.offset, str, len);
        stream->u.buffer.offset += len;
        return TRUE;
    } else if ((stream->sflags & STREAM_FLAG_NULL_FILE)) {
        return TRUE;
    } else {
        const char *strptr = str;
        
        assert(str != NULL);
        while(*strptr != NUL)
        {
            if(PutFileChar(&stream->u.textfile, *strptr++) == FALSE)
            {
                return FALSE;
            }
        }
    }
    return TRUE;
}

BOOL StreamPrintF(STREAM *stream, const char *format, ...)
{
    va_list argptr;
    char str[16*KBYTE];
    /* allocate room to hold the final string ... allocating 16K to be safe, */
    /* although a better method would be more deterministic */
    /* (originally allocated 1K, but now allocating 16K due to JavaScript */
    /* and VBScript comments ... suggested by Allan Todd) */

    /* convert formatted arguments into single string */
    va_start(argptr, format);
    vsnprintf(str, 16 * KBYTE, format, argptr);
    va_end(argptr);

    return PutStreamString(stream, str);
}   

void SuppressLinefeeds(STREAM *stream)
{
    if (!(stream->sflags & (STREAM_FLAG_BUFFER | STREAM_FLAG_NULL_FILE)))
        stream->u.textfile.flags |= TEXTFILE_FLAG_NO_CR;
}

void SingleLinefeeds(STREAM *stream)
{
    if (!(stream->sflags & (STREAM_FLAG_BUFFER | STREAM_FLAG_NULL_FILE)))
        stream->u.textfile.flags |= TEXTFILE_FLAG_FEW_CR;
}

void AllowLinefeeds(STREAM *stream)
{
    if (!(stream->sflags & (STREAM_FLAG_BUFFER | STREAM_FLAG_NULL_FILE)))
        stream->u.textfile.flags
            &= (~(TEXTFILE_FLAG_NO_CR | TEXTFILE_FLAG_FEW_CR));
}
