/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Fader.cpp,v 1.36 2004/08/23 09:45:42 comix Exp $
 *
 */


#include "Fader.h"

#include <qcursor.h>

Fader::Fader(QWidget * parent, bool bWithoutKnob) : QWidget(parent, "Fader"), Object( "Fader" )
{
	setMinimumSize( 18, 100 );
	setMaximumSize( 18, 100);
	resize( 18, 100 );

	m_bWithoutKnob = bWithoutKnob;
	peakValue_L = 0.0f;
	peakValue_R = 0.0f;
	value = 0;
	changed = true;

	// Create temp image
	temp.resize(18, 100);

	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/mixerPanel/fader_background.png" );
	bool ok = back.load( background_path.c_str() );
	if( ok == false ) {
		qWarning("Fader: Error loading pixmap");
	}

	// Leds image
	string leds_path = string(IMG_PATH) + string( "/img/mixerPanel/fader_leds.png" );
	ok = leds.load( leds_path.c_str() );
	if( ok == false ){
		qWarning( "Fader: Error loading pixmap" );
	}

	// Knob image
	string knob_path = string(IMG_PATH) + string( "/img/mixerPanel/fader_knob.png" );
	ok = knob.load( knob_path.c_str() );
	if( ok == false ){
		qWarning( "Fader: Error loading pixmap" );
	}

	//
//	setErasePixmap( back );
}



Fader::~Fader() {
//	cout << "fader destroy" << endl;
}



void Fader::mouseMoveEvent( QMouseEvent *ev ) {
	int y = ev->y() - 6;
	const int max_y = 100 - 12;

	int value = max_y - y;

	if (value > max_y) {
		value = max_y;
	}
	else if (value < 0) {
		value = 0;
	}

	int perc = (int)(value * 100.0 / max_y);
	setValue(perc);
	emit valueChanged(this);
	update();
}



void Fader::mousePressEvent(QMouseEvent *ev) {
	int y = ev->y() - 6;
	const int max_y = 100 - 12;
	const int min_y = 0;
	const int delta = max_y - min_y;

	int value = delta - y;

	if (value > max_y) {
		value = max_y;
	}
	else if (value < min_y) {
		value = min_y;
	}

	int perc = (int)(value * 100.0 / 88.0);
	setValue(perc);
	emit valueChanged(this);
	update();
}



void Fader::setValue(uint newValue) {
	if (newValue > 100) {
//		char tmp[200];
//		sprintf( tmp, "[setValue] Error: value > 100 (value = %d)", newValue );
//		errorLog( tmp );
		newValue = 100;
	}
	else if (newValue < 0) {
//		char tmp[200];
//		sprintf( tmp, "[setValue] Error: value < 0 (value = %d)", newValue );
//		errorLog( tmp );
		newValue = 0;
	}

	if (this->value != newValue) {
		this->value = newValue;
		changed = true;
	}
}



uint Fader::getValue() {
	return value;
}



///
/// Set peak value (0.0 .. 1.0)
///
void Fader::setPeak_L(float peak) {
	if (peak < 0.01f) {
//		char tmp[200];
//		sprintf( tmp, "[setPeak_L] Error: peak < 0.01 (peak = %f)", peak );
//		errorLog( tmp );
		peak = 0.01f;
	}
	else if (peak > 1.0f) {
//		char tmp[200];
//		sprintf( tmp, "[setPeak_L] Error: peak > 1 (peak = %f)", peak );
//		errorLog( tmp );
		peak = 1.0f;
	}

	if (this->peakValue_L != peak) {
		this->peakValue_L = peak;
		changed = true;
	}
}




///
/// Set peak value (0.0 .. 1.0)
///
void Fader::setPeak_R(float peak) {
	if (peak < 0.01f) {
//		char tmp[200];
//		sprintf( tmp, "[setPeak_R] Error: peak < 0.01 (peak = %f)", peak );
//		errorLog( tmp );
		peak = 0.01f;
	}
	else if (peak > 1.0f) {
//		char tmp[200];
//		sprintf( tmp, "[setPeak_R] Error: peak > 1 (peak = %f)", peak );
//		errorLog( tmp );
		peak = 1.0f;
	}

	if (this->peakValue_R != peak) {
		this->peakValue_R = peak;
		changed = true;
	}
}



void Fader::paintEvent( QPaintEvent*) {
	if (!isVisible()) {
		return;
	}

	if (changed) {
		changed = false;

//		infoLog( "paint" );

		// background
		bitBlt( &temp, 0, 0, &back, 0, 0, 18, 100, CopyROP );

		// leds
		float peak_L = peakValue_L * 100.0;
		uint offset_L = (uint)( 100.0 - peak_L );
		bitBlt( &temp, 0, offset_L, &leds, 0, offset_L, 9, 100, CopyROP );

		float peak_R = peakValue_R * 100.0;
		uint offset_R = (uint)( 100.0 - peak_R );
		bitBlt( &temp, 9, offset_R, &leds, 9, offset_R, 18, 100, CopyROP );

		if (m_bWithoutKnob == false) {
			// knob
			static const uint knob_height = 12;
			static const uint knob_width = 14;
			uint knob_y = (uint)( 100.0 - (88.0 * value / 100.0) );
			bitBlt( &temp, 2, knob_y - knob_height, &knob, 0, 0, knob_width, knob_height, CopyROP );
		}

		setErasePixmap( temp );

		// copio l'immagine definitiva
//		bitBlt( this, 0, 0, &temp, 0, 0, 18, 100, CopyROP );	// is this really necessary?
	}
}



void Fader::updateFader() {
	if (changed && isVisible()) {
		update();
	}
}



//////////////////////////////////



PanFader::PanFader(QWidget * parent) : QWidget(parent), Object( "PanFader" )
{
	setMinimumSize( 38, 14 );
	setMaximumSize( 38, 14);
	resize(38, 14);
	changed = true;
	value = 0;

	// Create temp image
	temp.resize( width(), height() );

	// Background image
	string background_path = string(IMG_PATH).append("/img/mixerPanel/panFader_background.png");
	bool ok = back.load(background_path.c_str());
	if( ok == false ) {
		qWarning("Fader: Error loading pixmap");
	}

}



PanFader::~PanFader() {
}



void PanFader::updateFader() {
	if (changed && isVisible()) {
		update();
	}
}


void PanFader::paintEvent( QPaintEvent*) {
	if (!isVisible()) {
		return;
	}


	if (changed) {
		changed = false;

		// background
		bitBlt(&temp, 0, 0, &back, 0, 0, 38, 14, CopyROP);

		QColor black(0, 150, 70);
		QPainter p(&temp);
		p.setPen(black);

		uint x_pos = (uint)(value / 100.0 * 36.0) + 1;

		// draw the vert line
		p.drawLine(x_pos - 1, 3, x_pos - 1, 10);
		p.drawLine(x_pos, 3, x_pos, 10);

		setErasePixmap(temp);
	}
	// copio l'immagine definitiva
//	bitBlt(this, 0, 0, &temp, 0, 0, 20, 14, CopyROP);
}



 void PanFader::mouseMoveEvent( QMouseEvent *ev ) {
	if (ev->state() == (LeftButton | ShiftButton) ) {	// restore central position with shift + left click
		float newValue = 50;
		setValue(newValue);
		emit valueChanged(this);
		return;
	}
	if (ev->state() == LeftButton ) {
		int x = ev->x();
		float newValue = x * 100.0 / 38.0;
		setValue(newValue);
		emit valueChanged(this);
	}
}



void PanFader::mousePressEvent(QMouseEvent *ev) {
	if (ev->state() == (ShiftButton) ) {	// restore central position with shift + left click
		float newValue = 50;
		setValue(newValue);
		emit valueChanged(this);
		return;
	}

	int x = ev->x();
	float newValue = x * 100.0 / 38.0;
	setValue(newValue);
	emit valueChanged(this);
}



void PanFader::setValue(float val) {
	if (val == value) {
		return;
	}
	if (val < 0.0) {
		val = 0.0;
	}
	else if (val > 100.0) {
		val = 100.0;
	}
//	if (val != value) {
//		value = val;
//		changed = true;
//	}
	value = val;
	changed = true;
}



///////////////////





QPixmap* Knob::m_background = NULL;


///
/// Constructor
///
Knob::Knob( QWidget* parent )  : QWidget(parent), Object( "Knob" )
{
//	infoLog( "INIT" );
	m_bChanged = true;
	m_nWidgetWidth = 18;
	m_nWidgetHeight = 18;
	m_fValue = 0.0;

	if ( m_background == NULL ) {
//		infoLog( "loading background pixmap" );
		string sBackground_path = string(IMG_PATH).append( "/img/mixerPanel/knob_images.png" );
		m_background = new QPixmap();
		if ( m_background->load( sBackground_path.c_str() ) == false ){
			errorLog( string("Error loading pixmap ") + sBackground_path );
		}
	}

	resize( m_nWidgetWidth, m_nWidgetHeight );
	m_temp.resize( m_nWidgetWidth, m_nWidgetHeight );
}



///
/// Destructor
///
Knob::~ Knob()
{
//	infoLog( "DESTROY" );
	delete m_background;
	m_background = NULL;
}



void Knob::paintEvent( QPaintEvent* ev )
{
	if (!isVisible()) {
		return;
	}

	if (m_bChanged) {
		m_bChanged = false;

		int nFrame = (int)(31.0 * m_fValue);
		int xPos = m_nWidgetWidth * nFrame;
		bitBlt(&m_temp, 0, 0, m_background, xPos, 0, m_nWidgetWidth, m_nWidgetHeight, CopyROP);

		setErasePixmap( m_temp );

		// copio l'immagine definitiva
//		bitBlt( this, 0, 0, &m_temp, 0, 0, m_nWidgetWidth, m_nWidgetHeight, CopyROP );
	}
}



void Knob::setValue( float fValue )
{
	if ( fValue == m_fValue ) {
		return;
	}

	if ( fValue < 0.0 ) {
		fValue = 0.0;
//		warningLog( "[setValue] fValue < 0" );
	}
	else if ( fValue > 1.0 ) {
		fValue = 1.0;
//		warningLog( "[setValue] fValue > 1" );
	}
	m_fValue = fValue;
	m_bChanged = true;
}



void Knob::mousePressEvent(QMouseEvent *ev)
{
	setCursor( QCursor( Qt::SizeVerCursor ) );

	m_fMousePressValue = m_fValue;
	m_fMousePressY = ev->y();
}



void Knob::mouseReleaseEvent( QMouseEvent *ev )
{
	setCursor( QCursor( Qt::ArrowCursor ) );
}



 void Knob::mouseMoveEvent( QMouseEvent *ev ) {
	float y = ev->y() - m_fMousePressY;
	float fNewValue = m_fMousePressValue - ( y / 100.0 );
	setValue( fNewValue );
	update();
	emit valueChanged(this);
}



void Knob::updateKnob()
{
	if (m_bChanged && isVisible()) {
		update();
	}
}
