/*
 * basic.c
 * * - Basic Authentication module
 * *
 * * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 * *
 * * This program is free software; you can redistribute it and/or
 * * modify it under the terms of the GNU General Public License
 * * as published by the Free Software Foundation; either version 2
 * * of the License, or (at your option) any later version.
 * * 
 * * This program is distributed in the hope that it will be useful,
 * * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * * GNU General Public License for more details.
 * * 
 * * You should have received a copy of the GNU General Public License
 * * along with this program; if not, write to the Free Software
 * * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"

#include <stdio.h>
#ifndef __USE_BSD
#define __USE_BSD
#endif
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif

#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include <sys/stat.h>

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "connection.h"
#include "log.h"
#include "logtime.h"
#include "sock.h"
#include "avl_functions.h"
#include "restrict.h"
#include "match.h"
#include "memory.h"
#include "basic.h"
#include "user.h"
#include "group.h"
#include "mount.h"

extern server_info_t info;
mutex_t authentication_mutex = {MUTEX_STATE_UNINIT};
mounttree_t *mounttree = NULL;
usertree_t *usertree = NULL;
grouptree_t *grouptree = NULL;
time_t lastrehash = 0;

void rehash_authentication_scheme()
{
	int rehash_it = 0;
	struct stat st;
	char *userfile, *mountfile, *groupfile;

	userfile = get_icecast_file(info.userfile, conf_file_e, R_OK);
	mountfile = get_icecast_file(info.mountfile, conf_file_e, R_OK);
	groupfile = get_icecast_file(info.groupfile, conf_file_e, R_OK);

	if (userfile)
		if (stat(userfile, &st) == 0) {
			if (st.st_mtime > lastrehash)
				rehash_it = 1;
		}
	if (!rehash_it && groupfile)
		if (stat(groupfile, &st) == 0) {
			if (st.st_mtime > lastrehash)
				rehash_it = 1;
		}
	if (!rehash_it && mountfile)
		if (stat(mountfile, &st) == 0) {
			if (st.st_mtime > lastrehash)
				rehash_it = 1;
		}
	if (rehash_it)
		parse_authentication_scheme();

	nfree(userfile);
	nfree(groupfile);
	nfree(mountfile);
}

void init_authentication_scheme()
{
	thread_create_mutex(&authentication_mutex);

	parse_authentication_scheme();
}

/*
 * Clean and setup authentication scheme.
 * Run every time any authentication file changes
 * Assert Class: 1
 */
void parse_authentication_scheme()
{
	thread_mutex_lock(&authentication_mutex);

	/*
	 * Make a clean slate 
	 */
	destroy_authentication_scheme();

	/*
	 * Parse user file and flip it into memory 
	 */
	parse_user_authentication_file();

	/*
	 * Dito with group file, with pointers to every user 
	 */
	parse_group_authentication_file();

	/*
	 * Dito with mount file, with pointers to every group 
	 */
	parse_mount_authentication_file();

	thread_mutex_unlock(&authentication_mutex);

	lastrehash = get_time();
}

void destroy_authentication_scheme()
{
	free_mount_tree(mounttree);

	free_group_tree(grouptree);

	free_user_tree(usertree);

	mounttree = create_mount_tree();
	grouptree = create_group_tree();
	usertree = create_user_tree();
}

int 
authenticate_user_request(connection_t *con, request_t *req)
{
	ice_user_t checkuser;

	grouptree_t *gt;
	group_t *group;

	avl_traverser trav = {0};

	if (con_get_user(con, &checkuser) == NULL)
		return 0;

	xa_debug(3, "DEBUG: Checking authentication for mount %s for user %s with pass %s", nullcheck_string (req->path), nullcheck_string (checkuser.name), 
		 nullcheck_string (checkuser.pass));

	rehash_authentication_scheme();

	thread_mutex_lock(&authentication_mutex);

	gt = get_grouptree_for_mount("all");

	if (!gt)
		gt = get_grouptree_for_mount(req->path);

	if (!gt)
		gt = get_grouptree_for_mount("rest");

	if (!checkuser.name || !checkuser.pass || !gt) {
		thread_mutex_unlock(&authentication_mutex);
		return 0;
	}
	if (user_authenticate(checkuser.name, checkuser.pass)) {
		while ((group = avl_traverse(gt, &trav))) {
			if (is_member_of(checkuser.name, group)) {
				nfree(checkuser.name);
				nfree(checkuser.pass);
				thread_mutex_unlock(&authentication_mutex);
				return 1;
			}
		}
	} else {
		xa_debug(1, "DEBUG: User authentication failed. Invalid user/password");
	}


	nfree(checkuser.name);
	nfree(checkuser.pass);
	thread_mutex_unlock(&authentication_mutex);
	return 0;
}

int need_authentication_on_mount(char *mount)
{
	request_t req;

	zero_request(&req);
	strcpy(req.path, mount);
	return need_authentication(&req);
}

int need_authentication(request_t * req)
{
	mount_t *mount;
	mount_t search;

	xa_debug(3, "DEBUG: Checking need for authentication on mount %s", req->path);

	rehash_authentication_scheme();

	thread_mutex_lock(&authentication_mutex);

	search.name = req->path;

	mount = avl_find(mounttree, &search);

	if (mount) {
		thread_mutex_unlock(&authentication_mutex);
		return 1;
	}
	thread_mutex_unlock(&authentication_mutex);

	return 0;
}
