/**
 * IceMC - a menu editor for IceWM
 * Copyright (c) 2000 Georg Mittendorfer
 */

///////////////////////////////////////////////////////
// IceMainWidget.cpp 
// Implementation of class IceMainWidget
// Stores the main data for IceMainWindow
// Declaration -> IceMainWindow.h
//
// created:  mig 00090?
// modified: mig 030222
///////////////////////////////////////////////////////

#include "IceMainWindow.h"
#include "IceFileSearch.h"
#include "IceListView.h"
#include "IceIconPreview.h"

#include <qcombobox.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpushbutton.h>
#include <qregexp.h>

#include "fileopen.xpm"

//#include "IceMainWindow.moc"

///////////////////////////////////////////////////////
// constructor & destructor
///////////////////////////////////////////////////////

IceMainWidget::IceMainWidget(QWidget * parent, const char * name, WFlags f) : QWidget (parent,name,f)
{
  // constructor: QWidget ( QWidget * parent=0, const char * name=0, WFlags f=0 )

  homeDir = std::getenv("HOME");
  setDefaultPaths();

  /* 3 textediting fields + labels */
  applEdit = new QLineEdit(this);
  iconEdit = new QLineEdit(this);
  execEdit = new QLineEdit(this);
  applEdit->setMinimumSize(180,25);
  iconEdit->setMinimumSize(180,25);
  execEdit->setMinimumSize(180,25);
  QLabel* applLabel = new QLabel(applEdit,"&Name",this);
  QLabel* iconLabel = new QLabel(iconEdit,"&Icon",this);
  QLabel* execLabel = new QLabel(execEdit,"&Exec",this);

  QPushButton* execPb = new QPushButton(this);
  QPushButton* iconPb = new QPushButton(this);
  execPb->setPixmap(fileopen);
  execPb->setMinimumSize(25,25);
  execPb->setMaximumWidth(25); 
  iconPb->setPixmap(fileopen);
  iconPb->setMinimumSize(25,25);
  execPb->setMaximumWidth(25);

  infoCombo = new QComboBox(false,this);
  infoCombo->setMinimumSize(85,25);
  infoCombo->setMaximumHeight(25);
  infoCombo->insertItem("menu");
  infoCombo->insertItem("prog");
  infoCombo->insertItem("separator");
  infoCombo->insertItem("restart");
  infoCombo->insertItem("menufile");
  infoCombo->insertItem("menuprog");

  mainview = new IceListView(this);
  mainview->addColumn("Application Name",120);
  mainview->addColumn("Icon",120);
  mainview->addColumn("Executable",250);
  mainview->setColumnWidthMode(0,IceListView::Maximum);
  mainview->setColumnWidthMode(2,IceListView::Maximum);
  mainview->setSorting(-1);
  mainview->setRootIsDecorated(true);
  mainview->setAllColumnsShowFocus(true);
  mainview->setSelectionMode(IceListView::Single);
  mainview->setMouseTracking(false);
  
  connect(mainview, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(displayListItem(QListViewItem*)));
  connect(applEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListAppl(const QString&)));
  connect(iconEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListIcon(const QString&)));
  connect(execEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListExec(const QString&)));
  connect(infoCombo, SIGNAL(activated(const QString&)), this, SLOT(updateInfoStr(const QString&)));
  connect(execPb, SIGNAL(clicked()), this, SLOT(newExec()));
  connect(iconPb, SIGNAL(clicked()), this, SLOT(newIcon()));

  /* layout */
  QVBoxLayout* mainLayout = new QVBoxLayout(this,2);
  QGridLayout* editLayout = new QGridLayout(5,6,10); // 5 rows 6 coloumns 10pix spacing

  editLayout->addWidget(applLabel,2,1,Qt::AlignRight);
  editLayout->addWidget(iconLabel,3,1,Qt::AlignRight);
  editLayout->addWidget(execLabel,4,1,Qt::AlignRight);
  editLayout->addWidget(applEdit,2,2);
  editLayout->addWidget(iconEdit,3,2);
  editLayout->addWidget(execEdit,4,2);
  editLayout->addMultiCellWidget(infoCombo,2,2,4,5);
  editLayout->addWidget(iconPb,3,4,Qt::AlignLeft);
  editLayout->addWidget(execPb,4,4,Qt::AlignLeft);
  editLayout->addColSpacing(3,10);
  editLayout->addColSpacing(6,20);
  editLayout->addRowSpacing(1,8);
  editLayout->addRowSpacing(5,2);

  mainLayout->addWidget(mainview);
  mainLayout->addLayout(editLayout,0);
  
  //  QPalette temp = new QPalette
  p = new QPainter(mainview); // or 'this'?
  cg = &((mainview->palette()).active()); // maybe this doesn't work => check
}

IceMainWidget::~IceMainWidget()
{
  delete p;
  // Qt does the rest of the job (i hope ;)
}

/////////////////////////////////////////////////////
// public static data
/////////////////////////////////////////////////////

QString IceMainWidget::homeDir = 0;
QString IceMainWidget::libPath = 0;

/////////////////////////////////////////////////////
// public methods
/////////////////////////////////////////////////////

void IceMainWidget::clearLineEdits()
{
  applEdit->clear();
  iconEdit->clear();
  execEdit->clear();
}

void IceMainWidget::setDefaultPaths()
{
  setLibPath();
  setIconPath();
  IceListView::path = std::getenv("PATH");
}

void IceMainWidget::setLibPath(QString userPath)
{
  libPath = homeDir + "/.icewm:/etc/X11/icewm:/usr/local/lib/X11/icewm"
    + ":/usr/X11R6/lib/X11/icewm:/usr/X11/lib/X11/icewm";
  if (!userPath.isEmpty())
    libPath = userPath + ":" + libPath;
}

void IceMainWidget::setIconPath(QString userPath)
{
  QString fName;
  QString iconPathTemp;
  QString iconPath = homeDir + "/.icewm:/etc/X11/icewm/icons:/usr/local/lib/X11/icewm/icons"
    + ":/usr/X11R6/lib/X11/icewm/icons:/usr/X11/lib/X11/icewm/icons";
  if (!userPath.isEmpty())
    iconPath = iconPath + ":" + userPath;
  else {
    IceFileSearch* search = new IceFileSearch();
    fName = search->fileReadable("preferences",iconPath);
    if (fName != "") { 
      // search preferences file for iconPath:
      // iconPathTemp = searchPref(fName,"IconPath");
      // if (!(iconPathTemp.isEmpty()))
      //   iconPath = iconPath + ":" + iconPathTemp; 
      //
    }
    delete search;
  }
  // data stored in icelistview! (ev change this?)
  IceListView::iconPath = iconPath;
}

QString IceMainWidget::getLibPath()
{
  return libPath;
}

QString IceMainWidget::getIconPath()
{
  return IceListView::iconPath;
}

QString IceMainWidget::getHomeDir()
{
  return homeDir;
}

void IceMainWidget::updateListItem(IceListViewItem* item, bool all)
{
  ;//qDebug("DEBUG: IceMainWidget::updateListItem");
  QString info;
  if (item) {
    mainview->setCurrentItem(item);
    ;//qDebug("DEBUG: updateItem: " + item->text(0));
    info = item->getInfo();
    // for all:
    updateListIcon(item->text(1));
    updateListExec(item->text(2));
    if (info == "separator")
      updateListAppl(" ------ ");
    if (all) // update all positions (signal should be sufficient)
      updateListAppl(item->text(0));
  }
  else
    qWarning("WARNING: null pointer (IceMainWidget::updateItem)");
}

/////////////////////////////////////////////////////
// public slots
/////////////////////////////////////////////////////

void IceMainWidget::displayListItem(QListViewItem* qItem)
{
  if (qItem) {

    // no update of QListView needed if only current item changed (update folder() -vs- performance)
    disconnect(applEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListAppl(const QString&)));
    disconnect(iconEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListIcon(const QString&)));
    disconnect(execEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListExec(const QString&)));

    IceListViewItem* item;
    QString info;
    // typesafe cast QListViewItem to IceListViewItem
    item = dynamic_cast<IceListViewItem*> (qItem);

    ;//qDebug("DEBUG: displayListItem: " + item->text(0));	
    
    applEdit->setEnabled(true);
    iconEdit->setEnabled(true);
    execEdit->setEnabled(true);  
    
    applEdit->setText(item->text(0));
    iconEdit->setText(item->text(1));
    execEdit->setText(item->text(2));    

    info = item->getInfo();

    connect(applEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListAppl(const QString&)));
    connect(iconEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListIcon(const QString&)));
    connect(execEdit, SIGNAL(textChanged(const QString&)), this, SLOT(updateListExec(const QString&)));

    if ((info == "menu") || (info == "separator")) {
      execEdit->setEnabled(false);
      infoCombo->setCurrentItem(0);
      if (info == "separator") {
	applEdit->setEnabled(false);
	iconEdit->setEnabled(false);
	infoCombo->setCurrentItem(2);
      }
    }
    else {
      infoCombo->setEnabled(true);
      if (info == "prog")
	infoCombo->setCurrentItem(1);
      else if (info == "restart")
	infoCombo->setCurrentItem(3);
      else if (info == "menufile")
	infoCombo->setCurrentItem(4);
      else if (info == "menuprog")
	infoCombo->setCurrentItem(5);
      else {
	popupMessage("... to err is human ...","WARNING - No Type Selected\n"
		     "Please select appropriate Type for Item\n");
	qWarning("WARNING: IceMainWidget::displayListItem; unknown info string" + info);
      }
    }
  }
  else
    ;//qDebug("DEBUG: IceMainWidget::displayListItem(...): null pointer");
    // happens i.e. if cut() on last item
}

void IceMainWidget::updateListAppl(const QString& appl)
{
  IceListViewItem* item;
  item = mainview->currentItem();
  if (item) {
    ;//qDebug("DEBUG: updateListAppl: " + item->text(0));
    item->setText(0,appl);
  }
}

void IceMainWidget::updateListIcon(const QString& icon)
{
  IceListViewItem* item;
  item = mainview->currentItem();
  if (item) {
    ;//qDebug("DEBUG: updateListIcon: " + item->text(0));
    item->setText(1,icon);
    item->setIcon();
  }
}

void IceMainWidget::updateListExec(const QString& exec)
{
  IceListViewItem* item;
  item = mainview->currentItem();

  if (item) {
    ;//qDebug("DEBUG: updateListExec: " + item->text(0));
    item->setText(2,exec);
    if (item->getInfo() == "menu") {
      item->setExecState();
    }
    else if (item->getInfo() == "prog" || item->getInfo() == "restart") {
      item->setExecState();
      mainview->updateRootFolder(item); // this is a performance problem!
    }
  }
}

void IceMainWidget::updateInfoStr(const QString& selected)
{
  IceListViewItem* item;
  QString info;
  item = mainview->currentItem();

  if (item) {
    ;//qDebug("DEBUG: updateInfoStr: " + selected + " " + info + " " + item->text(0));
    info = item->getInfo();
#ifdef DEBUG
    cout << "updateInfoStr: info=" << info << "; selected=" << selected << endl;
#endif
    if (info != selected) {
      if (info == "menu") {
	if (item->firstChild()) {
	 popupMessage("Error", "Folder not empty!");
	 infoCombo->setCurrentItem(0);
	}
        else {
	  item->setInfo(selected);
	  item->setExpandable(false);
	  updateListItem(item);
	  displayListItem(item);
	}
      }
      else {
	item->setInfo(selected);
	if (selected == "menu")
	  item->setExpandable(true);
	else
	  item->setExpandable(false);
	updateListItem(item);
	displayListItem(item);
      }
    }
  }
}

/**
 * Select an executable using a file dialog
 */
void IceMainWidget::newExec()
{
  ;//qDebug("DEBUG: IceMainWidget::newExec()");
  QString file;
  int ok;
  static QString startDir = homeDir;

  QFileDialog* dialog = new QFileDialog(startDir,QString::null,this,"newExec",true);
  dialog->setShowHiddenFiles(true);
  dialog->setMode(QFileDialog::ExistingFile);
  dialog->setCaption("Please Select Application");
  ok = dialog->exec();

  file = dialog->selectedFile();
  if ( (!file.isEmpty()) && (ok) ) {
    if (execEdit->isEnabled())
      execEdit->setText(file);
    else
      popupMessage("... to err is human ...",
		   "Application Executable Field is not enabled.\n"
		   "Please select appropriate Type for Item\n"
		   "(e.g. 'prog' instead of 'separator' or 'menu')");
  }
  else {
    // canceled
  }

  startDir = dialog->dirPath(); // for next use
  delete dialog;
}

/**
 * Select a new icon by using a file dialog
 */
void IceMainWidget::newIcon()
{
  // TODO: implement a subclass of QFileIconProvider for QFileDialog
  ;//qDebug("DEBUG: IceMainWidget::newIcon()");
  QString icon;
  int ok;
  static QString startDir = homeDir;
  QString filterTypes("XPM Icons (*.xpm);;IceWM Menu Icons (*_16x16.xpm);;All Files (*.*)");

  IceIconPreview *p = new IceIconPreview(); 
  QFileDialog *dialog = new QFileDialog(startDir,QString::null,this,"newExec",true);
  dialog->setShowHiddenFiles(true);
  //dialog->addFilter("All Files (*.*)"); // not compatible with qt 2.3
//    dialog->addFilter("XPM Icons (*.xpm)"); // not compatible with qt 2.3
//    dialog->addFilter("IceWM Icons (*_16x16.xpm)"); // not compatible with qt 2.3
  dialog->setFilters(filterTypes);
  dialog->setMode(QFileDialog::ExistingFile);
  dialog->setCaption("Please Select Icon");
  dialog->setContentsPreviewEnabled(true);
  dialog->setContentsPreview(p,p);
  dialog->setPreviewMode(QFileDialog::Contents);
  ok = dialog->exec();

  icon = dialog->selectedFile();
  if ( (!icon.isEmpty()) && (ok) ) {
    if (iconEdit->isEnabled()) {
      icon = icon.replace( QRegExp("_..x..\\.xpm"), "" );
      iconEdit->setText(icon);
    }
    else
      popupMessage("... to err is human ...","Icon Field is not enabled.\n"
		   "Please select appropriate Type for Item\n"
		   "(e.g. 'prog' instead of 'separator')");
  }
  else {
    // canceled
  }

  startDir = dialog->dirPath(); // for next use  
  delete dialog;
}

/////////////////////////////////////////////////////////////////
// private methods
/////////////////////////////////////////////////////////////////

void IceMainWidget::popupMessage(QString caption, QString msg)
{
  QMessageBox::warning( this, caption, msg, "Ok");
  //  QMessageBox warnBox(caption, msg, QMessageBox::Warning, QMessageBox::Ok | QMessageBox::Default, 0, 0);
  //  warnBox.show(); 
}









