/*
  oscarprotocol.cpp  -  Oscar Protocol Plugin

  Copyright (c) 2002 by Tom Linsky <twl6@po.cwru.edu>

  Kopete    (c) 2002-2003 by the Kopete developers  <kopete-devel@kde.org>

  *************************************************************************
  *                                                                       *
  * This program is free software; you can redistribute it and/or modify  *
  * it under the terms of the GNU General Public License as published by  *
  * the Free Software Foundation; either version 2 of the License, or     *
  * (at your option) any later version.                                   *
  *                                                                       *
  *************************************************************************
  */


#include <kgenericfactory.h>
#include <kdebug.h>

#include "aimprotocol.h"
#include "aimaccount.h"
#include "aimcontact.h"
#include "aimaddcontactpage.h"
#include "aimeditaccountwidget.h"

#include "accountselector.h"
#include "kopeteaccountmanager.h"
#include "kopeteglobal.h"
#include "kopeteuiglobal.h"

#include <kdialogbase.h>
#include <kmessagebox.h>

typedef KGenericFactory<AIMProtocol> AIMProtocolFactory;

K_EXPORT_COMPONENT_FACTORY( kopete_aim, AIMProtocolFactory( "kopete_aim_protocol" ) )

AIMProtocol* AIMProtocol::protocolStatic_ = 0L;


AIMProtocolHandler::AIMProtocolHandler() : Kopete::MimeTypeHandler(false)
{
	registerAsProtocolHandler(QString::fromLatin1("aim"));
}

void AIMProtocolHandler::handleURL(const KURL &url) const
{
/**
 * Send a Message  =================================================
 * aim:goim
 * aim:goim?screenname=screen+name
 * aim:goim?screenname=screen+name&message=message
 * Add Buddy  ======================================================
 * aim:addbuddy
 * aim:addbuddy?screenname=screen+name
 * Buddy Icon  =====================================================
 * aim:buddyicon
 * aim:buddyicon?src=image_source
 * aim:buddyicon?screename=screen+name
 * aim:buddyicon?src=image_source&screename=screen+name
 * Get and Send Files  =============================================
 * aim:getfile?screename=(sn)
 * aim:sendfile?screenname=(sn)
 * Register User  ==================================================
 * aim:RegisterUser?ScreenName=sn&Password=pw&SignonNow=False
 * Away Message  ===================================================
 * aim:goaway?message=brb+or+something
 **/

	AIMProtocol *proto = AIMProtocol::protocol();
	kdDebug(14190) << k_funcinfo << "URL url   : '" << url.url() << "'" << endl;
	QString command = url.path();

	if (command.startsWith("goim") || command.startsWith("addbuddy"))
	{
		if (command.startsWith("goim"))
			command.remove(0,4);
		else
			command.remove(0,8);

		if (!command.startsWith("?screenname="))
		{
			kdWarning(14190) << "Unhandled aim URI : " << url.url() << endl;
			return;
		}

		command.remove(0, 12);

		int andSign = command.find("&");
		if (andSign > 0) // strip off anything else for now
			command = command.left(andSign);
		command.replace("+", " ");

		QString screenname = tocNormalize(command);

		KopeteAccount *account = 0;
		QDict<KopeteAccount> accounts = KopeteAccountManager::manager()->accounts(proto);
		// do not show chooser if we only have one account to "choose" from
		if (accounts.count() == 1)
		{
			QDictIterator<KopeteAccount> it(accounts);
			account = it.current();

			if (KMessageBox::questionYesNo(Kopete::UI::Global::mainWidget(),
				i18n("Do you want to add '%1' to your contact list?").arg(command))
				!= KMessageBox::Yes)
			{
				kdDebug(14190) << k_funcinfo << "Cancelled" << endl;
				return;
			}
		}
		else
		{
			KDialogBase *chooser = new KDialogBase(0, "chooser", true,
				i18n("Choose Account"), KDialogBase::Ok|KDialogBase::Cancel,
				KDialogBase::Ok, false);
			AccountSelector *accSelector = new AccountSelector(proto, chooser,
				"accSelector");
			chooser->setMainWidget(accSelector);

			int ret = chooser->exec();
			KopeteAccount *account = accSelector->selectedItem();

			delete chooser;
			if (ret == QDialog::Rejected || account == 0)
			{
				kdDebug(14190) << k_funcinfo << "Cancelled" << endl;
				return;
			}
		}


		kdDebug(14190) << k_funcinfo <<
			"Adding Contact; screenname = " << screenname << endl;
		if (account->addContact(screenname, command, 0L,
			KopeteAccount::DontChangeKABC, QString::null, true))
		{
			//KopeteContact *contact = account->contacts()[screenname];
		}


	}
	else
	{
		kdWarning(14190) << "Unhandled aim URI : " << url.url() << endl;
	}
}




AIMProtocol::AIMProtocol(QObject *parent, const char *name, const QStringList &)
: KopeteProtocol( AIMProtocolFactory::instance(), parent, name ),
	statusOnline(KopeteOnlineStatus::Online, 1, this, OSCAR_ONLINE, QString::null, i18n("Online"), i18n("Online")),
	statusOffline(KopeteOnlineStatus::Offline, 1, this, OSCAR_OFFLINE, QString::null, i18n("Offline"), i18n("Offline")),
	statusAway(KopeteOnlineStatus::Away, 1, this, OSCAR_AWAY, "aim_away", i18n("Away"), i18n("Away")),
	statusConnecting(KopeteOnlineStatus::Connecting, 99, this, OSCAR_CONNECTING, "aim_connecting", i18n("Connecting..."), i18n("Connecting...")),
	awayMessage(Kopete::Global::Properties::self()->awayMessage()),
	clientFeatures("clientFeatures", i18n("Client Features"), 0, false)
{
	if (protocolStatic_)
		kdDebug(14190) << k_funcinfo << "AIM plugin already initialized" << endl;
	else
	{
		protocolStatic_ = this;
		// Create the config widget, this does it's magic I think
//		new OscarPreferences("aim_protocol", this);
	}
	addAddressBookField("messaging/aim", KopetePlugin::MakeIndexField);
}

AIMProtocol::~AIMProtocol()
{
	protocolStatic_ =0L;
}

AIMProtocol *AIMProtocol::protocol(void)
{
	return protocolStatic_;
}

KopeteContact *AIMProtocol::deserializeContact(KopeteMetaContact *metaContact,
	const QMap<QString, QString> &serializedData,
	const QMap<QString, QString> &/*addressBookData*/)
{
	QString contactId=serializedData["contactId"];
	QString accountId=serializedData["accountId"];
	QString displayName=serializedData["displayName"];

	// Get the account it belongs to
	QDict<KopeteAccount> accounts = KopeteAccountManager::manager()->accounts(this);
	KopeteAccount *account = accounts[accountId];

	if(!account)
	{
		kdDebug(14190) << k_funcinfo << "WARNING: Account for contact does not exist, skipping." << endl;
		return 0;
	}

	AIMContact *c = new AIMContact(contactId, displayName,
		static_cast<AIMAccount*>(account), metaContact);

	return c;
}

AddContactPage *AIMProtocol::createAddContactWidget(QWidget *parent, KopeteAccount *account)
{
	return (new AIMAddContactPage(account->isConnected(), parent));
}

KopeteEditAccountWidget *AIMProtocol::createEditAccountWidget(KopeteAccount *account, QWidget *parent)
{
	return (new AIMEditAccountWidget(this, account, parent));
}

KopeteAccount *AIMProtocol::createNewAccount(const QString &accountId)
{
	return (new AIMAccount(this, accountId));
}

#include "aimprotocol.moc"
// vim: set noet ts=4 sts=4 sw=4:
