/* @(#) conf.c 1.27 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: CORE
\***************************************************************/

#define _KOALAMUD_CONF_C "@(#) nitehawk@localhost.1ststep.net|lib/koala/conf.c|20000827025247|29135 @(#)"

#include "autoconf.h"

#include "version.h"
#include "koalatypes.h"

#include "conf.h"
#include "memory.h"
#include "log.h"
#include "network.h"

/* Gnome-XML (libxml) Includes */
#include "parser.h"

xmlDocPtr configfile = NULL;
xmlNodePtr daemonconfig = NULL;

koalaerror gendefaultconfig(const char *filename)
{
	xmlDocPtr doc;
	xmlNodePtr tree, subtree, uplinktree;

	doc = xmlNewDoc("1.0");
	doc->root = xmlNewDocNode(doc, NULL, "KOALACONFIG", "\n"
			"In order for this configuration to be used in a larger setup,\n"
			"the node ID's must be changed.\n"
			"This default configuration will work for either\n"
			"a client to zone setup or a client to hub to zone setup");
	xmlSetProp(doc->root, "configversion", "1");

	/* Build the tree for client daemon configuration */
	tree = xmlNewChild(doc->root, NULL, "CLIENT", "Client configuration section");
	xmlSetProp(tree, "Background", "false");
	xmlSetProp(tree, "NodeID", "10.0.0.2");
	subtree = xmlNewChild(tree, NULL, "Listen", NULL);
	xmlSetProp(subtree, "port", "6464");
	xmlSetProp(subtree, "bind", "any");
	subtree = xmlNewChild(tree, NULL, "Log", NULL);
	xmlSetProp(subtree, "message", "log/climsg.log");
	xmlSetProp(subtree, "error", "log/clierr.log");
	subtree = xmlNewChild(tree, NULL, "Timer", NULL);
	xmlSetProp(subtree, "ticklen", "100000");
	xmlSetProp(subtree, "reportingperiod", "600");
	subtree = xmlNewChild(tree, NULL, "Database", NULL);
	xmlSetProp(subtree, "path", "clientdb");
	subtree = xmlNewChild(tree, NULL, "Uplink", "\n"
			"Client daemons can only connect to a single uplink");
	uplinktree = xmlNewChild(subtree, NULL, "Link", "Localhost Uplink");
	xmlSetProp(uplinktree, "address", "localhost");
	xmlSetProp(uplinktree, "port", "8204");

	/* Build the tree for hub daemon configuration */
	tree = xmlNewChild(doc->root, NULL, "HUB", "Hub configuration section");
	xmlSetProp(tree, "Background", "false");
	xmlSetProp(tree, "NodeID", "10.0.0.1");
	subtree = xmlNewChild(tree, NULL, "Listen", NULL);
	xmlSetProp(subtree, "port", "8204");
	xmlSetProp(subtree, "bind", "any");
	subtree = xmlNewChild(tree, NULL, "Log", NULL);
	xmlSetProp(subtree, "message", "log/hubmsg.log");
	xmlSetProp(subtree, "error", "log/huberr.log");
	subtree = xmlNewChild(tree, NULL, "Timer", NULL);
	xmlSetProp(subtree, "ticklen", "100000");
	xmlSetProp(subtree, "reportingperiod", "600");
	subtree = xmlNewChild(tree, NULL, "Uplink", "\n"
			"Hub Daemons by default do not create any uplinks\n");
	uplinktree = xmlNewChild(subtree, NULL, "Link", "Localhost Uplink");
	xmlSetProp(uplinktree, "address", "localhost");
	xmlSetProp(uplinktree, "port", "8204");

	/* Build the tree for zone daemon configuration */
	tree = xmlNewChild(doc->root, NULL, "ZONE", "Zone configuration section");
	xmlSetProp(tree, "Background", "false");
	xmlSetProp(tree, "NodeID", "10.0.0.3");
	subtree = xmlNewChild(tree, NULL, "Listen", NULL);
	xmlSetProp(subtree, "port", "8204");
	xmlSetProp(subtree, "bind", "any");
	subtree = xmlNewChild(tree, NULL, "Log", NULL);
	xmlSetProp(subtree, "message", "log/zonmsg.log");
	xmlSetProp(subtree, "error", "log/zonerr.log");
	subtree = xmlNewChild(tree, NULL, "Timer", NULL);
	xmlSetProp(subtree, "ticklen", "100000");
	xmlSetProp(subtree, "reportingperiod", "600");
	subtree = xmlNewChild(tree, NULL, "Database", NULL);
	xmlSetProp(subtree, "path", "zonedb");
	subtree = xmlNewChild(tree, NULL, "Uplink", "\n"
			"zone daemons by default connect to a local uplink\n"
			"Change 'connect=yes' to 'connect=no' to change this behavior");
	xmlSetProp(subtree, "connect", "yes");
	uplinktree = xmlNewChild(subtree, NULL, "Link", "Localhost Uplink");
	xmlSetProp(uplinktree, "address", "localhost");
	xmlSetProp(uplinktree, "port", "8204");

	xmlSetDocCompressMode(doc, 0);
	xmlSaveFile(filename, doc);
	
	return KESUCCESS;
}

koalaerror readxmlconfig(const char *filename)
{
	configfile = xmlParseFile(filename);
	if (configfile == NULL)
	{
		logmsg(LOGERR, "Error parsing config file");
		return KEINVALIDCONFIG;
	}

	if (strcmp(configfile->root->name, "KOALACONFIG"))
	{
		logmsg(LOGERR, "Invalid configuration file");
		return KEINVALIDCONFIG;
	}

	finddaemonconfig();

	return KESUCCESS;
}

/*	finddaemonconfig - Locate the configuration section covering the currently
 *		running daemon type
 */
koalaerror finddaemonconfig(void)
{
	daemonconfig = configfile->root->childs;
	while (daemonconfig)
	{
		switch(koptions.daemontype)
		{
		case DAEMON_CLIENT:
			if (strcasecmp(daemonconfig->name, "CLIENT") == 0)
			{
				return confgetnodeid();
			}
			break;
		case DAEMON_ZONE:
			if (strcasecmp(daemonconfig->name, "ZONE") == 0)
			{
				return confgetnodeid();
			}
			break;
		case DAEMON_HUB:
			if (strcasecmp(daemonconfig->name, "HUB") == 0)
			{
				return confgetnodeid();
			}
			break;
		default:
			logmsg(LOGWARN, "Can't locate the correct daemon config"
					" section until the daemon type is known");
			return KEUNKNOWNDAEMONTYPE;
		}
		daemonconfig = daemonconfig->next;
	}
	return KESUCCESS;
}

/*	confgetnodeid - Retrieve the nodeid from the configuration and stuff it
 *		into koptions
 *	
 *	This function should only be called via finddaemonconfig.
 */
koalaerror confgetnodeid(void)
{
	xmlAttrPtr attr = NULL;

	attr = daemonconfig->properties;
	while (attr)
	{
		/* Check to see if this is the port attributed */
		if (strcasecmp(attr->name, "nodeid") == 0)
		{
			/* we need to read and interpret the eight bit octets into a
			 * single 32bit value. */
			char str[strlen(attr->val->content)];
			char *pos = str;
			char *tok = str;
			int shiftcount;
			strcpy(str, attr->val->content);  // copy the val to preserve the o

			/*  We start by shifting 24 bits and work down to a 0 bit shift */
			for ( shiftcount = 24 ; shiftcount >= 0; shiftcount -= 8)
			{
				tok = strsep(&pos, ".");
				kstate.nodeid |= (u_int8_t)strtoul(tok, NULL, 10) << shiftcount;
			}

			return KESUCCESS;
		}

		attr = attr->next;
	}

	/* default nodeid to a random number, just to be safe */
	kstate.nodeid = (unsigned int)random();

	return KEINVALIDCONFIG;
}

/*	confcreatelisten - Create listen socket from the configuration file
 */
koalaerror confcreatelisten(listnodeptr list)
{
	xmlNodePtr listencfg = NULL;
	int port = 0;
	char *bindaddr = NULL;
	xmlAttrPtr attr = NULL;

	listencfg = daemonconfig->childs;
	while(listencfg)
	{
		if (strcasecmp(listencfg->name, "Listen") == 0)
		{
			break;
		}
		listencfg = listencfg->next;
		if (!listencfg)
		{
			/* We walked off the end of the list, break out */
			logmsg(LOGWARN, "Listen configuration not found,"
					" not opening listen port");
			return -KEBIND;
		}
	}

	/* listencfg now points to the node containing listen configuration */
	/* find the 'port' and 'bind' properties */
	attr = listencfg->properties;
	while (attr)
	{
		/* Check to see if this is the port attributed */
		if (strcasecmp(attr->name, "port") == 0)
		{
			if (port == 0)
			{
				port = atoi(attr->val->content);
			}
		}

		/* Check to see if this is the bind attribute */
		if (strcasecmp(attr->name, "bind") == 0)
		{
			if (bindaddr == NULL)
			{
				bindaddr = attr->val->content;
			}
		}

		attr = attr->next;
	}

	/* Verify both attributes were specified */
	if (port == 0)
	{
		logmsg(LOGWARN, "Listen tag did not include port attribute,"
				" not opening listener");
		return -KEBIND;
	}

	if (bindaddr == NULL)
	{
		logmsg(LOGWARN, "Listen tag did not include bind attribute,"
				" not opening listener");
		return -KEBIND;
	}

	/* Open the listen socket(s) */
	return netlisten(list, bindaddr, port);
}

koalaerror confopenlogs(void)
{
	xmlAttrPtr attr = NULL;
	xmlNodePtr logcfg = NULL;
	char outlogpath[PATH_MAX] = "output.log";
	char errlogpath[PATH_MAX] = "error.log";

	logcfg = daemonconfig->childs;
	while(logcfg)
	{
		if (strcasecmp(logcfg->name, "Log") == 0)
		{
			break;
		}
		logcfg = logcfg->next;
		if (!logcfg)
		{
			/* We walked off the end of the list, break out */
			logmsg(LOGERR, "Log configuration not found,"
					" not opening log files");
			return -KEINVALIDCONFIG;
		}
	}

	/* logcfg now points to the node containing log configuration */
	/* find the 'message' and 'error' properties */
	attr = logcfg->properties;
	while (attr)
	{
		/* Check to see if this is the port attributed */
		if (strcasecmp(attr->name, "message") == 0)
		{
			strcpy(outlogpath, attr->val->content);
		}

		/* Check to see if this is the bind attribute */
		if (strcasecmp(attr->name, "error") == 0)
		{
			strcpy(errlogpath, attr->val->content);
		}

		attr = attr->next;
	}

	return loginit(outlogpath, errlogpath);
}

bool confquerybackground(void)
{
	xmlAttrPtr attr = NULL;

	attr = daemonconfig->properties;
	while(attr)
	{
		if (strcasecmp(attr->name, "Background") == 0)
		{
			if (strcasecmp(attr->val->content, "true") == 0)
				return TRUE;
			if (strcasecmp(attr->val->content, "yes") == 0)
				return TRUE;
		}
		attr = attr->next;
	}
	return FALSE;
}

/* 	confcreateuplink -
 * 		Create an uplink connection to each uplink listed in the config file.
 * 		Client daemons will only create a single uplink.
 * 		Zone daemons will only create one uplink and only if the connect
 *	 		property is 'yes' or 'true' or '1'
 *	 	Hub daemons will connect to every uplink listed in the config section
 *
 *	 	The single parameter is a pointer to the linked list that the new
 *	 	descriptors should be linked into.
 */
koalaerror confcreateuplink(listnodeptr uplinklist)
{
	xmlNodePtr uplinkcfg = NULL;
	int port = 0;
	char *targaddr = NULL;
	xmlAttrPtr attr = NULL;
	pdescriptor uplink = NULL;
	bool zoneuplink = FALSE;
	koalaerror uplinkreturn;
	int uplinkscreated=0;

	uplinkcfg = daemonconfig->childs;
	/* First off, we need to find the section of the config that contains
	 * uplink data
	 */
	while(uplinkcfg)
	{
		if (strcasecmp(uplinkcfg->name, "uplink") == 0)
		{
			break;
		}
		uplinkcfg = uplinkcfg->next;
		if (!uplinkcfg)
		{
			/* We walked off the end of the list, break out */
			logmsg(LOGWARN, "No uplink configuration found."
					"  No uplinks will be connected.");
			logmsg(LOGWARN, "An admin can create an uplink"
					" in game using 'link'");
			return -KENOUPLINK;
		}
	}

	/* uplinkcfg now contains the node with 'uplinkcfg->childs' containing
	 * uplink configurations
	 */

	/* If this is a zone daemon, we now have the information to see if we need
	 * to create an uplink
	 */
	if (koptions.daemontype == DAEMON_ZONE)
	{
		/* Look for 'connect' property */
		attr = uplinkcfg->properties;
		while(attr)
		{
			if (strcasecmp(attr->name, "connect") == 0)
			{
				/* Check for 'yes', 'true', and '1'.  Anything else is assumed
				 * to be no
				 */
				if (strcasecmp(attr->val->content, "yes") == 0)
				{
					zoneuplink = TRUE;
					break;
				}
				if (strcasecmp(attr->val->content, "true") == 0)
				{
					zoneuplink = TRUE;
					break;
				}
				if (strcasecmp(attr->val->content, "1") == 0)
				{
					zoneuplink = TRUE;
					break;
				}
			}
			attr = attr->next;
		}
		if (!zoneuplink)
		{
			/* Connect property not found, assuming to be no */
			logmsg(LOGINFO, "No connect property found, running zone daemon in"
					"listen mode");
			return KENOUPLINK;
		}
	}

	/* At this point, we know that we are going to create at least one uplink.
	 *	Set uplinkcfg to point to what should be the first uplink config and
	 *	start creating uplinks - fall out of the loop when one uplink is
	 *	created for zone or client daemons
	 *
	 *	Only Client and Zone daemons are limited on the number of uplinks they
	 *	can create.  A hub daemon will attempt to connect to every uplink
	 *	listed in its config.  Client and zone daemons connect to the first
	 *	link listed that yields a successful connect.
	 */
	uplinkcfg = uplinkcfg->childs;
	/* Here's to complex while conditions :P */
	while ((((koptions.daemontype == DAEMON_CLIENT) ||
			(koptions.daemontype == DAEMON_ZONE)) && (uplinkscreated < 1))
			&& uplinkcfg)
	{
		/* First, make sure the current 'uplinkcfg' is actually an uplink
		 * config.  If it isn't, cycle until we find one then continue.
		 * Falling off the end of the list while 'uplinkscreated' is still 0
		 * is an error
		 */
		if (strcasecmp(uplinkcfg->name, "link") != 0)
		{
			/* Start looking for a node named 'link' */
			uplinkcfg = uplinkcfg->next;
			while(uplinkcfg)
			{
				/* Good, we found one */
				if (strcasecmp(uplinkcfg->name, "link") == 0)
					break;

				/* Keep Looking */
				uplinkcfg = uplinkcfg->next;
			}
		}

		/* Oops, we didn't find any links */
		if (uplinkcfg == NULL && uplinkscreated==0)
		{
			logmsg(LOGWARN, "No valid links found, proceeding offline");
			return KENOUPLINK;
		}
		/* Catch a null before we have problems */
		if (uplinkcfg == NULL)
			continue;

		/* Now that we know uplinkcfg points to a 'link', find the address and
		 * port that we are going to connect to
		 */
		attr = uplinkcfg->properties;
		while (attr)
		{
			/* Check to see if this is the port attributed */
			if (strcasecmp(attr->name, "port") == 0)
			{
				port = atoi(attr->val->content);
			}

			/* Check to see if this is the address attribute */
			if (strcasecmp(attr->name, "address") == 0)
			{
				targaddr = attr->val->content;
			}

			attr = attr->next;
		}

		/* Verify both attributes were specified */
		if (port == 0)
		{
			logmsg(LOGWARN, "Link tag did not include port attribute, not"
					"connecting to uplink");
			uplinkcfg = uplinkcfg->next;
			continue;
		}

		if (targaddr == NULL)
		{
			logmsg(LOGWARN, "Link tag did not include address attribute, not"
					"connecting to uplink");
			uplinkcfg = uplinkcfg->next;
			continue;
		}

		/* Log that we are attempting to uplink to the current uplink */
		logmsg(LOGINFO, "Attempting to create uplink to: %s",
				uplinkcfg->childs->content);

		/* We need to allocate memory for the uplink descriptor now */
		if ((uplink = allocdescriptor()) == NULL)
		{
			/* EEK!! malloc fail! */
			logmsg(LOGERR,
				"Unable to allocate memory for new uplink descriptor!");
			return KENOMEM;
		}
		
		/* Open the uplink socket */
		uplinkreturn = netuplink(uplink, targaddr, port);
		/* Did we get an uplink? */
		if (uplinkreturn == KESUCCESS)
		{
			uplinkscreated++;
			/* Link the new descriptor into the list and null the pointer */
			listaddnode(uplinklist, uplink);
			uplink = NULL;
		}
		else
		{
			/* Uhoh, bad things happened.  Lets try the next one */
			/* Free memory from uplink */
			freedescriptor(uplink);
			uplink = NULL;
		}

		uplinkcfg = uplinkcfg->next;
	}

	return KESUCCESS;
}

/* confgetoptions - get generic options from the config file */
koalaerror confgetoptions(void)
{
	xmlAttrPtr attr = NULL;
	xmlNodePtr cfg = NULL;

	cfg = daemonconfig->childs;
	while(cfg)
	{
		if (strcasecmp(cfg->name, "timer") == 0)
		{
			/* cfg now points to the node containing log configuration */
			/* find the 'message' and 'error' properties */
			attr = cfg->properties;
			while (attr)
			{
				/* Check to see if this is the port attributed */
				if (strcasecmp(attr->name, "ticklen") == 0)
				{
					koptions.ticklen = 
						(unsigned int)strtol(attr->val->content, NULL, 10);
				}

				/* Check to see if this is the bind attribute */
				if (strcasecmp(attr->name, "reportingperiod") == 0)
				{
					koptions.reportingperiod = 
						(unsigned int)strtol(attr->val->content, NULL, 10);
				}

				attr = attr->next;
			}

		}
		if (strcasecmp(cfg->name, "Database") == 0)
		{
			/* cfg now points to the node containing database configuration */
			attr = cfg->properties;
			while (attr)
			{
				/* Get path attribute */
				if (strcasecmp(attr->name, "path") == 0)
				{
					// Allocate Memory
					koptions.dbrootpath = kmalloc(strlen(attr->val->content),
							ALLOC_GENERIC);
					if (koptions.dbrootpath == NULL)
					{
						return KENOMEM;
					}
					// Copy the path
					strcpy(koptions.dbrootpath, attr->val->content);
				}

				attr = attr->next;
			}

		}
		cfg = cfg->next;
	}


	return KESUCCESS;
}

/* confgetoptionsfromsection
 *	Read a section of options from the configuration file.  The section is
 *	described by the string 'section'.  The options to be read are in an array
 *	of 'confoption_t' structs containing the option name, an enum of the
 *	option type, and a union with pointers to the object variables.  In the
 *	case of strings, this function will allocate memory for the string.
 *
 *	If an option is not found, the variables sent are left unchanged.
 */
koalaerror confgetoptionsfromsection(char *section, confoption_t *optionlist,
		int num)
{
	int i = 0;
	xmlAttrPtr opt = NULL;
	xmlNodePtr sect = NULL;

	/* Find the section */
	sect = daemonconfig->childs;
	while (sect)
	{
		if (strcasecmp(sect->name, section) == 0)
			/* We found it */
			break;
		
		sect = sect->next;
	}

	/* Make sure we found the section */
	if (sect == NULL)
	{
		return KECONFSECTIONNOTFOUND;
	}

	/* Now that we found the appropriate section, we now have to scan through
	 * the attributes for the matching options */
	opt = sect->properties;
	while (i < num)
	{
		if (opt == NULL)
		{
			opt = sect->properties;
			i++;
		}

		if (strcasecmp(opt->name, optionlist[i].optname) == 0)
		{
			/* We found the current option, copy data to target pointer */
			switch (optionlist[i].opttype)
			{
			case OPT_STRING:
				/* First we need to allocate memory for the string */
				*(optionlist[i].opt.ostring)=kmalloc(strlen(opt->val->content),
						ALLOC_DATABASE);
				if (*(optionlist[i].opt.ostring) == NULL)
				{
					return KENOMEM;
				}
				/* Copy option string */
				strcpy(*(optionlist[i].opt.ostring), opt->val->content);
				break;

			case OPT_UINT:
				*(optionlist[i].opt.ouint) = 
						(unsigned int)strtoul(opt->val->content, NULL, 10);
				break;

			case OPT_INT:
				*(optionlist[i].opt.oint) = 
						(int)strtol(opt->val->content, NULL, 10);
				break;

			case OPT_ULONG:
				*(optionlist[i].opt.oulong) = 
						strtoul(opt->val->content, NULL, 10);
				break;

			case OPT_LONG:
				*(optionlist[i].opt.olong) = 
						strtol(opt->val->content, NULL, 10);
				break;

			case OPT_DOUBLE:
				*(optionlist[i].opt.odouble) = atof(opt->val->content);
				break;

			case OPT_BOOL:
				if (strcasecmp(opt->val->content, "yes") == 0)
					*(optionlist[i].opt.obool) = TRUE;
				if (strcasecmp(opt->val->content, "true") == 0)
					*(optionlist[i].opt.obool) = TRUE;
				if (strcasecmp(opt->val->content, "1") == 0)
					*(optionlist[i].opt.obool) = TRUE;
				if (strcasecmp(opt->val->content, "on") == 0)
					*(optionlist[i].opt.obool) = TRUE;
				if (strcasecmp(opt->val->content, "no") == 0)
					*(optionlist[i].opt.obool) = FALSE;
				if (strcasecmp(opt->val->content, "false") == 0)
					*(optionlist[i].opt.obool) = FALSE;
				if (strcasecmp(opt->val->content, "0") == 0)
					*(optionlist[i].opt.obool) = FALSE;
				if (strcasecmp(opt->val->content, "off") == 0)
					*(optionlist[i].opt.obool) = FALSE;
				break;

			default:
				/* Major problems here */
				logmsg(LOGCRIT, "Fatal error reading configuration options");
			}

			i++;
			opt = sect->properties;
			continue;
		}
		opt = opt->next;
	}

	return KESUCCESS;
}
