// =============================================================================
//
//      --- kvi_variablecache.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviVariableCache"

#include "kvi_debug.h"
#include "kvi_string.h"
#include "kvi_variablecache.h"

KviVariableCache::KviVariableCache()
{
	m_pVarList = new QPtrList<KviVariable>;
	m_pVarList->setAutoDelete(true);
	m_pDictList = new QPtrList<KviDictionary>;
	m_pDictList->setAutoDelete(true);
}

KviVariableCache::~KviVariableCache()
{
	while( m_pDictList->first() )
		deleteDict(m_pDictList->first());
	if( m_pDictList ) { delete m_pDictList; m_pDictList = 0; }
	if( m_pVarList  ) { delete m_pVarList;  m_pVarList  = 0; }
}

KviVariable *KviVariableCache::set(const char *name, const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable *v = getVariable(name);
	if( v ) {
		v->szValue = value;
		return v;
	}
	return insertNewVariable(name, value);
}

void KviVariableCache::concat(const char *name, const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable *v = getVariable(name);
	if( v ) {
		v->szValue.append(value);
		return;
	}
	insertNewVariable(name, value);
}

void KviVariableCache::concatWithSeparator(const char *name, char sep, const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable *v = getVariable(name);
	if( v ) {
		v->szValue.append(sep);
		v->szValue.append(value);
		return;
	}
	insertNewVariable(name, value);
}

KviVariable *KviVariableCache::insertNewVariable(const char *name, const char *value)
{
	KviVariable *v = new KviVariable;
	v->szName = name;
	v->szValue = value;
	// Insert in alphabetic order
	int idx = 0;
	for( KviVariable *b = m_pVarList->first(); b; b = m_pVarList->next() ) {
		if( kvi_strcmpCI(b->szName.ptr(), name) < 0 ) {
			m_pVarList->insert(idx, v);
			return v;
		}
		idx++;
	}
	m_pVarList->append(v);
	return v;
}

void KviVariableCache::unset(const char *name)
{
	__range_valid(name);
	KviVariable *v = getVariable(name);
	if( v ) m_pVarList->removeRef(v);
}

QPtrList<KviVariable> *KviVariableCache::variableList()
{
	return m_pVarList;
}

KviVariable *KviVariableCache::getVariable(const char *name)
{
	__range_valid(name);
	int r;
	for(KviVariable *v = m_pVarList->first(); v; v = m_pVarList->next() ) {
		r = kvi_strcmpCI(v->szName.ptr(), name);
		if( r == 0 )
			return v; // Equal
		if( r < 0 )
			return 0; // Found a greater one
	}
	return 0;
}

const char *KviVariableCache::find(const char *name)
{
	__range_valid(name);
	KviVariable *v = getVariable(name);
	if( v ) return v->szValue.ptr();
	return 0;
}

// =============================================================================
// Dictionaries

void KviVariableCache::deleteDict(KviDictionary *dict)
{
	__range_valid(dict->m_pVarCache);
	__range_valid(m_pDictList->find(dict) != -1);
	if( dict->m_pVarCache ) {
		delete dict->m_pVarCache;
		dict->m_pVarCache = 0;
	}
	m_pDictList->removeRef(dict);
}

KviDictionary *KviVariableCache::insertNewDict(const char *name)
{
	__range_invalid(getDict(name));
	KviDictionary *dict = new KviDictionary;
	dict->szName = name;
	dict->m_pVarCache = new KviVariableCache();
	// Insert in alphabetic order
	int idx = 0;
	for( KviDictionary *b = m_pDictList->first(); b; b = m_pDictList->next() ) {
		if( kvi_strcmpCI(b->szName.ptr(), name) < 0 ) {
			m_pDictList->insert(idx, dict);
			return dict;
		}
		idx++;
	}
	m_pDictList->append(dict);
	return dict;
}

KviDictionary *KviVariableCache::getDict(const char *name)
{
	__range_valid(name);
	int r;
	for( KviDictionary *v = m_pDictList->first(); v; v = m_pDictList->next() ) {
		r = kvi_strcmpCI(v->szName.ptr(), name);
		if( r == 0 )
			return v; // Equal
		if( r < 0 )
			return 0; // Found a greater one
	}
	return 0;
}

KviVariable *KviVariableCache::getDictVariable(const char *dictname, const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviDictionary *dict = getDict(dictname);
	if( !dict ) return 0;
	return dict->m_pVarCache->getVariable(varname);
}

const char *KviVariableCache::findDictVariable(const char *dictname, const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviVariable *v = getDictVariable(dictname, varname);
	if( v ) return v->szValue.ptr();
	return 0;
}

KviVariable *KviVariableCache::setDictVariable(const char *dictname, const char *varname, const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary *dict = getDict(dictname);
	if( !dict ) {
		dict = insertNewDict(dictname);
		return dict->m_pVarCache->insertNewVariable(varname, value);
	} else return dict->m_pVarCache->set(varname, value);
}

void KviVariableCache::unsetDictVariable(const char *dictname, const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviDictionary *dict = getDict(dictname);
	if( !dict ) return;
	dict->m_pVarCache->unset(varname);
	if( dict->m_pVarCache->m_pVarList->count() == 0 ) deleteDict(dict);
}

void KviVariableCache::concatDictVariable(const char *dictname, const char *varname, const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary *dict =  getDict(dictname);
	if( !dict ) dict = insertNewDict(dictname);
	dict->m_pVarCache->concat(varname, value);
}

void KviVariableCache::concatWithSeparatorDictVariable(
	const char *dictname, const char *varname, char sep, const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary *dict = getDict(dictname);
	if( !dict ) dict = insertNewDict(dictname);
	dict->m_pVarCache->concatWithSeparator(varname, sep, value);
}

QPtrList<KviDictionary> *KviVariableCache::dictList()
{
	return m_pDictList;
}
