// =============================================================================
//
//      --- kvi_systray_onlinetimer.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviSysTrayOnlineTimer"

#include <qpainter.h>
#include <qpixmap.h>

#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_string.h"
#include "kvi_systray.h"
#include "kvi_systray_onlinetimer.h"

void kvi_timeFormatDuration(time_t diff, KviStr &buffer)
{
	int days = diff / 86400;    // 86400 secs in a day
	diff = diff % 86400;
	int hours = diff / 3600;    // 3600 secs in a hour
	diff = diff % 3600;
	int mins = diff /    60;    // 60 secs in a minute
	diff = diff % 60;           // diff == secs!
	buffer.sprintf(_i18n_("%d d %d h %d m %d s"), days, hours, mins, diff);
}

KviSysTrayOnLineTimer::KviSysTrayOnLineTimer(KviSysTray *parent)
	: KviSysTrayWidget(parent, _i18n_("Online Timer"), "systray_online_timer")
{
	m_iConnectionStartTime = 0;
	m_timer                = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayOnLineTimer::~KviSysTrayOnLineTimer()
{
	if( m_timer )
		killTimer(m_timer);
}

QString KviSysTrayOnLineTimer::icon()
{
	return SYSTRAY_ICON_ONLINE_TIMER;
}

void KviSysTrayOnLineTimer::show()
{
	m_timer = startTimer(1000);
	KviSysTrayWidget::show();
}

void KviSysTrayOnLineTimer::hide()
{
	if( m_timer ) {
		killTimer(m_timer);
		m_timer = 0;
	}
	KviSysTrayWidget::hide();
}

void KviSysTrayOnLineTimer::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_timer )
		paintEvent(0);
}

void KviSysTrayOnLineTimer::run()
{
	m_iConnectionStartTime = time(0);
	if( isVisible() )
		paintEvent(0);
}

void KviSysTrayOnLineTimer::stop()
{
	m_iConnectionStartTime = 0;
	if( isVisible() )
		paintEvent(0);
}

void KviSysTrayOnLineTimer::getConnectionTime(KviStr &buffer)
{
	if( m_iConnectionStartTime == 0 )
		kvi_timeFormatDuration((time_t) 0, buffer);
	else
		kvi_timeFormatDuration((time_t) (time(0) - m_iConnectionStartTime), buffer);
}

int KviSysTrayOnLineTimer::getIntConnectionTime()
{
	if( m_iConnectionStartTime == 0 )
		return 0;
	else
		return (int) (time(0) - m_iConnectionStartTime);
}

unsigned int KviSysTrayOnLineTimer::getConnectionStartTime()
{
	return (unsigned int) m_iConnectionStartTime;
}

void KviSysTrayOnLineTimer::paintEvent(QPaintEvent *)
{
	if( !isVisible() )
		return;

	QPainter pa(m_pSysTray->getMemoryBuffer());
	int widgetWidth  = width();
	int widgetHeight = height();

	paintDefaultBackground(&pa);
	pa.setPen(m_pSysTray->getForeground());

	KviStr timeStr;
	KviStr dateStr;
	getConnectionTime(timeStr);
	if( m_iConnectionStartTime == 0 )
		dateStr = _i18n_("Not connected");
	else {
		struct tm *t = localtime(&(m_iConnectionStartTime));
		dateStr.sprintf(
			_i18n_("Started at %d%d:%d%d:%d%d"),
			t->tm_hour / 10, t->tm_hour % 10, t->tm_min / 10, t->tm_min % 10, t->tm_sec / 10, t->tm_sec % 10
		);
		if( t->tm_isdst > 0 )
			dateStr += _i18n_(" (DST)");
	}

	if( widgetHeight > 32 ) {
		// Big mode
		g_pOptions->m_fntSysTray.setPointSize(16);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, widgetHeight - 6, timeStr.ptr(), timeStr.len());
		g_pOptions->m_fntSysTray.setPointSize(10);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, 14, dateStr.ptr(), dateStr.len());
	} else {
		// Small mode
		timeStr.append(" - ");
		timeStr.append(dateStr);
		g_pOptions->m_fntSysTray.setPointSize(14);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, (widgetHeight + 12) >> 1, timeStr.ptr(), timeStr.len());
	}

	paintDefaultFrame(&pa);

	bitBlt(this, 0, 0, m_pSysTray->getMemoryBuffer(), 0, 0, widgetWidth, widgetHeight);
}

#include "m_kvi_systray_onlinetimer.moc"
