/**
 * @file libleakbug/leakbug.c Leakbug
 *
 * $Id: leakbug.c,v 1.14 2002/06/04 16:55:31 chipx86 Exp $
 *
 * @Copyright (C) 2001-2002 The GNUpdate Project. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <stdio.h>

#define LEAKBUG_NO_DEBUG

#include <leakbug.h>
#include <stdarg.h>
#include "config.h"

typedef struct _MemSlot MemSlot;

struct _MemSlot
{
	unsigned long address;
	size_t size;
	void *ptr;

	char *filename;
	int line;

	MemSlot *prev;
	MemSlot *next;
};

typedef enum
{
	LB_TEXT = 0,
	LB_WARNING,
	LB_ERROR,
	LB_FATAL

} LbPrintType;

static MemSlot *firstMemSlot = NULL;
static MemSlot *lastMemSlot  = NULL;

static FILE *outstream = NULL;
static char useLogfile = 0;

#ifdef HAVE_ATEXIT
# define EXIT_FUNC(proc) static void (proc)(void)
#else
# ifdef HAVE_ON_EXIT
#  define EXIT_FUNC(proc) static void (proc)(int i, void *v)
# else
#  define EXIT_FUNC(proc) static void (proc)(void)
# endif
#endif

EXIT_FUNC(__uninitialize)
{
	if (outstream != NULL && useLogfile == 1)
	{
		fclose(outstream);

		outstream = NULL;
		useLogfile = 0;
	}
}

static MemSlot *
__newMemSlot(void)
{
	MemSlot *slot = (MemSlot *)malloc(sizeof(MemSlot));

	slot->address  = 0;
	slot->size     = 0;
	slot->line     = 0;
	slot->filename = NULL;
	slot->next     = NULL;
	slot->prev     = NULL;

	return slot;
}

static void
__freeMemSlot(MemSlot *slot)
{
	free(slot->filename);
	free(slot);
}

static void
__addMemSlot(MemSlot *slot)
{
	if (firstMemSlot == NULL)
		firstMemSlot = slot;

	if (lastMemSlot == NULL)
		slot->prev = NULL;
	else
	{
		slot->prev = lastMemSlot;
		lastMemSlot->next = slot;
	}

	lastMemSlot = slot;
}

static void
__removeMemSlot(MemSlot *slot)
{
	if (slot->prev != NULL)
		slot->prev->next = slot->next;
	else
		firstMemSlot = slot->next;

	if (slot->next != NULL)
		slot->next->prev = slot->prev;
	else
		lastMemSlot = slot->prev;
}

static void
lbPrint(LbPrintType type, const char *msg, ...)
{
	va_list args;

	if (outstream == NULL)
	{
		const char *logfile = getenv("LB_LOGFILE");

		if (logfile == NULL || *logfile == '\0')
			outstream = stderr;
		else
		{
			useLogfile = 1;
			outstream = fopen(logfile, "w");

			if (outstream == NULL)
			{
				outstream = stderr;
			}
			else
			{
#ifdef HAVE_ATEXIT
				atexit(__uninitialize);
#else
# ifdef HAVE_ON_EXIT
				on_exit(__uninitialize, NULL);
# endif /* HAVE_ON_EXIT */
#endif /* HAVE_ATEXIT */
			}
		}
	}

	va_start(args, msg);

	if (type == LB_TEXT)
	{
		fprintf(outstream, "<LB> ");
	}
	else
	{
		fprintf(outstream, "<LB> %s: ",
				(type   == LB_WARNING ? "WARNING" :
				 (type  == LB_ERROR   ? "ERROR"   :
				  (type == LB_FATAL   ? "FATAL" : "UNKNOWN"))));
	}

	vfprintf(outstream, msg, args);
	va_end(args);
}

void *
lbMalloc(size_t size, const char *filename, int line, int debugLevel)
{
	MemSlot *slot;

	slot = __newMemSlot();

	slot->ptr = malloc(size);
	
	if (slot->ptr == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Unable to allocate memory of "
					"%d bytes at 5s, line %d\n", size, filename, line);

		return NULL;
	}

	slot->address  = (unsigned long)slot->ptr;
	slot->size     = size;
	slot->line     = line;
	slot->filename = strdup(filename);

	__addMemSlot(slot);

	return slot->ptr;
}

void *
lbCalloc(size_t nmemb, size_t size, const char *filename, int line,
		 int debugLevel)
{
	MemSlot *slot = __newMemSlot();

	slot->ptr = calloc(nmemb, size);

	if (slot->ptr == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Unable to allocate memory of"
					" %d*%d (%d) bytes at %s, line %d\n", nmemb, size,
					nmemb * size, filename, line);

		return NULL;
	}

	slot->address  = (unsigned long)slot->ptr;
	slot->size     = nmemb*size;
	slot->line     = line;
	slot->filename = strdup(filename);

	__addMemSlot(slot);

	return slot->ptr;
}

void *
lbRealloc(void *data, size_t size, const char *filename, int line,
		  int debugLevel)
{
	MemSlot *slot;
	int newSlot = 0;

	if (data == NULL)
		return lbMalloc(size, filename, line, debugLevel);

	if (size == 0)
	{
		lbFree(data, filename, line, debugLevel);
		return NULL;
	}

	/* Find the memslot of this data. */
	for (slot = firstMemSlot; slot != NULL; slot = slot->next)
	{
		if (data == slot->ptr)
			break;
	}

	if (slot == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Memory address passed to realloc "
					"unknown at %s, line %d\n", filename, line);

		slot = __newMemSlot();
		newSlot = 1;
	}

	slot->ptr = realloc(data, size);

	if (slot->ptr == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Unable to allocate memory of "
					"%d bytes at %s, line %d\n", size, filename, line);

		return NULL;
	}

	slot->address  = (unsigned long)slot->ptr;
	slot->size     = size;
	slot->line     = line;
	slot->filename = strdup(filename);

	if (newSlot == 1)
		__addMemSlot(slot);

	return slot->ptr;
}

void
lbFree(void *ptr, const char *filename, int line, int debugLevel)
{
	MemSlot *slot;

	if (ptr == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Freeing already freed memory at"
					" %s, line %d\n", filename, line);

		return;
	}

	for (slot = firstMemSlot; slot != NULL; slot = slot->next)
	{
		if (slot->ptr == ptr)
			break;
	}

	if (slot == NULL)
	{
		if (debugLevel & LEAKBUG_WARNINGS)
			lbPrint(LB_WARNING, "Freeing unknown memory (%p) at "
					"%s, line %d\n", ptr, filename, line);
		return;
	}

	free(ptr);

	__removeMemSlot(slot);
	__freeMemSlot(slot);
}

void *
lbRegister(void *ptr, unsigned int dataSize, const char *filename,
		   int line, int debugLevel)
{
	MemSlot *slot;

	if (ptr == NULL || dataSize == 0)
		return NULL;

	slot = __newMemSlot();

	slot->filename = strdup(filename);
	slot->line     = line;
	slot->ptr      = ptr;
	slot->address  = (unsigned long)slot->ptr;
	slot->size     = dataSize;

	__addMemSlot(slot);

	return ptr;
}

void **
lbRegisterArray(void **ptr, unsigned int dataSize,
				unsigned int numElements, const char *filename,
				int line, int debugLevel)
{
	int i;
	
	if (ptr == NULL)
		return NULL;
	
	lbRegister(ptr, sizeof(void *), filename, line, debugLevel);

	for (i = 0; i < numElements; i++)
		lbRegister(ptr[i], dataSize, filename, line, debugLevel);

	return ptr;
}

void
lbFreeLeaks(void)
{
	MemSlot *slot, *nextSlot;

	lbPrint(LB_WARNING, "Automatically freeing leaked memory!\n");

	for (slot = firstMemSlot; slot != NULL; slot = nextSlot)
	{
		nextSlot = slot->next;

		free(slot->ptr);
		__freeMemSlot(slot);
	}
}

void
lbDumpLeaks(void)
{
	MemSlot *slot;

	lbPrint(LB_TEXT, "Displaying leaked memory:\n");

	if (firstMemSlot == NULL)
	{
		lbPrint(LB_TEXT, "None.\n");
		return;
	}

	for (slot = firstMemSlot; slot != NULL; slot = slot->next)
	{
		lbPrint(LB_TEXT, "%s:%d - %p (%d bytes)\n",
				slot->filename, slot->line, slot->ptr, slot->size);
	}
}

