/*
 * dvb_ca.c: generic DVB CA functions
 *
 * Copyright (C) 2004 Andrew de Quincey
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

/*
 * In the following, public functions always do locking of slots, private functions do not
 * necessarily do locking of slots - they may expect the caller to have locked */

#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/module.h>
#include <asm/semaphore.h>

#include "dvb_ca.h"
#include "dvb_functions.h"

static int dvb_ca_debug = 0;
#define dprintk if (dvb_ca_debug) printk

#define HOST_LINK_BUF_SIZE 0x200

#define RX_BUFFER_SIZE 65540
#define TX_BUFFER_SIZE 65540

#define CTRLIF_DATA 0
#define CTRLIF_COMMAND 1
#define CTRLIF_STATUS 1
#define CTRLIF_SIZE_LOW 2
#define CTRLIF_SIZE_HIGH 3

#define CMDREG_HC 1 /* Host control */
#define CMDREG_SW 2 /* Size write */
#define CMDREG_SR 4 /* Size read */
#define CMDREG_RS 8 /* Reset interface */

#define STATUSREG_RE 1 /* read error */
#define STATUSREG_WE 2 /* write error */
#define STATUSREG_FR 0x40 /* module free */
#define STATUSREG_DA 0x80 /* data available */
#define STATUSREG_TXERR (STATUSREG_RE|STATUSREG_WE) /* general transfer error */


static int dvb_ca_en50221_link_init(struct dvb_ca* ca, int slot);


static u8* findstr(u8* haystack, int hlen, u8* needle, int nlen) {
        int i;

        for(i=0; i<= hlen - nlen; i++) {
                  if (!strncmp(haystack+i, needle, nlen)) return haystack+i;
        }

        return NULL;
}


/* ******************************************************************************** */
/* Modified version of wait_event_interruptible so we can check for errors */

#define __dvb_ca_wait_event_interruptible(wq, condition, ret)                 \
do {                                                                          \
        wait_queue_t __wait;                                                  \
        init_waitqueue_entry(&__wait, current);                               \
                                                                              \
        add_wait_queue(&wq, &__wait);                                         \
        for (;;) {                                                            \
                set_current_state(TASK_INTERRUPTIBLE);                        \
                if (ret = condition)                                          \
                        break;                                                \
                if (!signal_pending(current)) {                               \
                        schedule();                                           \
                        continue;                                             \
                }                                                             \
                ret = -ERESTARTSYS;                                           \
                break;                                                        \
        }                                                                     \
        current->state = TASK_RUNNING;                                        \
        remove_wait_queue(&wq, &__wait);                                      \
} while (0)

#define dvb_ca_wait_event_interruptible(wq, condition)                        \
({                                                                            \
        int __ret = 0;                                                        \
        if (!(condition))                                                     \
                __dvb_ca_wait_event_interruptible(wq, condition, __ret);      \
        __ret;                                                                \
})


/* ******************************************************************************** */
/* Functions for controlling access to slots */

/**
 * Safely increment the usage counter for a CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
int dvb_ca_slot_acquire(struct dvb_ca* ca, int slot) {
        int status;

        if (status = down_interruptible(&ca->slot_info[slot].sem)) return status;

        if (!ca->slot_info[slot].cam_present) {
                up(&ca->slot_info[slot].sem);
                return -EIO;
        }
        ca->slot_info[slot].usage_counter++;

        up(&ca->slot_info[slot].sem);
        return 0;
}


/**
 * Safely decrement the usage counter for a CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
int dvb_ca_slot_release(struct dvb_ca* ca, int slot) {
        int status;

        if (status = down_interruptible(&ca->slot_info[slot].sem)) return status;

        ca->slot_info[slot].usage_counter--;

        up(&ca->slot_info[slot].sem);
        return(0);
}


/**
 * Acquire a slot exclusively. The slot semaphore will be left locked on successful
 * exit of this function.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
int dvb_ca_slot_acquire_exclusive(struct dvb_ca* ca, int slot) {
        int status;

        while(1) {
                /* lock the slot */
                if (status = down_interruptible(&ca->slot_info[slot].sem)) return status;

                /* if there is no CAM, exit straight away */
                if (!ca->slot_info[slot].cam_present) {
                        up(&ca->slot_info[slot].sem);
                        return -EIO;
                }

                /* if there are no other users of the slot, we've finished */
                if (ca->slot_info[slot].usage_counter == 0) {
                        ca->slot_info[slot].usage_counter++;
                        break;
                }

                /* sleep a bit and try again */
                up(&ca->slot_info[slot].sem);
                dvb_delay(1);
        }

        /* DO NOT UNLOCK! */
        return 0;
}


/**
 * Release an exclusively owned slot. The slot semaphore will be unlocked by this function.
 *
 * @param ca CA instance.
 * @param slot Slot concerned.
 *
 * @return 0 on success, <0 on failure.
 * */
int dvb_ca_slot_release_exclusive(struct dvb_ca* ca, int slot) {
        ca->slot_info[slot].usage_counter--;
        up(&ca->slot_info[slot].sem);
        return(0);
}





/* ******************************************************************************** */
/* Functions for link level connection_ids */

/**
 * Internal function to destroy and unlink a dvb_ca_connection structure.
 *
 * @param cacon The structure to destroy.
 * @param unlink If 1, it will be unlinked from the list.
 */
static void dvb_ca_connection_destroy(struct dvb_ca_connection* cacon, int unlink) {
        /* unlink it from the list. */
        if (unlink)
                list_del(&cacon->connection);

        /* destroy buffers etc */
        if (cacon->rx_buffer.data) vfree(cacon->rx_buffer.data);
        if (cacon->tx_buffer.data) vfree(cacon->tx_buffer.data);
        vfree(cacon);
}



/**
 * Get or create a dvb_ca_connection structure for a particular slot/connection_id.
 *
 * @param ca CA device instance.
 * @param slot Slot id.
 * @param connection_id Connection id to retrieve structure for.
 * @param create If 1, and a connection struct was not found, a new one will be created.
 *
 * @return A dvb_ca_connection structure, or NULL.
 */
static struct dvb_ca_connection* dvb_ca_connection_get(struct dvb_ca* ca, int slot, int connection_id, int create) {
        struct dvb_ca_connection* cacon;
        u8* mem;

        /* is there already a record for this connection_id */
        list_for_each_entry(cacon, &ca->slot_info[slot].connections, next) {
                /* if we found it, return it immediately */
                if (cacon->connection_id == connection_id) {
                        cacon->last_used = jiffies;
                        return cacon;
                }
        }

        /* if we get here, and we've been asked not to create a structure, just return NULL */
        if (!create) {
                return NULL;
        }

        /* did not find it => create a new connection */
        cacon = vmalloc(sizeof(struct dvb_ca_connection));
        if (cacon == NULL) {
                goto error;
        }

        /* setup the structure */
        memset(cacon, 0, sizeof(struct dvb_ca_connection));
        cacon->connection_id = connection_id;
        cacon->rx_partial_pkt = -1;
        cacon->tx_partial_pkt_size = -1;
        cacon->last_used = jiffies;

        if (!(mem = vmalloc(RX_BUFFER_SIZE))) {
                goto error;
        }
        dvb_ringbuffer_init(&cacon->rx_buffer, mem, RX_BUFFER_SIZE);

        if (!(mem = vmalloc(TX_BUFFER_SIZE))) {
                goto error;
        }
        dvb_ringbuffer_init(&cacon->tx_buffer, mem, TX_BUFFER_SIZE);

        /* success */
        list_add_tail(&cacon->next, &ca->slot_info[slot].connections);
        return cacon;

error:
        if (cacon != NULL) {
                dvb_ca_connection_destroy(cacon, 0);
        }
        return NULL;
}





/* ******************************************************************************** */
/* EN50221 physical interface functions */


/**
 * Wait for flags to become set on the STATUS register on a CAM interface,
 * checking for errors and timeout.
 *
 * @param ca CA instance.
 * @param slot Slot on interface.
 * @param waitfor Flags to wait for.
 * @param timeout_ms Timeout in milliseconds.
 *
 * @return 0 on success, nonzero on error.
 */
static int dvb_ca_en50221_wait_if_status(struct dvb_ca* ca, int slot, u8 waitfor, int timeout_ms) {
        unsigned long timeout;

        /* loop until timeout elapsed */
        timeout = jiffies + ((HZ * timeout_ms) / 1000);
        while(1) {
                /* read the status and check for error */
                int res = ca->read_cam_control(ca, slot, CTRLIF_STATUS);
                if (res < 0) return -EIO;

                /* if we got the flags, it was successful! */
                if (res & waitfor) {
                        return 0;
                }

                /* check for timeout */
                if (time_after(jiffies,timeout)) {
                        break;
                }

                /* wait for a bit */
                mdelay(100); // FIXME: is this timeout good? need to investigate
        }

        /* if we get here, we've timed out */
        return -ETIMEDOUT;
}



/**
 * Reset the CAM control interface.
 *
 * @param ca CA instance.
 * @param slot Slot id.
 *
 * @return 0 on success, nonzero on failure.
 */
static int dvb_ca_en50221_reset_if(struct dvb_ca* ca, int slot) {
        int ret;

        /* reset the interface and wait for FR to be set */
        if (ret = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, CMDREG_RS)) return ret;
        if (ret = dvb_ca_cam_wait_if_status(ca, slot, STATUSREG_FR, 2)) return ret;

        /* need to perform link initialisation again */
        return dvb_ca_en50221_link_init(ca, slot);
}


/**
 * Initialise the link layer connection to a CAM.
 *
 * @param ca CA instance.
 * @param slot Slot id.
 *
 * @return 0 on success, nonzero on failure.
 */
static int dvb_ca_en50221_link_init(struct dvb_ca* ca, int slot) {
        int ret;
        int buf_size;
        u8 buf[2];

        /* reset the interface */
        if (ret = dvb_ca_en50221_reset_if(ca, slot)) return ret;

        /* set the host link buffer size temporarily. it will be overwritten with the
         * real negotiated size later. */
        ca->slot_info[i]->link_buf_size = 2;

        /* OK, setup to read the buffer size */
        if (ret = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, CMDREG_SR)) return ret;
        if (ret = dvb_ca_en50221_wait_if_status(ca, slot, STATUSREG_DA, 2)) return ret;

        /* read the buffer size from the CAM */
        if ((ret = dvb_ca_en50221_read_data(ca, slot, buf, 2)) != 2) return ret;

        /* set everything back to 0 again */
        if (ret = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, 0)) return ret;

        /* store it, and choose the minimum of our buffer and the CAM's buffer size */
        bufsize = (buf[0] << 8) | buf[1];
        if (bufsize > HOST_LINK_BUF_SIZE) bufsize = HOST_LINK_BUF_SIZE;
        ca->slot_info[i]->link_buf_size = bufsize;

        /* setup the buffer size in the link_buf */
        buf[0] = bufsize >> 8;
        buf[1] = bufsize & 0xff;

        /* OK, setup to write chosen buffer size to CAM */
        if (ret = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, CMDREG_SW)) return ret;
        if (ret = dvb_ca_en50221_wait_if_status(ca, slot, STATUSREG_FR, 2)) return ret;

        /* write the buffer size to the CAM */
        if ((ret = dvb_ca_en50221_write_data(ca, slot, buf, 2)) != 2) return ret;

        /* set everything back to 0 again */
        if (ret = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, 0)) return ret;

        /* success */
        return 0;
}



/**
 * Parse attribute memory of a CAM module, extracting Config register, and checking
 * it is a DVB CAM module.
 *
 * @param ca CA instance.
 * @param slot Slot id.
 *
 * @return 0 on success, <0 on failure.
 */
static int dvb_ca_en50221_parse_attributes(struct dvb_ca* ca, int slot) {
        int address = 0;
        int tupleLength;
        int tupleType;
        char tuple[257];
        char* dvb_str;
        int i;
        int rasz;
        int got_device0a = 0;
        int got_device0c = 0;
        int got_vers1 = 0;
        int got_manfid = 0;
        int got_config = 0;
        int got_cftableentry = 0;
        int end_chain = 0;


        while(!end_chain) {
                /* grab the next tuple length and type */
                if ((tupleType = ca->read_attribute_mem(ca, slot, address++)) < 0) return tupleType;
                if ((tupleLength = ca->read_attribute_mem(ca, slot, address++)) < 0) return tupleLength;

                /* read in the whole tuple */
                for(i=0; i< tupleLength; i++) {
                        tuple[i] = ca->read_attribute_mem(ca, slot, address+i);
                }
                address += tupleLength;

                /* deal with the types of tuple */
                switch(tupleType) {
                case 0x1D: // CISTPL_DEVICE_0A
                        got_device0a = 1;
                        break; /* don't bother parsing */

                case 0x1C: // CISTPL_DEVICE_0C
                        got_device0c = 1;
                        break; /* don't bother parsng */

                case 0x15: // CISTPL_VERS_1
                        got_vers1 = 1;
                        break; /* don't bother parsing */

                case 0x20: // CISTPL_MANFID
                        /* check tuple length is valid */
                        if (tupleLength != 4) break;

                        /* OK, extract manid and devid */
                        ca->slot_info[i].manid = (tuple[1] << 8) | tuple[0];
                        ca->slot_info[i].devid = (tuple[3] << 8) | tuple[2];

                        got_manfid = 1;
                        break;

                case 0x1A: // CISTPL_CONFIG
                        /* check tuple length is valid */
                        if (tupleLength < 3) break;

                        /* OK, grab length of configbase */
                        rasz = tuple[0] & 3;

                        /* check there is enough space for everything */
                        if (tupleLength < (3 + rasz + 14)) break;

                        /* extract the configbase */
                        ca->slot_info[i].config_base = 0;
                        for(i=0; i< rasz+1; i++) {
                                ca->slot_info[i].config_base |= (tuple[2+i] << (8*i));
                        }

                        /* check it contains the correct DVB string */
                        dvb_str = findstr(tuple, tupleLength, "DVB_CI_V", 8);
                        if (dvb_str == NULL) break;
                        if (tupleLength < ((dvb_str - tuple) + 12)) break;

                        /* is it a version we support? */
                        if (strncmp(dvb_str + 8, "1.00", 4)) {
                                printk("dvb_ca: Unsupported DVB CAM module version %c%c%c%c\n",
                                       dvb_str[8], dvb_str[9], dvb_str[10], dvb_str[11]);
                                return -EINVAL;
                        }

                        got_config = 1;
                        break;

                case 0x1D: // CISTPL_CFTABLE_ENTRY
                        if (tupleLength < (2+11+17)) break;

                        /* get the config option */
                        ca->slot_info[i].config_option = tuple[0] & 0x3f;

                        /* does it contain an interrupt descriptor structure? */
                        i = 1;
                        if (tuple[0] & 0x80) i++;
                        if (tuple[i] & 0x10) {
                                ca->slot_info[i].irq_supported = 1;
                        } else
                                ca->slot_info[i].irq_supported = 0;
                        }

                        /* OK, check it contains the correct strings */
                        if ((findstr(tuple, tupleLength, "DVB_HOST", 8) == NULL) ||
                            (findstr(tuple, tupleLength, "DVB_CI_MODULE", 13) == NULL)) break;

                        got_cftableentry = 1;
                        break;

                case 0xFF: // end of chain
                        end_chain = true;
                        break;

                default: /* Unknown tuple type */
                        /* just skip this tuple and move to the next one */
                        break;
                }
        }

        // OK, did we get everything we were looking for?
        if (!got_device0a || !got_device0c || !got_vers1 || !got_manfid || !got_config || !got_cftablenentry) {
                printk("dvb_ca: Non-CAM PC card detected\n");
                return -EINVAL;
        }

        // success!
        return 0;
}



/**
 * Initialise a CAM module.
 *
 * @param ca CA instance.
 * @param slot Slot containing the CAM.
 */
static void dvb_ca_en50221_init(struct dvb_ca* ca, int slot) {
        int tmp;

        /* reset the CAM module */
        tmp = ca->read_attribute_mem(ca, slot, ca->slot_info[slot].config_base);
        ca->write_attribute_mem(ca, slot, ca->slot_info[slot].config_base, tmp | 0x80);
        dvb_delay(1);
        ca->write_attribute_mem(ca, slot, ca->slot_info[slot].config_base, tmp);
        dvb_delay(1);

        /* set the config option */
        tmp = (tmp & 0x3f) | ca->slot_info[slot].config_option;
        ca->write_attribute_mem(ca, slot, ca->slot_info[slot].config_base, tmp);
}


/**
 * This function talks to an EN50221 CAM control interface. It reads a buffer of
 * data from the CAM. The data can either be stored in a supplied buffer, or
 * automatically be added to the correct place in the CA slot/connection buffer structures.
 *
 * @param ca CA instance.
 * @param slot Slot to read from.
 * @param buf If non-NULL, the data will be written to this buffer. If NULL,
 * the data will be added into the buffering system as a normal fragment.
 * @param count Size of buf. Ignored if buf is NULL.
 *
 * @return Number of bytes read, or < 0 on error
 */
static int dvb_ca_en50221_read_data(struct dvb_ca* ca, int slot, u8* buf, int count) {
        int bytes_read;
        int status;
        u8 buf[HOST_LINK_BUF_SIZE];
        u8 last_fragment;
        u8 connection_id;

        /* if the CA device is not open, do not attempt to receive data from it */
        if (!ca->open) return -EIO;

        /* acquire the slot */
        if (status = dvb_ca_slot_acquire(ca, slot)) return status;

        /* reset the interface if there's been a tx error */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (status & STATUSREG_TXERR) dvb_ca_en50221_reset_if(ca, slot);

        /* check if data is available */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (!(status & STATUSREG_DA)) {
                /* no data */
                status = 0;
                goto exit;
        }

        /* read the amount of data */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_SIZE_HIGH)) < 0) goto exit;
        bytes_read = status << 8;
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_SIZE_LOW)) < 0) goto exit;
        bytes_read |= status;

        /* check it will fit */
        if (bytes_read > ca->slot_info[i].link_buf_size) {
                printk("dvb_ca: CAM tried to send a buffer larger than the link buffer size!\n");
                status = -EIO;
                goto exit;
        }

        /* fill the buffer */
        for(i=0; i < bytes_read; i++) {
                /* read byte and check */
                if ((status = ca->read_cam_control(ca, slot, CTRLIF_DATA)) < 0) goto exit;

                /* OK, store it in the buffer */
                buf[i] = res;

                /* at this point, RE=1, DA=0 during the transfer */
        }

        /* check for read error (RE should now go to 0) */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (status & STATUSREG_RE) {
                status = -EIO;
                goto exit;
        }

        /* check there are at least two bytes received! */
        if (bytes_read < 2) {
                printk("dvb_ca: CAM sent a buffer that was less than 2 bytes!\n");
                status = -EIO;
                goto exit;
        }

        /* extract information about this buffer */
        connection_id = buf[0];
        last_fragment = (buf[1] & 0x80) ? 0 : 1; // received bit clear => last fragment

        /* OK, add it to the receive buffer */
        if ((status = dvb_ca_buf_copy_from_cam(ca, buf+2, bytes_read-2, last_fragment, slot, connection_id)) < 0) {
                goto exit;
        }

        /* return the number of bytes read */
        status = bytes_read;

    exit:
        ca->write_cam_control(ca, slot, CTRLIF_COMMAND, 0);
        dvb_ca_slot_release(ca, slot);
        return status;
}


/**
 * This function talks to an EN50221 CAM control interface. It writes a buffer of data
 * to a CAM. The data to write will can either come from a supplied buffer, or from the
 * buffering system identified by the slot/connection_id.
 *
 * @param ca CA instance.
 * @param slot Slot to read from.
 * @param connection_id Connection id to write data from. Ignored if buf is non-NULL.
 * @param buf If non-NULL, the data in this buffer is treated as a complete link-level packet to
 * be written. If NULL, the data will be extracted from the buffering system for the slot/connection_id.
 * @param count Size of buf. Ignored if buf is NULL.
 *
 * @return Number of bytes written, or < 0 on error.
 */
static int dvb_ca_en50221_write_data(struct dvb_ca* ca, int slot, int connection_id, u8* buf, int count) {
        int status;
        u8 buf[HOST_LINK_BUF_SIZE];
        u8 last_fragment;
        int bytes_write;


        /* if the CA device is not open, do not attempt to send data to it */
        if (!ca->open) return -EIO;

        /* acquire the slot */
        if (status = dvb_ca_slot_acquire(ca, slot)) return status;

        /* reset the interface if there's been a tx error */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (status & STATUSREG_TXERR) dvb_cam_reset_if(ca, slot);

        /* check if interface does not have data available */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (status & STATUSREG_DA) {
                status = -EAGAIN;
                goto exit;
        }

        /* OK, set HC bit */
        if (status = ca->write_cam_control(ca, slot, CTRLIF_COMMAND, CMDREG_HC)) goto exit;

        /* check if interface is free */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (!(status & STATUSREG_FR)) {
                /* it wasn't free => try again later */
                res = -EAGAIN;
                goto exit;
        }

        /* grab some data to send */
        bytes_write = dvb_ca_buf_copy_to_cam(ca, buf + 2, ca->slot_info[i].link_buf_size - 2, &last_fragment, slot, connection_id);
        if (bytes_write <= 0) {
                res = bytes_write;
                goto exit;
        }

        /* setup the buffer header */
        buf[0] = connection_id;
        buf[1] = last_fragment ? 0 : 0x80;
        bytes_write += 2; // for the header

        /* send the amount of data */
        if (status = ca->write_cam_control(ca, slot, CTRLIF_SIZE_HIGH, bytes_write >> 8)) goto exit;
        if (status = ca->write_cam_control(ca, slot, CTRLIF_SIZE_LOW, bytes_write & 0xff)) goto exit;

        /* send the buffer */
        for(i=0; i < bytes_read; i++) {
                if (status = ca->write_cam_control(ca, slot, CTRLIF_DATA, ca->slot_info[slot].link_buf[i])) goto exit;

                /* At this point, WE=1, FR=0 during the transfer */
        }

        /* check for write error (WE should now be 0) */
        if ((status = ca->read_cam_control(ca, slot, CTRLIF_STATUS)) < 0) goto exit;
        if (status & STATUSREG_WE) {
                res = -EIO;
                goto exit;
        }

        /* return the number of bytes read */
        status = bytes_write;

exit:
        ca->write_cam_control(ca, slot, CTRLIF_COMMAND, 0);
        dvb_ca_slot_release(ca, slot);
        return status;
}



/**
 * A CAM has been inserted => initialise it.
 *
 * @param ca CA instance.
 * @param slot Slot containing the CAM to initialise.
 */
static int dvb_ca_en50221_slot_init(struct dvb_ca* ca, int slot) {
        int status;

        dprintk ("%s\n", __FUNCTION__);

        /* acquire the slot exclusively */
        if (status = dvb_ca_slot_acquire_exclusive(ca, slot)) return status;

        /* reset slot */
        ca->slot_reset(ca, slot);

        /* check it is a DVB cam and get hardware info */
        if (status = dvb_ca_en50221_parse_attributes(ca, slot)) {
                dvb_ca_slot_release_exclusive(ca, slot);
                return status;
        }

        /* setup CAM hardware */
        if (status = dvb_ca_en50221_init(ca, slot)) {
                dvb_ca_slot_release_exclusive(ca, slot);
                return status;
        }

        /* perform CAM link protocol init */
        if (status = dvb_ca_en50221_link_init(ca, slot)) {
                dvb_ca_slot_release_exclusive(ca, slot);
                return status;
        }

        /* OK, slot is now active */
        INIT_LIST_HEAD(&ca->slot[info].connections);
        init_MUTEX(&ca->slot_info[slot].task_sem);
        ca->slot_info[slot].tasks = 0;
        ca->slot_info[slot].cam_present = 1;

        /* success */
        dvb_ca_slot_release_exclusive(ca, slot);
        return 0;
}


/**
 * A CAM has been removed => shut it down.
 *
 * @param ca CA instance.
 * @param slot Slot to shut down.
 */
static int dvb_ca_en50221_slot_shutdown(struct dvb_ca* ca, int slot) {
        struct dvb_ca_connection* cacon;
        struct dvb_ca_connection* tmp;
        int status;

        dprintk ("%s\n", __FUNCTION__);

        /* acquire the slot exclusively */
        if (status = dvb_ca_slot_acquire_exclusive(ca, slot)) return status;

        /* there is no longer a CAM present in this slot */
        ca->slot_info[slot].cam_present = 0;

        /* bypass the TS feed to that slot, then reset it */
        ca->slot_ts_bypass(ca, slot, 1);
        ca->slot_reset(ca, slot);

        /* flush the buffers on this slot */
        /* cannot use dvb_ca_buf_flush_buffers() here because it locks exclusively as well */
        list_for_each_entry_safe(cacon, tmp, &ca->slot_info[slot].connections, next) {
                dvb_ca_connection_destroy(cacon, 1);
        }

        /* finished modifying slot structures */
        dvb_ca_slot_release_exclusive(ca, slot);

        /* need to wake up all read and write processes to check if they're now
           trying to write to a defunct CAM */
        wake_up_interruptible(&ca->read_queue);
        wake_up_interruptible(&ca->write_queue);

        /* success */
        return 0;
}


/* ******************************************************************************** */
/* Monitoring thread */


/**
 * Kernel thread which monitors CA slots for CAM changes, and performs data transfers.
 */
static int dvb_ca_thread(void* data) {
        struct dvb_ca *ca = (struct dvb_ca*) data;
        struct dvb_ca_connection* cacon;
        struct dvb_ca_connection* tmp;
        char name[15];
        int slot;
        unsigned long scan_connections = jiffies + (60*HZ);

        dprintk ("%s\n", __FUNCTION__);

        /* setup kernel thread */
        snprintf(name, sizeof(name), "kdvb-ca");
        dvb_kernel_thread_setup(name);

        /* main loop */
        while(!ca->exit) {
                up (&ca->thread_sem);      /* is locked when we enter the thread... */

                /* sleep for a bit */
                interruptible_sleep_on_timeout(&ca->thread_queue, HZ/10);
                if (signal_pending(current)) break;
                if (down_interruptible(&ca->thread_sem)) break;
                if (ca->exit) break;

                /* go through all the slots looking for CAM changes */
                for(slot=0; slot < ca->slot_count; slot++) {

                        /* poll for CAM changes if there aren't any IRQs for that
                         * OR: if we've had a CAM_CHANGE task set */
                        if ((!(ca->flags & DVB_CA_FLAG_EN50221_IRQ_CAMCHANGE)) ||
                            (ca->slot_info[slot].tasks & DVB_CA_TASK_CAM_CHANGED)) {

                                /* determine the hardware status of the CAM */
                                int slot_status = ca->slot_status(ca, slot);
                                int cam_present = (slot_status & DVB_CA_SLOT_CAM_PRESENT) ? 1: 0;
                                int cam_changed = (slot_status & DVB_CA_SLOT_CAM_CHANGED) ? 1: 0;
                                if (!cam_changed) {
                                        cam_changed = (cam_present != ca->slot_info[slot].cam_present);
                                }

                                /* if cam status has changed => update */
                                if (cam_changed) {
                                        if (cam_present) dvb_ca_en50221_cam_slot_init(ca, slot);
                                        else dvb_ca_en50221_cam_slot_shutdown(ca, slot);
                                }
                        }

                        /* do any reading and writing necessary */
                        if (ca->open && (ca->slot_info[slot].cam_present)) {
                                // FIXME: sort it out!

                                /* if the CAM or the CI interface itself doesn't support IRQs, need to poll for reads */
                                if ((!ca->slot_info[slot].irq_supported) || (!ca->flags & DVB_CA_FLAG_EN50221_IRQ_READ)) {
                                        /* if there is data to be read, read it! */
                                        while(dvb_ca_cam_read_data(ca, slot) > 0);
                                }
                        }

                        /* tidy up any defunct connection structures every 60 seconds */
                        if ((ca->slot_info[slot].cam_present) && (time_after(jiffies, scan_connections))) {

                                if (status = dvb_ca_slot_acquire_exclusive(ca, slot)) break;

                                list_for_each_entry_safe(cacon, tmp, &ca->slot_info[slot].connections, next) {

                                        /* tear down a connection structure if it hasn't been used for 120 seconds */
                                        unsigned long timeout = cacon->last_used + (120 * HZ);
                                        if (time_after(jiffies, timeout)) {
                                                dvb_ca_connection_destroy(cacon, 1);
                                        }
                                }

                                dvb_ca_slot_release_exclusive(ca, slot);
                        }
                }
        }

        up(&ca->thread_sem);
        mb();

        /* completed */
        wake_up_interruptible (&ca->thread_queue);
        return(0);
}




/* ******************************************************************************** */
/* Packet buffer functions */


/**
 * Flush buffers associated with a particular CA slot.
 *
 * @param ca CA instance.
 * @param slot Slot whose buffers are to be to flushed.
 *
 * @return 0 on success, <0 on failure.
 */
int dvb_ca_buf_flush_buffers(struct dvb_ca* ca, int slot) {
        struct dvb_ca_connection* cacon;
        struct dvb_ca_connection* tmp;
        int status;

        if (status = dvb_ca_slot_acquire_exclusive(ca, slot)) return status;

        /* just destroy all the connections for this slot */
        list_for_each_entry_safe(cacon, tmp, &ca->slot_info[slot].connections, next) {
                dvb_ca_connection_destroy(cacon, 1);
        }

        dvb_ca_slot_release_exclusive(ca, slot);
        return 0;
}



/**
 * This function does not talk to a CAM; it is a utility function for grabbing data from the
 * connection buffers in chunks suitable for use by hardware-dependent code.
 *
 * Copy data to be sent to the CAM from the cache into the
 * supplied buffer. This call is for lower layers to get data still to
 * be sent to the CAM. The buffer is filled by the layers which receive data
 * from userspace.
 *
 * @param ca CA instance.
 * @param dest Where to put the data.
 * @param len Size of destination buffer.
 * @param last_fragment. This will be set to 1 if this is the last fragment of a packet,
 *                       or 0 if there is still more data in this packet to be transmitted.
 * @param slot Slot concerned.
 * @param connection_id Connection id concerned.
 *
 * @return Number of bytes copied to dest, or <0 on error.
 */
int dvb_ca_buf_copy_to_cam(dvb_ca* ca, u8 *dest, int len, u8* last_fragment, u8 slot, u8 connection_id)
{
        struct dvb_ca_connection* cacon;
        int status;

        /* in the buffer, each packet has a 3 byte header as follows:
           0: High byte of total packet size.
           1: Low byte of total packet size.
           2: Flag byte. 0=> packet incomplete. 1=> packet complete.
           These are all updated as data fragments are appended to the packet.
        */


        /* acquire the slot */
        if (status = dvb_ca_slot_acquire(ca, slot)) return status;

        /* get the connection struct for this slot/connection id */
        cacon = dvb_ca_connection_get(ca, slot, connection_id, 0);

        /* if we didn't find a connection struct, there is no data! */
        if (cacon == NULL) {
                dvb_ca_slot_release(ca, slot);
                *last_fragment = 1;
                return 0;
        }

        /* if there is no data, there is no point going on */
        if (dvb_ringbuffer_avail(&cacon->tx_buffer) == 0) {
                dvb_ca_slot_release(ca, slot);
                *last_fragment = 1;
                return 0;
        }

        /* if we're not in the middle of sending a packet... */
        if (cacon->tx_partial_pkt_size == -1);
                /* ensure there is actually enough data for a header */
                if (dvb_ringbuffer_avail(&cacon->tx_buffer) < 3) {
                        dvb_ca_slot_release(ca, slot);
                        *last_fragment = 1;
                        return 0;
                }

                /* if the packet data is not available yet, exit straight away */
                if (DVB_RINGBUFFER_PEEK(&cacon->tx_buffer, 2) != 1) {
                        dvb_ca_slot_release(ca, slot);
                        *last_fragment = 1;
                        return 0;
                }

                /* get the size of the next packet */
                cacon->tx_partial_pkt_size = DVB_RINGBUFFER_PEEK(&cacon->tx_buffer, 0) << 8;
                cacon->tx_partial_pkt_size |= DVB_RINGBUFFER_PEEK(&cacon->tx_buffer, 1);
                DVB_RINGBUFFER_SKIP(&cacon->tx_buffer, 3);
        }

        /* don't copy more data than is actually available */
        if (cacon->tx_partial_pkt_size < len) len = cacon->partial_pkt_size;

        /* copy the data into dest */
        dvb_ringbuffer_read(&cacon->tx_buffer, dest, len, 0);

        /* update the amount of consumed data */
        cacon->tx_partial_pkt_size -= len;
        if (cacon->tx_partial_pkt_size == 0) {
                /* whole packet consumed => move to next */
                cacon->tx_partial_pkt_size = -1;

                /* wake up the wait queues now writing is completed */
                wake_up_interruptible(&ca->read_queue);
                wake_up_interruptible(&ca->write_queue);
        }

        /* ok, return the amount of data copied this time */
        *last_fragment = (cacon->tx_partial_pkt_size == -1) ? 1 : 0;
        dvb_ca_slot_release(ca, slot);
        return len;
}



/**
 * This function does not talk to a CAM; it is a utility function for caching data retrieved
 * from a CAM by some hardware-dependent method and storing it in the connection buffers.
 *
 * Copy data recieved from the CAM into the cache. This call is for lower layers
 * to cache data received from the CAM. The buffer is emptied by the higher layers
 * which send data to userspace.
 *
 * @param ca CA instance.
 * @param data Buffer containing data.
 * @param len Number of bytes.
 * @param last_fragment 1 if this is the last fragment of a packet, or 0 if not.
 * @param slot Slot the data was received from.
 * @param connection_id Connection id the data was received from.
 *
 * @return 0 on success, or -1 if there was not enough space.
 * If -1 is returned, the packet will be automatically discarded. Any further data written
 * for this packet (last_fragment==0) will also be discarded until the next packet (i.e. the
 * write AFTER the write with last_fragment==1).
 */
int dvb_ca_buf_copy_from_cam(dvb_ca* ca, u8 *data, int len, u8 last_fragment, u8 slot, u8 connection_id)
{
        struct dvb_ca_connection* cacon;
        int total_length = 0;
        int status;


        /* in the buffer, each packet has a 3 byte header as follows:
           0: High byte of total packet size.
           1: Low byte of total packet size.
           2: Flag byte. 0=> packet incomplete. 1=> packet complete.
           These are all updated as data fragments are appended to the packet.
        */

        /* acquire the slot */
        if (status = dvb_ca_slot_acquire(ca, slot)) return status;

        /* get the connection struct for this slot/connection id */
        cacon = dvb_ca_connection_get(ca, slot, connection_id, 1);
        if (cacon == NULL) {
                dvb_ca_slot_release(ca, slot);
                return -1;
        }

        /* check if we're discarding the current packet */
        if (cacon->rx_partial_pkt == -2) {
                /* if it is the last_fragment of a packet, set the flags appropriately */
                if (last_fragment) {

                        /* we're NOT in the middle of a partial any more */
                        cacon->rx_partial_pkt = -1;
                }

                /* indicate we're discarding this bit */
                dvb_ca_slot_release(ca, slot);
                return -1;
        } else if (cacon->rx_partial_pkt == -1) {
                /* check we've got enough space for the data + 3 byte header */
                if (dvb_ringbuffer_free(&cacon->rx_buffer) < (len + 3)) {

                        /* if it is not the last fragment, we need to discard any other data written for this packet */
                        if (!last_fragment) {

                                /* set the rx_partial_pkt to indicate discarding in progress */
                                cacon->rx_partial_pkt = -2;
                        }

                        /* indicate we're discarding */
                        dvb_ca_slot_release(ca, slot);
                        return -1;
                }

                /* record position partial packet starts at */
                cacon->rx_partial_pkt = cacon->rx_buffer.pwrite - cacon->rx_buffer.data;

                /* build the dummy (for the moment) packet header */
                DVB_RINGBUFFER_WRITE_BYTE(&cacon->rx_buffer, 0);
                DVB_RINGBUFFER_WRITE_BYTE(&cacon->rx_buffer, 0);
                DVB_RINGBUFFER_WRITE_BYTE(&cacon->rx_buffer, 0);
        } else {
                /* check we've got enough space for this fragment, if not, need to discard the current packet */
                if (dvb_ringbuffer_free(&cacon->rx_buffer) < len) {
                        /* roll the write pointer back to the start of the packet being discarded */
                        cacon->rx_buffer.pwrite = cacon->rx_buffer.data + cacon->rx_partial_pkt;

                        /* mark the current packet to be discarded */
                        cacon->rx_partial_pkt = -2;

                        /* if it is the last_fragment of a packet, set the flags appropriately */
                        if (last_fragment) {
                                /* we're NOT in the middle of a packet */
                                cacon->rx_partial_pkt = -1;
                        }

                        dvb_ca_slot_release(ca, slot);
                        return -1;
                }

                /* get the current length of the partial packet */
                total_length = cacon->rx_buffer.data[(cacon->rx_partial_pkt + 0) % cacon->rx_buffer.size] << 8;
                total_length |= cacon->rx_buffer.data[(cacon->rx_partial_pkt + 1) % cacon->rx_buffer.size];
        }

        /* store the data */
        dvb_ringbuffer_write(&cacon->rx_buffer, data, len, 0);

        /* update the length of the packet in the packet header */
        total_length += len;
        cacon->rx_buffer.data[(cacon->rx_partial_pkt + 0) % cacon->rx_buffer.size] = total_length >> 8;
        cacon->rx_buffer.data[(cacon->rx_partial_pkt + 1) % cacon->rx_buffer.size] = (total_length & 0xff);

        /* if this is the last fragment, tidy up */
        if (last_fragment) {
                /* mark the packet as complete */
                cacon->rx_buffer.data[(cacon->rx_partial_pkt + 2) % cacon->rx_buffer.size] = 1;

                /* we no longer have a partial packet */
                cacon->rx_partial_pkt = -1;

                /* ok, wake up everything now reading is complete */
                wake_up_interruptible(&ca->read_queue);
                wake_up_interruptible(&ca->write_queue);
        }

        /* success */
        dvb_ca_slot_release(ca, slot);
        return 0;
}


#define DVB_CA_BUF_COPY_FROM_USER_CONDITION                                                   \
({                                                                                            \
        int __status;                                                                         \
                                                                                              \
        if ((__status = dvb_ca_slot_acquire(ca, slot)) == 0) {                                \
                cacon = dvb_ca_connection_get(ca, slot, connection_id, 1);                    \
                if (cacon == NULL) {                                                          \
                        __status = -EIO;                                                      \
                        dvb_ca_slot_release(ca, slot);                                        \
                } else {                                                                      \
                        __status = dvb_ringbuffer_free(&cacon->tx_buffer) >= (count+3)        \
                        if (!__status) dvb_ca_slot_release(ca, slot);                         \
                }                                                                             \
        }                                                                                     \
        __status;                                                                             \
})


/**
 * Function to copy a packet supplied by the user into the appropriate slot/connection's tx_buffer.
 *
 * @param ca CA instance.
 * @param file Associated file.
 * @param buf Source buffer (userspace).
 * @param count Size of buffer.
 * @param ppos Position in file (ignored).
 * @param slot Slot to write to.
 * @param connection_id Id of connection to write to.
 *
 * @return Number of bytes copied, or <0 on error.
 */
static ssize_t dvb_ca_buf_copy_from_user(dvb_ca* ca, struct file *file, const char *buf, size_t count, loff_t *ppos, int slot, int connection_id) {
        dvb_ca_connection* cacon;
        int status;
        int write_pos;

        /* Packets in the buffer have a 3 byte header as follows:
           0: MSB of total packet size.
           1: LSB of total packet size.
           2: Flag byte. 0=> packet incomplete. 1=> packet complete.
           These are all updated as data fragments are appended to the packet.
        */

        /* acquire the slot */
        if (status = dvb_ca_slot_acquire(ca, slot)) return status;

        /* locate the connection to write to */
        cacon = dvb_ca_connection_get(ca, slot, connection_id, 1);
        if (cacon == NULL) {
                dvb_ca_slot_release(ca, slot);
                return -EIO;
        }

        /* check if the packet is WAY too big */
        if ((count + 3) > cacon->tx_buffer.size) {
                dvb_ca_slot_release(ca, slot);
                return -EINVAL;
        }

        /* OK, block if there isn't enough space at the moment */
        if (dvb_ringbuffer_free(&cacon->tx_buffer) < (count + 3)) {

                /* if we're in non blocking mode... */
                if (file->f_flags & O_NONBLOCK) {
                        dvb_ca_slot_release(ca, slot);
                        return -EWOULDBLOCK;
                }

                /* wait for some space */
                dvb_ca_slot_release(ca, slot);
                if ((status = dvb_ca_wait_event_interruptible(ca->write_queue, DVB_CA_BUF_COPY_FROM_USER_CONDITION)) < 0) {
                        /* slot is already released if the above failed */
                        return status;
                }

                /* the wait was successful => slot is now reacquired */
        }

        /* write the packet header */
        write_pos = cacon->tx_buffer.pwrite - cacon->tx_buffer.data;
        DVB_RINGBUFFER_WRITE_BYTE(&cacon->tx_buffer, count >> 8);
        DVB_RINGBUFFER_WRITE_BYTE(&cacon->tx_buffer, count & 0xff);
        DVB_RINGBUFFER_WRITE_BYTE(&cacon->tx_buffer, 0);

        /* write the packet data to the buffer */
        if (dvb_ringbuffer_write(&cacon->tx_buffer, buf+3, count, 1) != count) {

                /* roll the write pointer back to the start of the packet being discarded */
                cacon->tx_buffer.pwrite = cacon->tx_buffer.data + write_pos;

                /* tell userspace */
                dvb_ca_slot_release(ca, slot);
                return -EFAULT;
        }

        /* mark the packet as available */
        cacon->tx_buffer.data[(write_pos + 2) % cacon->tx_buffer.size] = 1;

        /* return number of bytes written */
        dvb_ca_slot_release(ca, slot);
        return count;
}

// TODO: we should maybe remember the previous slot/connection and the next time start
// at the one after, to give them equal priority.
#define DVB_CA_BUF_COPY_TO_USER_CONDITION                                                       \
({                                                                                              \
        int __status;                                                                           \
                                                                                                \
        for(slot=0; slot < ca->slot_count; slot++) {                                            \
                                                                                                \
                if (dvb_ca_slot_acquire(ca, slot) == 0) {                                       \
                                                                                                \
                        /* go through each connection id to see if there is data available */   \
                        list_for_each_entry(cacon, &ca->slot_info[slot].connections, next) {    \
                                __status = dvb_ringbuffer_avail(&cacon->rx_buffer) >= 3;        \
                                if (__status) goto completed;                                   \
                        }                                                                       \
                        dvb_ca_slot_release(ca, slot);                                          \
                }                                                                               \
         }                                                                                      \
    completed:                                                                                  \
         __status;                                                                              \
})


/**
 * Function to copy a packet received from the CAM to a buffer supplied by the user.
 *
 * @param ca CA instance.
 * @param file Associated file.
 * @param buf Destination buffer (userspace).
 * @param count Size of buffer.
 * @param ppos Position in file (ignored).
 * @param _slot Updated to contain the slot received from.
 * @param connection_id Updated to contain the connection_id received from.
 *
 * @return Number of bytes copied, or <0 on error.
 */
static ssize_t dvb_ca_buf_copy_to_user(dvb_ca* ca, struct file *file, char *buf, size_t count, loff_t *ppos, u8* _slot, u8* connection_id) {
        dvb_ca_connection* cacon;
        int slot;
        int packet_size;
        int split;
        int status;

        /* Packets in the buffer have a 3 byte header as follows:
           0: MSB of total packet size.
           1: LSB of total packet size.
           2: Flag byte. 0=> packet incomplete. 1=> packet complete.
           These are all updated as data fragments are appended to the packet.
        */

        /* check there is some space in the user-supplied destination buffer */
        if (count == 0) return 0;

        /* get a slot/connection to read from */
        if ((status = DVB_CA_BUF_COPY_TO_USER_CONDITION) == 0) {

                /* if we're in nonblocking mode, exit immediately */
                if (file->f_flags & O_NONBLOCK)
                        return -EWOULDBLOCK;

                /* wait for some data */
                if ((status = dvb_ca_wait_event_interruptible(ca->read_queue, DVB_CA_BUF_COPY_TO_USER_CONDITION)) < 0) {
                        /* check for errors. No need to relase slot, as we won't have one */
                        return status;
                }
                /* the wait was successful => slot is now acquired */
        }

        /* check for errors. No need to relase slot, as we won't have one */
        if (status < 0) return status;

        /* is the packet marked complete? */
        if (DVB_RINGBUFFER_PEEK(&cacon->rx_buffer, 2) != 1) {
                dvb_ca_slot_release(ca, slot);
                return 0;
        }

        /* get the packet's size */
        packet_size = DVB_RINGBUFFER_PEEK(&cacon->rx_buffer,0) << 8;
        packet_size |= DVB_RINGBUFFER_PEEK(&cacon->rx_buffer,1);

        /* is there enough space in the destination buffer */
        if (count < packet_size) {
                dvb_ca_slot_release(ca, slot);
                return -EINVAL;
        }

        /* Copy the data into the user buffer */
        /* Cannot use dvb_ringbuffer_read() here because we have to ignore the
         * three byte packet header. We can't just skip it beforehand, because
         * the read COULD -EFAULT, and we'd have to roll back. In the meantime,
         * something could have overwritten the packet header we skipped. */
        if ((cacon->rx_buffer.pread + 3 + packet_size) > cacon->rx_buffer.size) {
                split = cacon->rx_buffer.size - (cacon->rx_buffer.pread + 3);
        } else {
                split = 0;
        }
        if (split > 0) {
                if (copy_to_user(buf, cacon->rx_buffer.data + cacon->rx_buffer.pread + 3, split)) {
                        dvb_ca_slot_release(ca, slot);
                        return -EFAULT;
                }
                if (copy_to_user(buf+split, cacon->rx_buffer.data, packet_size - split)) {
                        dvb_ca_slot_release(ca, slot);
                        return -EFAULT;
                }
        } else {
                if (copy_to_user(buf, cacon->rx_buffer.data + cacon->rx_buffer.pread + 3, packet_size)) {
                        dvb_ca_slot_release(ca, slot);
                        return -EFAULT;
                }
        }

        /* the copy was successful => skip the packet header and the packet data */
        DVB_RINGBUFFER_SKIP(&cacon->rx_buffer, 3 + packet_size);

        /* return number of bytes read */
        *_slot = slot;
        *connection_id = cacon->connection_id;
        dvb_ca_slot_release(ca, slot);
        return packet_size;
}




/* ******************************************************************************** */
/* File IO interface functions */




/**
 * Implementation of file open syscall.
 *
 * @param inode Inode concerned.
 * @param file File concerned.
 *
 * @return 0 on success, <0 on failure.
 */
int dvb_ca_io_open(struct inode *inode, struct file *file) {
        dvb_device_t *dvbdev=(dvb_device_t *) file->private_data;
        dvb_ca *ca=(dvb_ca*) dvbdev->priv;
        int i;

        int err=dvb_generic_open(inode, file);
        if (err<0)
                return err;

        ca->open = 1;
        return 0;
}


/**
 * Implementation of file close syscall.
 *
 * @param inode Inode concerned.
 * @param file File concerned.
 *
 * @return 0 on success, <0 on failure.
 */
int dvb_ca_io_release(struct inode *inode, struct file *file) {
        dvb_device_t *dvbdev=(dvb_device_t *) file->private_data;
        dvb_ca *ca=(dvb_ca*) dvbdev->priv;
        int i;

        /* mark the CA device as closed */
        ca->open = 0;

        /* flush connection buffers */
        for(i=0; i< ca->slot_count; i++) {
                dvb_ca_buf_flush_buffers(ca, slot);
        }

        int err=dvb_generic_release(inode, file);
        if (err<0)
                return err;
        return 0;
}


/**
 * Implementation of poll() syscall.
 *
 * @param file File concerned.
 * @param wait poll wait table.
 *
 * @return Standard poll mask.
 */
unsigned int dvb_ca_io_poll(struct file *file, poll_table *wait) {
        dvb_device_t* dvbdev = (dvb_device_t*) file->private_data;
        dvb_ca* ca = (dvb_ca*) dvbdev->priv;
        unsigned int mask=0;
        int slot;
        struct dvb_ca_connection* cacon;


        /* go through each slot looking to see if there is data available */
        for(slot=0; slot < ca->slot_count; slot++) {

                /* acquire the slot */
                dvb_ca_slot_acquire(ca, slot);

                /* go through each connection id to see if there is data available */
                list_for_each_entry(cacon, &ca->slot_info[slot].connections, next) {

                        /* if the RX queue is NOT empty... */
                        if (dvb_ringbuffer_avail(&cacon->rx_buffer) > 3) {
                                /* and if the first packet on the RX buffer is marked as complete, data is ready to be read */
                                if (DVB_RINGBUFFER_PEEK(&cacon->rx_buffer, 2) == 1)
                                        mask|= POLLIN;
                        }

                        /* is there data still to be written? */
                        if (!dvb_ringbuffer_empty(&cacon->tx_buffer))
                                mask|=POLLOUT;
                }

                /* finished with this slot now */
                dvb_ca_slot_release(ca, slot);
        }

        /* if there is something, return now */
        if (mask) return mask;

        /* wait for something to happen */
        poll_wait(file, &ca->read_queue, wait);

        /* see if there is some data available now */
        for(slot=0; slot < ca->slot_count; slot++) {

                /* acquire the slot */
                dvb_ca_slot_acquire(ca, slot);

                /* go through each connection id to see if there is data available */
                list_for_each_entry(cacon, &ca->slot_info[slot].connections, next) {

                        /* if the RX queue is NOT empty... */
                        if (dvb_ringbuffer_avail(&cacon->rx_buffer) > 3) {
                                /* and if the first packet on the RX buffer is marked as complete, data is ready to be read */
                                if (DVB_RINGBUFFER_PEEK(&cacon->rx_buffer, 2) == 1)
                                        mask|= POLLIN;
                        }

                        /* is there data still to be written? */
                        if (!dvb_ringbuffer_empty(&cacon->tx_buffer))
                                mask|=POLLOUT;
                }

                /* finished with this slot now */
                dvb_ca_slot_release(ca, slot);
        }

        return mask;
}

/**
 * Real ioctl implementation.
 * NOTE: CA_SEND_MSG/CA_GET_MSG ioctls have userspace buffers passed to them.
 *
 * @param inode Inode concerned.
 * @param file File concerned.
 * @param cmd IOCTL command.
 * @param arg Associated argument.
 *
 * @return 0 on success, <0 on error.
 */
int dvb_ca_io_do_ioctl(struct inode *inode, struct file *file, unsigned int cmd, void *parg) {
        dvb_device_t *dvbdev=(dvb_device_t *) file->private_data;
        dvb_ca* ca=(dvb_ca*) dvbdev->priv;
        int err=0;
        int slot, connection_id;
        ca_msg_t msg;
        int msg_buf_offset = &msg.p.en50021.buf - &msg;

        switch (cmd) {
        case CA_RESET:
                // FIXME: update
                err=dvb_ci_ll_reset(&ci->wbuf, file, (int) parg, &ci->info[0]);
                break;

        case CA_GET_CAP:
        {
                if (ca->flags & DVB_CA_FLAG_EN50221) {
                        ca_cap_t *cap = (ca_cap_t*) parg;

                        cap->slot_num=ca->slot_count;
                        cap->slot_type=CA_EN50221;
                        cap->descr_num=0;
                        cap->descr_type=0;
                }
                break;
        }


        case CA_GET_SLOT_INFO:
        {
                if (ca->flags & DVB_CA_FLAG_EN50221) {
                        ca_slot_info_t *info=(ca_slot_info_t *)parg;

                        if (info->num > ca->slot_count)
                                return -EINVAL;

                        info->type = CA_EN50221;
                        info->flags = 0;
                        if (ca->slot_info[info->num].cam_present)
                                  info->flags = CA_CI_MODULE_PRESENT | CA_CI_MODULE_READY;
                }
                break;
        }

        case CA_GET_MSG:
                if (ca->flags & DVB_CA_FLAG_EN50221) {
                        if (copy_from_user(&msg, parg, msg_buf_offset)) return -EFAULT;

                        err = dvb_ca_buf_copy_to_user(ca, file, parg + msg_buf_offset, msg.p.en50221.length, NULL, &slot, &connection_id);
                        if (err >= 0) {
                                msg.index = slot;
                                msg.type = CA_EN50221;
                                msg.p.en50221.connection_id = connection_id;
                                msg.p.en50221.length = err;
                                if (copy_to_user(&msg, parg, msg_buf_offset)) return -EFAULT;
                        }
                }
                break;

        case CA_SEND_MSG:
                if ((ca->flags & DVB_CA_FLAG_EN50221) && (msg->type == CA_EN50221)) {
                        if (copy_from_user(&msg, parg, msg_buf_offset)) return -EFAULT;

                        err = dvb_ca_buf_copy_from_user(ca, file, parg + msg_buf_offset, msg.p.en50221.length, NULL,
                                                        msg.index, msg.p.en50221.connection_id);
                }
                break;

        case CA_GET_DESCR_INFO:
        case CA_SET_DESCR:
        case CA_SET_PID:
                if (ca->flags & DVB_CA_FLAG_EN50221) {
                        err=-EINVAL;
                }
                break;

        default:
                err=-EINVAL;
                break;
        }

        if (ca->ioctl)
                err=ca->ioctl(ca, cmd, parg);
        if (err<0 && err!=-EINVAL)
                return err;
        return 0;
}


/**
 * Wrapper for ioctl implementation.
 *
 * @param inode Inode concerned.
 * @param file File concerned.
 * @param cmd IOCTL command.
 * @param arg Associated argument.
 *
 * @return 0 on success, <0 on error.
 */
int dvb_ca_io_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg) {
        /* need to pass userspace buffer to these two */
        if ((cmd == CA_GET_MSG) || (cmd == CA_SEND_MSG)) {
                return dvb_ca_do_ioctl(inode, file, cmd, (void*) arg);
        }

        /* otherwise, just use the easy route */
        return generic_usercopy(inode, file, cmd, arg, dvb_ca_io_do_ioctl);
}


/**
 * Implementation of write() syscall.
 *
 * @param file File structure.
 * @param buf Source buffer.
 * @param count Size of source buffer.
 * @param ppos Position in file (ignored).
 *
 * @return Number of bytes read, or <0 on error.
 */
ssize_t dvb_ca_io_write(struct file *file, const char *buf, size_t count, loff_t *ppos) {
        dvb_device_t *dvbdev=(dvb_device_t *) file->private_data;
        dvb_ca *ca=(dvb_ca*) dvbdev->priv;
        u8 hdr[2];
        u8 slot, connection_id;

        /* Incoming packet has a 2 byte header. hdr[0] = slot_id, hdr[1] = connection_id */
        if (count < 2) return -EINVAL;

        /* extract slot and connection id */
        if (copy_from_user(hdr, buf, 2)) return -EFAULT;
        slot = hdr[0];
        connection_id = hdr[1];

        /* copy the data from the buffer */
        status = dvb_ca_buf_copy_from_user(ca, file, buf+2, count-2, ppos, slot, connection_id);
        if (status < 0) return status;

        /* success */
        return status + 2;
}


/**
 * Implementation of read() syscall.
 *
 * @param file File structure.
 * @param buf Destination buffer.
 * @param count Size of destination buffer.
 * @param ppos Position in file (ignored).
 *
 * @return Number of bytes read, or <0 on error.
 */
ssize_t dvb_ca_io_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
        dvb_device_t *dvbdev=(dvb_device_t *) file->private_data;
        dvb_ca *ca=(dvb_ca*) dvbdev->priv;
        int status;
        u8 hdr[2];
        u8 slot, connection_id;

        /* Outgoing packet has a 2 byte header. hdr[0] = slot_id, hdr[1] = connection_id */
        if (count < 2) return -EINVAL;

        /* copy the data into the buffer */
        status = dvb_ca_buf_copy_to_user(ca, file, buf+2, count-2, ppos, &slot, &connection_id);
        if (status < 0) return status;

        /* set the slot and connection id */
        hdr[0] = slot;
        hdr[1] = connetion_id;
        if (copy_to_user(buf, hdr, 2)) return -EFAULT;

        /* success */
        return status + 2;
}

static struct file_operations dvb_ca_fops = {
        owner: THIS_MODULE,
        read: dvb_ca_io_read,
        write: dvb_ca_io_write,
        ioctl: dvb_ca_io_ioctl,
        open: dvb_ca_io_open,
        release: dvb_ca_io_release,
        poll: dvb_ca_io_poll,
};

static dvb_device_t dvbdev_ca = {
        priv: 0,
        users: 1,
        writers: 1,
        fops: &dvb_ca_fops,
};





/* ******************************************************************************** */
/* Initialisation/shutdown functions */


/**
 * Initialise a new DVB CA device.
 *
 * @param dvb_adapter DVB adapter to attach the new CA device to.
 * @param ca_dev The dvb_device_t to return.
 * @param ca The dvb_ca instance.
 *
 * @return 0 on success, nonzero on failure
 */
int dvb_ca_init(dvb_adapter_t *dvb_adapter, dvb_device_t **ca_dev, struct dvb_ca* ca) {
        int ret;

        dprintk ("%s\n", __FUNCTION__);

    // FIXME: update

        /* make sure everything is within tolerances */
        if (ca->slot_count > DVB_CA_MAX_SLOTS) {
                return -EINVAL;
        }

        /* initialise the system parts */
        memset(&(ca->slot_info[0]), 0, sizeof(struct dvb_ca_slot_info) * DVB_CA_MAX_SLOTS);
        init_MUTEX(&ca->thread_sem);
        init_waitqueue_head(&ca->thread_queue);
        init_waitqueue_head(&ca->read_queue);
        init_waitqueue_head(&ca->write_queue);
        ca->exit = 0;
        ca->thread_pid = 0;

        /* register the DVB device */
        ret = dvb_register_device(dvb_adapter, ca_dev, &dvbdev_ca, ca, DVB_DEVICE_CA);
        return ret;

        /* create a kthread if necessary */
        if (ca->flags & DVB_CA_FLAG_EN50221) {

                if (signal_pending(current)) return -EINTR;
                if (down_interruptible (&ca->thread_sem)) return -EINTR;

                mb();

                /* create a kthread for monitoring this CA device */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
                {
                /* bug in linux 2.4.x: kernel_thread() fails when the process calling
                * open() is run in the debugger (fixed in kernel 2.5). I hope this
                * workaround does not have any ugly side effects...
                */
                int pt = current->ptrace;
                current->ptrace = 0;
#endif

                ret = kernel_thread (dvb_ca_thread, ca, 0);

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
                current->ptrace = pt;
                }
#endif

                if (ret < 0) {
                        printk("dvb_ca_init: failed to start kernel_thread (%d)\n", ret);
                        up(&ca->thread_sem);
                        return ret;
                }
                ca->thread_pid = ret;
        }

        /* done */
        return 0;
}



/**
 * Release a DVB CA device.
 *
 * @param ca_dev The dvb_device_t instance for the CA device.
 * @param ca The associated dvb_ca instance.
 */
void dvb_ca_release(dvb_device_t *ca_dev, struct dvb_ca* ca) {
        dprintk ("%s %s\n", __FUNCTION__, ca->name);

    // FIXME: update

        /* shutdown the thread if there was one */
        if (ca->thread_pid) {

                /* tell the thread to stop */
                ca->exit = 1;
                mb();

                /* check if the thread is really alive */
                if (kill_proc(fe->thread_pid, 0, 1) == -ESRCH) {
                        printk("dvb_ca_shutdown: thread PID %d already died\n", ca->thread_pid);
                        /* make sure the mutex was not held by the thread */
                        init_MUTEX (&ca->thread_sem);
                        return;
                }

                wake_up_interruptible(&ca->thread_queue);
                interruptible_sleep_on(&ca->thread_queue);

                /* paranoia check */
                if (ca->thread_pid)
                        printk("dvb_ca_shutdown: warning: thread PID %d won't exit\n", ca->thread_pid);
        }

        /* disable all CAM modules if we're an EN50221 style CAM */
        if (ca->flags & DVB_CA_FLAG_EN50221) {
                for(i=0; i< ca->slot_count; i++) {
                        dvb_ca_en50221_slot_shutdown(ca, i);
                }
        }

        /* finally, unregister the device */
        dvb_unregister_device(ca_dev);
}
