#include <config.h>

#ifdef __GNUG__
#pragma implementation
#endif

#include "math_deliminset.h"
#include "math_parser.h"
#include "math_support.h"
#include "math_mathmlstream.h"
#include "math_streamstr.h"
#include "math_extern.h"


using std::max;

namespace {

string convertDelimToLatexName(string const & name)
{
	if (name == "<")
		return name;
	if (name == "(")
		return name;
	if (name == "[")
		return name;
	if (name == ".")
		return name;
	if (name == ">")
		return name;
	if (name == ")")
		return name;
	if (name == "]")
		return name;
	if (name == "/")
		return name;
	if (name == "|")
		return name;
	return '\\' + name + ' ';
}

}



MathDelimInset::MathDelimInset(string const & l, string const & r)
	: MathNestInset(1), left_(l), right_(r)
{}


MathDelimInset::MathDelimInset
		(string const & l, string const & r, MathArray const & ar)
	: MathNestInset(1), left_(l), right_(r)
{
	cell(0) = ar;
}


MathInset * MathDelimInset::clone() const
{
	return new MathDelimInset(*this);
}


void MathDelimInset::write(WriteStream & os) const
{
	os << "\\left" << convertDelimToLatexName(left_) << cell(0)
	   << "\\right" << convertDelimToLatexName(right_);
}


void MathDelimInset::normalize(NormalStream & os) const
{
	os << "[delim " << convertDelimToLatexName(left_) << ' '
	   << convertDelimToLatexName(right_) << ' ' << cell(0) << ']';
}


int MathDelimInset::dw() const
{
	int w = height() / 5;
	if (w > 8)
		w = 8;
	if (w < 4)
		w = 4;
	return w;
}


void MathDelimInset::metrics(MathMetricsInfo & mi) const
{
	cell(0).metrics(mi);
	Dimension t;
	mathed_char_dim(mi.base.font, 'I', t);
	int h0 = (t.a + t.d) / 2;
	int a0 = max(cell(0).ascent(), t.a)   - h0;
	int d0 = max(cell(0).descent(), t.d)  + h0;
	dim_.a = max(a0, d0) + h0;
	dim_.d = max(a0, d0) - h0;
	dim_.w = cell(0).width() + 2 * dw() + 8;
}


void MathDelimInset::draw(MathPainterInfo & pi, int x, int y) const
{
	int const w = dw();
	int const b = y - ascent();
	cell(0).draw(pi, x + w + 4, y);
	mathed_draw_deco(pi, x + 4, b, w, height(), left_);
	mathed_draw_deco(pi, x + width() - w - 4, b, w, height(), right_);
}


bool MathDelimInset::isParanthesis() const
{
	return left_ == "(" && right_ == ")";
}


bool MathDelimInset::isBrackets() const
{
	return left_ == "[" && right_ == "]";
}


bool MathDelimInset::isAbs() const
{
	return left_ == "|" && right_ == "|";
}


void MathDelimInset::maplize(MapleStream & os) const
{
	if (isAbs()) {
		if (cell(0).size() == 1 && cell(0).front()->asMatrixInset())
			os << "linalg[det](" << cell(0) << ')';
		else
			os << "abs(" << cell(0) << ')';
	}
	else
		os << left_ << cell(0) << right_;
}

void MathDelimInset::maximize(MaximaStream & os) const
{
	if (isAbs()) {
		if (cell(0).size() == 1 && cell(0).front()->asMatrixInset())
			os << "determinant(" << cell(0) << ')';
		else
			os << "abs(" << cell(0) << ')';
	}
	else
		os << left_ << cell(0) << right_;
}


void MathDelimInset::mathematicize(MathematicaStream & os) const
{
	if (isAbs()) {
		if (cell(0).size() == 1 && cell(0).front()->asMatrixInset())
			os << "Det" << cell(0) << ']';
		else
			os << "Abs[" << cell(0) << ']';
	}
	else
		os << left_ << cell(0) << right_;
}


void MathDelimInset::mathmlize(MathMLStream & os) const
{
	os << "<fenced open=\"" << left_ << "\" close=\""
		<< right_ << "\">" << cell(0) << "</fenced>";
}


void MathDelimInset::octavize(OctaveStream & os) const
{
	if (isAbs())
		os << "det(" << cell(0) << ')';
	else
		os << left_ << cell(0) << right_;
}
