/* Menus
 *
 * There are several widgets involved in displaying menus. The
 * Gtk::MenuBar widget is a horizontal menu bar, which normally appears
 * at the top of an application. The Gtk::Menu widget is the actual menu
 * that pops up. Both Gtk::MenuBar and Gtk::Menu are derived from
 * Gtk::MenuShell; a Gtk::MenuShell contains menu items
 * (Gtk::MenuItem). Each menu item contains text and/or images and can
 * be selected by the user.
 *
 * There are several kinds of menu item, including plain Gtk::MenuItem,
 * Gtk::CheckMenuItem which can be checked/unchecked, Gtk::RadioMenuItem
 * which is a check menu item that's in a mutually exclusive group,
 * Gtk::SeparatorMenuItem which is a separator bar, Gtk::TearoffMenuItem
 * which allows a GtkMenu to be torn off, and Gtk::ImageMenuItem which
 * can place a Gtk::Image or other widget next to the menu text.
 *
 * A Gtk::MenuItem can have a submenu, which is simply a Gtk::Menu to pop
 * up when the menu item is selected. Typically, all menu items in a menu bar
 * have submenus.
 *
 * The Gtk::OptionMenu widget is a button that pops up a Gtk::Menu when clicked.
 * It's used inside dialogs and such.
 *
 * Gtk::ItemFactory provides a higher-level interface for creating menu bars
 * and menus; while you can construct menus manually, most people don't
 * do that. There's a separate demo for Gtk::ItemFactory.
 *
 */

#include "demowindow.h"
#include <inti/gtk/accelgroup.h>
#include <inti/gtk/box.h>
#include <inti/gtk/menubar.h>
#include <inti/gtk/menu.h>
#include <inti/gtk/tearoffmenuitem.h>
#include <inti/gtk/radiomenuitem.h>
#include <inti/gtk/separatormenuitem.h>
#include <inti/gtk/checkmenuitem.h>
#include <inti/gtk/optionmenu.h>
#include <inti/gtk/separator.h>
#include <inti/gtk/button.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

using namespace Inti;

class Menu : public Gtk::Menu
{
public:
	Menu(int depth, bool tearoff);
	~Menu();
};

class MenuWindow : public Gtk::Window
{
protected:
	virtual bool on_delete_event(const Gdk::EventAny& event) { return true; }

public:
	MenuWindow();
	~MenuWindow();
};
//!

// Menu

Menu::Menu(int depth, bool tearoff)
{
	if (tearoff)
	{
		Gtk::TearoffMenuItem *menuitem = new Gtk::TearoffMenuItem;
		append(*menuitem);
		menuitem->show();
	}

	Gtk::RadioMenuItem::Group *group = 0;
	for (int i = 0, j = 1; i < 5; i++, j++)
	{
		char buf[32];
		sprintf(buf, "item %2d - %d", depth, j);
		Gtk::RadioMenuItem *menuitem = new Gtk::RadioMenuItem(group, buf);
		group = menuitem->get_group();

		append(*menuitem);
 		menuitem->show();
		if (i == 3)
			menuitem->set_sensitive(false);

		depth -= 1;
		if (depth >= 1)
		{
			Gtk::Menu *submenu = new Menu(depth, true);
			menuitem->set_submenu(*submenu);
		}
	}
}

Menu::~Menu()
{
}

// MenuWindow

MenuWindow::MenuWindow()
{
	Pointer<Gtk::AccelGroup> accel_group = new Gtk::AccelGroup;
	add_accel_group(accel_group);
	set_title("Menus");
	set_border_width(0);

	Gtk::VBox *box1 = new Gtk::VBox;
	add(*box1);
	box1->show();
      
	Gtk::MenuBar *menubar = new Gtk::MenuBar;
	box1->pack_start(*menubar, false);
	menubar->show();

	Gtk::Menu *menu = new Menu(2, true);
	Gtk::MenuItem *menuitem = new Gtk::MenuItem("test\nline2");
	menuitem->set_submenu(*menu);
	menubar->append(*menuitem);
	menuitem->show();

	menu = new Menu(3, true);
	menuitem = new Gtk::MenuItem("foo");
	menuitem->set_submenu(*menu);
	menubar->append(*menuitem);
	menuitem->show();

	menu = new Menu(4, true);
	menuitem = new Gtk::MenuItem("bar");
	menuitem->set_submenu(*menu);
	menuitem->set_right_justified(true);
	menubar->append(*menuitem);
	menuitem->show();

	Gtk::VBox *box2 = new Gtk::VBox(false, 10);
	box2->set_border_width(10);
	box1->pack_start(*box2);
	box2->show();
      
	menu = new Menu(1, false);
	menu->set_accel_group(accel_group);
      
	menuitem = new Gtk::SeparatorMenuItem;
	menu->append(*menuitem);
	menuitem->show();
      
	menuitem = new Gtk::CheckMenuItem("Accelerate Me");
	menu->append(*menuitem);
	menuitem->show();
	menuitem->add_accelerator("activate", *accel_group, Gtk::AccelKey(GDK_F1, 0));
	
	menuitem = new Gtk::CheckMenuItem("Accelerator Locked");
	menu->append(*menuitem);
	menuitem->show();
	menuitem->add_accelerator("activate", *accel_group, Gtk::AccelKey(GDK_F2, 0, Gtk::ACCEL_VISIBLE | Gtk::ACCEL_LOCKED));

	menuitem = new Gtk::CheckMenuItem("Accelerators Frozen");
	menu->append(*menuitem);
	menuitem->show();
	menuitem->add_accelerator("activate", *accel_group, Gtk::AccelKey(GDK_F3, 0));

	Gtk::OptionMenu *optionmenu = new Gtk::OptionMenu;
	optionmenu->set_menu(*menu);
	optionmenu->set_history(3);
 	box2->pack_start(*optionmenu);
	optionmenu->show();

	Gtk::HSeparator *separator = new Gtk::HSeparator;
	box1->pack_start(*separator, false);
	separator->show();

	box2 = new Gtk::VBox(false, 10);
	box2->set_border_width(10);
	box1->pack_start(*box2, false);
	box2->show();

	Gtk::Button *button = new Gtk::Button("close");
	button->sig_clicked().connect(slot(this, &Gtk::Object::dispose));
	box2->pack_start(*button);
	button->set_flags(Gtk::CAN_DEFAULT);
	button->grab_default();
	button->show();
	show();
}

MenuWindow::~MenuWindow()
{
}

Gtk::Window*
do_menus()
{
	static Gtk::Window *window = 0;

	if (!window)
		window = new MenuWindow;
	else
	{
		window->dispose();
		window = 0;
	}
	return window;
}

