#include "itemfactory.h"
#include <inti/gtk/accelgroup.h>
#include <inti/gtk/menubar.h>
#include <inti/gtk/eventbox.h>
#include <inti/gtk/label.h>
#include <inti/bind.h>

ItemFactoryWindow::ItemFactoryWindow()
{
	set_title("Item Factory");
	set_size_request(300, 200);

	// Boxes don't receive button events so use an eventbox. The eventbox is added first
	// and then all the other widgets added to it.
	Gtk::EventBox *eventbox = new Gtk::EventBox;
	add(*eventbox);
	
	// Set the events the eventbox is to receive. These can be any number of or'd (|) values
	// from the Gdk::EventMask enumeration.
	eventbox->set_events(Gdk::BUTTON_PRESS_MASK);

	// Add the packing box to eventbox
	Gtk::VBox *vbox = new Gtk::VBox(false, 1);
	vbox->set_border_width(1);
	eventbox->add(*vbox);

	// Create the menubar. The item factory constructor takes 3 parameters.
	// Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU, or GTK_TYPE_OPTION_MENU.
	// Param 2: The path of the menu.
	// Param 3: A pointer to a Gtk::AccelGroup.  The item factory sets up the accelerator table while generating menus.
	Pointer<Gtk::AccelGroup> accel_group = new Gtk::AccelGroup;
	item_factory = new Gtk::ItemFactory(GTK_TYPE_MENU_BAR, "<main>", accel_group);
	add_accel_group(accel_group);

	// Create menu items. The create_items() function generates the menu items. It takes a reference to the window.
	item_factory->create_items(*this);

	// Retrieve a pointer to the File menu. Do not include the underscore when specifying the
	// submenu path. ItemFactory parses out all the underscores at creation.
	Gtk::Menu *file_menu = item_factory->submenu("/File");
	
	// Bind the file menu to the button_press event and use it as the popup menu.
	eventbox->sig_button_press_event().connect(bind(slot(this, &ItemFactoryWindow::on_button_press), file_menu));

	// Retrieve a pointer to the menubar
	Gtk::MenuBar *menu = item_factory->menu_bar();

	// Pack the menubar into the vbox
	vbox->pack_start(*menu, false);
	
	// Add a label that tells the user to click the mouse button inside the client area.
	Gtk::Label *label = new Gtk::Label("Click mouse button here...");
	vbox->pack_start(*label);

	// Show everything
	label->show();
	menu->show();
	vbox->show();
	eventbox->show();
}

ItemFactoryWindow::~ItemFactoryWindow()
{
}

bool
ItemFactoryWindow::on_button_press(GdkEventButton *event, Gtk::Menu *menu)
{
	menu->popup(event->button, event->time);
	return true;
}

void
ItemFactoryWindow::menu_item_selected(const char *parent, const char *item)
{
	g_message("ItemFactory: activated the \"%s\" menu item: \"%s\"", parent, item);
}

void
ItemFactoryWindow::on_file_new()
{
	menu_item_selected("File", "New");
}

void
ItemFactoryWindow::on_file_open()
{
	menu_item_selected("File", "Open");
}

void
ItemFactoryWindow::on_file_save()
{
	menu_item_selected("File", "Save");
}

void
ItemFactoryWindow::on_file_save_as()
{
	menu_item_selected("File", "Save As");
}

void
ItemFactoryWindow::on_file_quit()
{
	dispose();
}

void
ItemFactoryWindow::on_options_test()
{
	menu_item_selected("Options", "Test");
}

void
ItemFactoryWindow::on_help_about()
{
	menu_item_selected("Help", "About");
}

// Item factory map macros are defined in inti/gtk/itemfactory.h

BEGIN_ITEM_FACTORY_MAP(ItemFactoryWindow)
	IFM_BRANCH("/_File"),
	IFM_ITEM("/File/_New", "<control>N", on_file_new),
	IFM_ITEM("/File/_Open", "<control>0", on_file_open),
	IFM_ITEM("/File/_Save", "<control>S", on_file_save),
	IFM_ITEM("/File/Save _As", 0, on_file_save_as),
	IFM_SEPARATOR("/File/sep1"),
	IFM_ITEM("/File/Quit", "<control>Q", on_file_quit),
	IFM_BRANCH("/_Options"),
	IFM_ITEM("/_Options/Test", 0, on_options_test),
	IFM_LAST_BRANCH("/_Help"),
	IFM_ITEM("/Help/_About", 0, on_help_about),
END_ITEM_FACTORY_MAP

// Convenience macro for a simple main function

INTI_MAIN(ItemFactoryWindow)

