/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/integerobject.h
//! @brief A template base class for GTK+ integer objects.
//!
//! Provides a template base class for integer objects such as Gdk::Keyval, G::Quark and G::Unichar.

#ifndef INTI_INTEGER_OBJECT_H
#define INTI_INTEGER_OBJECT_H

namespace Inti {

//! @class IntegerObject integerobject.h inti/integerobject.h
//! @brief A C++ template base class for integer objects.
//!
//! IntegerObject is a template base class for integer objects like
//! Gdk::Keyval, G::Quark and G::Unichar.

template<typename IntegerType>
class IntegerObject
{
protected:
	IntegerType value_;

//! @name Constructors
//! @{

	IntegerObject(IntegerType value) : value_(value) {}
	//!< Construct a new integer object.
	//!< @param value The integer value to initialize the integer object with.
	
//! @}

public:
	typedef IntegerObject<IntegerType> Base;
	//!< Convenient base class typedef for derived classes.

//! @name Accessors
//! @{

	operator IntegerType () const { return value_; }
	//!< Conversion operator; converts an IntegerObject to an IntegerType.

	bool operator==(const IntegerObject& other) const { return value_ == other.value_; }
	//!< Equality operator.
	//!< @param other Another IntegerObject.
	//!< @return <EM>true</EM> if the value of this integer object and other are equal.

	bool operator!=(const IntegerObject& other) const { return value_ != other.value_; }
	//!< Inequality operator.
	//!< @param other Another IntegerObject.
	//!< @return <EM>true</EM> if the value of this integer object and other are not equal.

//! @}
};

} // namespace Inti

#endif // INTI_INTEGER_OBJECT_H


