#ifndef H_hodgkin_huxley
#define H_hodgkin_huxley 1	// Don't let C++ see this twice.

#include "misc.h"

//
// A patch of membrane that follows the Hodgkin-Huxley equations.
// We compute m, h, and n at time t+dt using the implicit Euler method except
// we use the voltage at time t.
//
// This class does not compute voltages or currents; it merely acceps the
// voltage as an argument and computes a conductance.
//
class Hodgkin_Huxley
{
public:
//
// Constructor.  Argument:
// 1) Initial voltage.
//
  Hodgkin_Huxley(Float ini_v = -65.) { initialize(ini_v); }

//
// Initialize the variables to their steady state value for some given voltage.
// Arguments:
// 1) The voltage.
//
  void initialize(Float ini_v);

//
// Advance the variables to the next timestep.  Arguments:
// 1) The time step.
// 2) The current value of the voltage.
//
// We use the imlicit Euler method, except that we use the given voltage (which
// presumably will be the voltage from time t if we're computing m, n, and h
// at time t+1).
//
// Currently, you can change the timestep freely without having to modify any
// precomputed constants, as far as this class is concerned.
//
  void advance(Float dt, Float v);

//
// Compute the sodium and potassium conductances.  Units are uS or uS/um^2
// (whatever units you used for g_Na_bar and g_K_bar).
//
  Float g_Na() const { return g_Na_bar * m * m * m * h; }
  Float g_K()  const { Float nsq = n*n;  return g_K_bar * nsq * nsq; }

//
// Compute the net conductance and the net reversal potential.  Arguments:
// 1) Where to return the net conductance.
// 2) Where to return the net reversal potential times the conductance.
//    (The net reversal potential isn't really useful anywhere; we always want
//    it times the net conductance.  Returning it this way avoids a floating
//    point division and a floating point multiplication.)
// 3) The area (optional).  Since conductances are usually given in
//    uS/um^2, it can be useful to multiply by the area to get an absolute
//    conductance value.
//
  void net_g_and_E(Float &net_g, Float &net_g_times_E, Float area=1.0) const {
    Float gNa = g_Na()*area;	// Compute these values just once.  We have
    Float gK = g_K()*area;	// to use them twice.
    Float gLeak = g_leak * area;
    net_g = gNa+gK+gLeak;
    net_g_times_E = gNa*E_Na + gK*E_K + gLeak*E_leak;
  } //%output net_g, net_g_times_E

//
// Data members.  You can change these values freely at any time without
// causing any side effects.  The constants are currently static because we
// aren't trying to vary them spatially.
//
// You can also copy these values using the operator= for this structure.
//
  static Float g_Na_bar, g_K_bar;// The peak conductances of the sodium and
  static Float g_leak;		// potassium conductances.  Units are
				// uS or uS/um^2.  The units here determine
				// the units for g_Na() and g_K().
  static Float E_Na, E_K;	// The reversal potential for the sodium and
  static Float E_leak;		// potassium conductances.  Units are mV.

  Float m, h, n;		// The current value of the state variables.
};

#endif // H_hodgkin_huxley
