/*
 ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
 *
 *
 * formatted with JxBeauty (c) johann.langhofer@nextra.at
 */


package  com.sap.dbtech.jdbc;

import  com.sap.dbtech.jdbc.translators.DBTechTranslator;
import  com.sap.dbtech.jdbc.exceptions.*;
import  java.sql.SQLException;


/**
 * This class stores informations about input and/or output parameters like number of parametrs, datatype and etc.
 * It provides basis functionality to access this informations. This basis class will extended by ParameterMetaDataSapDB
 * and ResultSetMetaDataSapDB.
 */
public class ParameterInfo {
    /**
     * DBTechTranslator[] paramInfo
     */
    DBTechTranslator[] paramInfo;

    /**
     * Creates a new ParameterInfo object
     * @param infos com.sap.dbtech.jdbc.translators.DBTechTranslator[]
     * @since 1.4 JDBC 3.0
     *
     */
    public ParameterInfo (DBTechTranslator[] infos) {
        this.paramInfo = infos;
    }

    /**
     * Retrieves a <code>DBTechTranslator</code> object which contain informations about a
     * parameter corresponding to paramIndex.
     * @param paramIndex the first parameter is 1, the second is 2, ...
     * @return a <code>DBTechTranslator</code>
     * @throws SQLException if the given index is invalid for the parameter set
     */
    public DBTechTranslator findParamInfo (int paramIndex) throws SQLException {
        DBTechTranslator info;
        try {
            info = this.paramInfo[paramIndex - 1];
        } catch (ArrayIndexOutOfBoundsException exc) {
            throw  new InvalidColumnException(paramIndex, this);
        }
        return  info;
    }

    /**
     * Retrieves the fully-qualified name of the Java class whose instances
     *  should be passed to the method <code>PreparedStatement.setObject</code>.
     * @param index the first parameter is 1, the second is 2, ...
     * @return the fully-qualified name of the class in the Java programming
     *          language.
     * @throws SQLException if a database access error occurs
     */
    public String getParameterClassName (int index) throws SQLException {
        return  this.findParamInfo(index).getColumnClassName();
    }

    /**
     * Retrieves the designated parameter's SQL type.
     * @param index the first parameter is 1, the second is 2, ...
     * @return SQL type from <code>java.sql.Types</code>
     * @throws SQLException if a database access error occurs
     */
    public int getParameterType (int index) throws SQLException {
        return  this.findParamInfo(index).getColumnType();
    }

    /**
     * Retrieves the designated parameter's database-specific type name.
     * @param index the first parameter is 1, the second is 2, ...
     * @return type the name used by the database.
     * @throws SQLException if a database access error occurs
     */
    public String getParameterTypeName (int index) throws SQLException {
        return  this.findParamInfo(index).getColumnTypeName();
    }

    /**
     * Retrieves whether null values are allowed in the designated parameter.
     * @param index the first parameter is 1, the second is 2, ...
     * @return the nullability status of the given parameter; one of
     *         <code>ParameterMetaData.parameterNoNulls</code>,
     *         <code>ParameterMetaData.parameterNullable</code>, or
     *         <code>ParameterMetaData.parameterNullableUnknown</code>
     * @throws SQLException if a database access error occurs
     */
    public int isNullable (int index) throws SQLException {
        return  this.findParamInfo(index).isNullable();
    }

    /**
     * Retrieves the designated parameter's number of decimal digits.
     * @param index the first parameter is 1, the second is 2, ...
     * @return precision
     * @throws SQLException if a database access error occurs
     */
    public int getPrecision (int index) throws SQLException {
        return  this.findParamInfo(index).getPrecision();
    }

    /**
     * Retrieves the designated parameter's number of digits to right of the decimal point.
     * @param index the first parameter is 1, the second is 2, ...
     * @return Scale
     * @throws SQLException if a database access error occurs
     */
    public int getScale (int index) throws SQLException {
        return  this.findParamInfo(index).getScale();
    }

    /**
     * Retrieves whether values for the designated parameter can be signed numbers.
     * @param index the first parameter is 1, the second is 2, ...
     * @return </B><DD><code>true</code> if so; <code>false</code> otherwise
     * @throws SQLException if a database access error occurs
     */
    public boolean isSigned (int index) throws SQLException {
        return  true;
    }

    /**
     * Retrieves the number of parameters for which this object contains
     * information.
     * @return the number of parameters
     * @throws SQLException if a database access error occurs
     */
    public int getParameterCount () throws SQLException {
        return  this.paramInfo.length;
    }
}



