/*!
    @file           DBMCli_EventHandlerMainExtendDB.cpp
    @author         MarcW
    @brief          Main program for handling DB_ABOVE_LIMIT kernel events

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

/* 
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
*/
#include <signal.h>
#include <stdio.h>

#include "SAPDB/ToolsCommon/Tools_ParameterParser.hpp"
#include "SAPDB/ToolsCommon/Tools_Properties.hpp"
#include "SAPDB/ToolsCommon/Tools_PipeCall.hpp"
#include "SAPDB/DBM/Cli/DBMCli_EventHandlerExtendDB.hpp"
#include "SAPDB/DBM/Cli/DBMCli_EventingBranding.hpp"
#include "SAPDB/DBM/Cli/DBMCli_Logo.hpp"

/*!
    @brief points to the handler

    Points to the event handler for DB_ABOVE_LIMIT events.
    @see DBMCli_EventHandlerExtendDB
*/
DBMCli_EventHandler* pEventHandler = NULL;

/*! @brief get parameters from command line */
Tools_ParameterParser* pParameterParser = NULL;

/*! @brief return code of main function if program fails */
#define PRG_FATAL        1001

/*!
    @brief abort handler

    @param sig [IN] signal sent by OS
*/
extern "C" void abortHandler(int sig) {
  switch (sig) {
    case SIGINT:
    case SIGTERM:
    case SIGABRT:
      if (pEventHandler != NULL) {
        delete pEventHandler;
      } // end if
      exit(0);
      break;

    default:
      break;
  } // end switch
}

/*!
    @brief main function

    This program is normally called from the dispatcher. At least an event and DB connect data
    should be on the command line (with -evt, -d, -u, -n). Reads arguments from command line
    and initializes the event handling class.
*/
int main (int argcIn, char* argvIn[])
{
    // accept signals
    signal(SIGINT,   abortHandler);
    signal(SIGTERM,  abortHandler);
    signal(SIGABRT,  abortHandler);

    int        argc            = 0; 
    char **    argv            = NULL;

    int nRC = Tools_PipeCall::ReadArguments(
            argcIn,
            argvIn,
            argc,
            argv,
            Tools_PipeCall::ReadXUser);

    if( nRC != 0 ) {
        fprintf(stderr, "%s: could not read arguments from commandline. reason: %s\n",
            argvIn[0],
            Tools_PipeCall::ErrorReason());
        fflush(stderr);
        exit(1);
    }

    // read arguments
    pParameterParser  = new Tools_ParameterParser();

    DBMCli_String sName, sDesc;
    //node
    sName = "n";
    sDesc = "node     (name of servernode)";
    Tools_Parameter node(sName, sDesc, false);
    //dbname
    sName = "d";
    sDesc = "dbname   (name of datbase)";
    Tools_Parameter db(sName, sDesc);
    //userpw
    sName = "u";
    sDesc = "user,pwd (user for authorization)";
    Tools_Parameter userpw(sName, sDesc, false, true);
    //xuser key
    sName = "U";
    sDesc = "user_key  (from XUSER)";
    Tools_Parameter xuserkey(sName, sDesc, false, true);
    //limit
    sName = "l";
    sDesc = "limit    (between 60% and 95%)";
    Tools_Parameter limit(sName, sDesc, false);
    //log file
    sName = "lf";
    sDesc = "logfile  (log file)";
    Tools_Parameter logfile(sName, sDesc, false);

    //verbosity
    sName = "v";
    sDesc = "verbose    (informative output to stdout)";
    Tools_Parameter verbose(sName, sDesc, false, false);

    pParameterParser->addFormalParameter(node);
    pParameterParser->addFormalParameter(db);
    pParameterParser->addFormalParameter(userpw);
    pParameterParser->addFormalParameter(xuserkey);
    pParameterParser->addFormalParameter(limit);
    pParameterParser->addFormalParameter(logfile);
    pParameterParser->addFormalParameter(verbose);

    pParameterParser->setActualParameters( argc, argv );

    if( pParameterParser->isParameterSet(verbose.getName()) ) {
        DBMCli_Logo oLogo;
        oLogo.PrintLogo(DBMCli_EventingBranding::sProductName + ", DB Extender", true);
        oLogo.PrintVersion(DBMCli_EventingBranding::sVersionLabel, true);
    }

    // abort, if parameters are not okay
    if( !pParameterParser->isParameterlistValid() ){
        pParameterParser->printUsage(stderr);
        delete pParameterParser;
        exit(PRG_FATAL);
    }

    SAPDBErr_MessageList   oMessageList;

    // create event handler
    pEventHandler = new DBMCli_EventHandlerExtendDB(
                                   argc,
                                   argv,
                                   oMessageList);

    if( pEventHandler != NULL ) {
        // finally handle the event
        pEventHandler->HandleEvent();
        delete pEventHandler;
    }

    delete pParameterParser;
    return 0;
} // end main
