/*!
  @ingroup ActionHandling
  @file    Log_ActionDeleteAllObjects.hpp
  @author  UweH
  @brief   defines class Log_ActionDeleteAllObjects

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/
#ifndef Log_ActionDeleteAllObjects_H
#define Log_ActionDeleteAllObjects_H

#include "ggg00.h"  // tgg00_TransContext, tgg00_FileId
#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Container/Container_Vector.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_IAction.hpp"
#include "liveCache/LVC_Types.hpp"

class Log_AfterImage;
class Log_BeforeImage;
class Data_SplitSpaceWriter;
class Data_SplitSpaceReader;
 
/// Action to handle the oms request delete all objects in one container.
class Log_ActionDeleteAllObjects : public Log_IAction
{
public:
    /// Sets the object container in which all objects should be deleted.
    void SetFileno (tgg00_ObjFileNo fileno)
    {
        m_Head.fileno = fileno;
    }
	/// Retrieves the object container id this action relates to.
    tgg00_ObjFileNo GetFileno () const
    {
        return m_Head.fileno;
    }
    /// The given values for a deleted object are saved to be written to the log later.
    bool AddDeletedObject (const tgg92_KernelOid &oid,
                           const tgg91_PageRef   &version)
    {
        return m_Objects.InsertEnd(ObjectDescription(oid,version));
    }
    /// If objects are added to this action, true is returned.
    bool AddedObjectsExists()
    {
        return ! m_Objects.IsEmpty();
    }
    /// All stored objects are removed. All resources are released.
    void Reset()
    {
        m_Objects.Delete();
        m_Head.fileno     = 0;
        m_Head.numObjects = 0;
    }
    /// The minimal length of the persistent image is returned.
    static SAPDB_UInt GetMinimalLengthToCopyEntry()
    {
        return sizeof (Head);
    }
    /// The minimal length of the persistent image is returned.
    virtual SAPDB_UInt GetMinimalPersistentLength() const
    {
        return sizeof (Head);
    }
    /// The length of the persistent image is returned.
    virtual SAPDB_UInt GetPersistentLength() const;
    /// The Space must be initialized with the value given by GetPersistentLength().
    virtual void WritePersistentFormat (Data_SplitSpaceWriter &Writer,
                                        bool                  &isOK) const;
    /// All internal components are initialized with the reader.
    virtual void ReadPersistentFormat (Data_SplitSpaceReader  &Reader,
                                       SAPDBMem_IRawAllocator &Allocator,
                                       bool                   &isOK);
    /// This writes the members to the knltrace.
    virtual void WriteToTrace (const char * Title) const;
    /// The action is executed in redo context.
    void Redo (tgg00_TransContext   &Trans,
               const Log_AfterImage &AfterImage);
    /// The action is rollbacked. This is not implemented.
    void Undo (tgg00_TransContext    &Trans,
               const Log_BeforeImage &BeforeImage)
    {
        Trans.trError_gg00 = e_not_implemented;
    }
private:
    /// This is the persistent header written to the log volumes.
    struct Head
    {
        /// The container identifier
        tgg00_ObjFileNo fileno;     // 4
        /// How many objects are stored to be deleted.
        SAPDB_UInt4     numObjects; // 4
        /// constructor of a nil header.
        Head()
        {
            fileno     = 0;
            numObjects = 0;
        }
    };
    /// This describes one object to be deleted.
    struct ObjectDescription
    {
        /// the objects oid
        tgg92_KernelOid oid;     // 8
        /// the objects version 
        tgg91_PageRef   version; // 6
        /// filler
        SAPDB_UInt2     filler;  // 2
        /// constructor of nil element
        ObjectDescription()
        {
            oid.gg92SetNil();
            version.gg91SetNilRef();
            filler = 0;
        }
        /// constructor of valid element
        ObjectDescription(const tgg92_KernelOid &newoid,
                          const tgg91_PageRef   &newversion)
        {
            oid     = newoid;
            version = newversion;
            filler  = 0;
        }
    };
    /// defines a list of objects to be deleted
    typedef Container_Vector<ObjectDescription> ObjectDescriptions;
    /// this is the persistent head
    Head m_Head;
    /// this is the list of objects
    ObjectDescriptions m_Objects;
};
#endif  /* Log_ActionDeleteAllObjects_H */