# -*- test-case-name: nevow.test.test_guard -*-
# Copyright (c) 2004 Divmod.
# See LICENSE for details.

"""
Resource protection for Nevow. If you wish to use twisted.cred to protect your
Nevow application, you are probably most interested in
L{SessionWrapper}.
"""

__metaclass__ = type
__version__ = "$Revision: 1.5 $"[11:-2]

import random
import time
import md5
import warnings
import urllib

# Twisted Imports

from twisted.python import log, components
from twisted.web.util import Redirect
from twisted.internet import reactor, defer
from twisted.cred.error import Unauthorized, LoginFailed, UnauthorizedLogin
from twisted.cred.credentials import UsernamePassword, Anonymous

# Nevow imports
from nevow import inevow, url


def _sessionCookie():
    return md5.new("%s_%s" % (str(random.random()) , str(time.time()))).hexdigest()


class GuardSession(components.Componentized):
    """A user's session with a system.

    This utility class contains no functionality, but is used to
    represent a session.
    """
    def __init__(self, guard, uid):
        """Initialize a session with a unique ID for that session.
        """
        components.Componentized.__init__(self)
        self.guard = guard
        self.uid = uid
        self.expireCallbacks = []
        self.checkExpiredID = None
        self.setLifetime(60)
        self.portals = {}
        self.touch()

    # New Guard Interfaces

    def getLoggedInRoot(self):
        """Get the most-recently-logged-in avatar.
        """
        # XXX TODO: need to actually sort avatars by login order!
        if len(self.portals) != 1:
            raise RuntimeError("Ambiguous request for current avatar.")
        return self.portals.values()[0][0]

    def resourceForPortal(self, port):
        return self.portals.get(port)

    def setResourceForPortal(self, rsrc, port, logout):
        self.portalLogout(port)
        self.portals[port] = rsrc, logout
        return rsrc

    def portalLogout(self, port):
        p = self.portals.get(port)
        if p:
            r, l = p
            try: l()
            except: log.err()
            del self.portals[port]
        log.msg('Logout of portal %r'%port)

    # timeouts and expiration

    def setLifetime(self, lifetime):
        """Set the approximate lifetime of this session, in seconds.

        This is highly imprecise, but it allows you to set some general
        parameters about when this session will expire.  A callback will be
        scheduled each 'lifetime' seconds, and if I have not been 'touch()'ed
        in half a lifetime, I will be immediately expired.
        """
        self.lifetime = lifetime

    def notifyOnExpire(self, callback):
        """Call this callback when the session expires or logs out.
        """
        self.expireCallbacks.append(callback)

    def expire(self):
        """Expire/logout of the session.
        """
        log.msg("expired session %s" % str(self.uid))
        del self.guard.sessions[self.uid]

        # Logout of all portals
        for portal in self.portals.keys():
            self.portalLogout(portal)

        for c in self.expireCallbacks:
            try:
                c()
            except:
                log.err()
        self.expireCallbacks = []
        if self.checkExpiredID:
            self.checkExpiredID.cancel()
            self.checkExpiredID = None

    def touch(self):
        self.lastModified = time.time()

    def checkExpired(self):
        self.checkExpiredID = None
        # If I haven't been touched in 15 minutes:
        if time.time() - self.lastModified > self.lifetime / 2:
            if self.guard.sessions.has_key(self.uid):
                self.expire()
            else:
                log.msg("no session to expire: %s" % str(self.uid))
        else:
            log.msg("session given the will to live for %s more seconds" % self.lifetime)
            self.checkExpiredID = reactor.callLater(self.lifetime,
                                                    self.checkExpired)
    def __getstate__(self):
        d = self.__dict__.copy()
        if d.has_key('checkExpiredID'):
            del d['checkExpiredID']
        return d

    def __setstate__(self, d):
        self.__dict__.update(d)
        self.touch()
        self.checkExpired()


def urlToChild(request, orig, *ar, **kw):
    c = '/'.join(ar)
    if orig[-1] == '/':
        # this SHOULD only happen in the case where the URL is just the hostname
        ret = orig + c
    else:
        ret = orig + '/' + c
    if request.method == 'POST':
        args = {}
    else:
        args = request.args.copy()
    args.update(kw)
    if args:
        ret += '?'+urllib.urlencode(args, 1)
    return ret


SESSION_KEY = '__session_key__'
LOGIN_AVATAR = '__login__'
LOGOUT_AVATAR = '__logout__'


def nomind(*args): return None


class SessionWrapper:

    __implements__ = inevow.IResource

    sessionLifetime = 3600

    sessionFactory = GuardSession

    def __init__(self, portal, cookieKey=None, mindFactory=None):
        self.portal = portal
        if cookieKey is None:
            cookieKey = "woven_session_" + _sessionCookie()
        self.cookieKey = cookieKey
        self.sessions = {}
        if mindFactory is None:
            mindFactory = nomind
        self.mindFactory = mindFactory
        # Backwards compatibility; remove asap
        self.resource = self

    def renderHTTP(self, ctx):
        request = inevow.IRequest(ctx)
        prepath = request.prePathURL()
        request.setupSession = lambda : self.createSession(request, prepath, segments=[])

        d = defer.maybeDeferred(self._delegate, ctx, [])
        def _cb((resource, segments), ctx):
            assert not segments
            res = inevow.IResource(resource)
            return res.renderHTTP(ctx)
        d.addCallback(_cb, ctx)
        return d

    def locateChild(self, ctx, segments):
        request = inevow.IRequest(ctx)
        path = segments[0]
        cookie = request.getCookie(self.cookieKey)
        prepath = request.prePathURL()
        request.setupSession = lambda : self.createSession(request, prepath, segments)
        
        if path.startswith(SESSION_KEY):
            key = path[len(SESSION_KEY):]
            if key not in self.sessions:
                return Redirect(urlToChild(request, prepath, *segments[1:], **{'__start_session__':1})), ()
            self.sessions[key].setLifetime(self.sessionLifetime)
            if cookie == key:
                # /sessionized-url/${SESSION_KEY}aef9c34aecc3d9148/foo
                #                  ^
                #                  we are this getChild
                # with a matching cookie
                self.sessions[key].sessionJustStarted = True
                return Redirect(urlToChild(request, prepath, *segments[1:], **{'__session_just_started__':1})), ()
            else:
                # We attempted to negotiate the session but failed (the user
                # probably has cookies disabled): now we're going to return the
                # resource we contain.  In general the getChild shouldn't stop
                # there.
                # /sessionized-url/${SESSION_KEY}aef9c34aecc3d9148/foo
                #                  ^ we are this getChild
                # without a cookie (or with a mismatched cookie)
                request.session = self.sessions[key]
                return self.checkLogin(request, segments[1:],
                                       sessionURL=segments[0])
        else:
            # /sessionized-url/foo
            #                 ^ we are this getChild
            # with or without a session
            return self._delegate(ctx, segments)

    def _delegate(self, ctx, segments):
        """Delegate renderHTTP or locateChild to wrapped resource."""
        request = inevow.IRequest(ctx)
        cookie = request.getCookie(self.cookieKey)
        prepath = request.prePathURL()

        if cookie in self.sessions:
            # with a session
            s = request.session = self.sessions[cookie]
            return self.checkLogin(request, segments)
        else:
            # without a session

            # support HTTP auth, no redirections
            userpass = request.getUser(), request.getPassword()
            if userpass != ('',''):
                if self.sessions.has_key(userpass):
                    s = request.session = self.sessions[userpass]
                    return self.checkLogin(request, segments)
                else:
                    sz = self.sessions[userpass] = self.sessionFactory(self, userpass)
                    sz.checkExpired()
                    creds = UsernamePassword(*userpass)
                    mind = self.mindFactory(request, creds)
                    def afterLogin((iface, rsrc, logout)):
                        return (sz.setResourceForPortal(rsrc, self.portal, logout), segments)
                    def errorLogin(fail):
                        sz.expire()
                    return self.portal.login(creds, mind, inevow.IResource).addCallback(
                        afterLogin).addErrback(errorLogin)

            # no, really, without a session
            ## Redirect to the URL with the session key in it, plus the segments of the url
            rd = self.createSession(request, prepath, segments)
            return rd, ()
    
    def createSession(self, request, prepath, segments):
        """Create a new session for this request, and redirect back to the path
        given by segments."""
        
        newCookie = _sessionCookie()
        request.addCookie(self.cookieKey, newCookie, path="/")
        sz = self.sessions[newCookie] = self.sessionFactory(self, newCookie)
        sz.args = request.args
        sz.fields = getattr(request, 'fields', {})
        sz.content = request.content
        sz.method = request.method
        sz.received_headers = request.received_headers
        sz.checkExpired()
        return Redirect(urlToChild(request, prepath, SESSION_KEY+newCookie, *segments))
        
    def checkLogin(self, request, segments, sessionURL=None):
        root = url.URL.fromRequest(request)
        if sessionURL is not None:
            root = root.child(sessionURL)
        request.rememberRootURL(str(root))

        s = request.getSession()
        spoof = False
        if getattr(s, 'sessionJustStarted', False):
            del s.sessionJustStarted
            spoof = True
        if getattr(s, 'justLoggedIn', False):
            del s.justLoggedIn
            spoof = True
        if spoof:
            request.args = s.args
            request.fields = s.fields
            request.content = s.content
            request.method = s.method
            request.received_headers = s.received_headers

        if segments and segments[0] == LOGIN_AVATAR:
            return self.login(request, s, self.getCredentials(request), segments[1:])
        elif segments and segments[0] == LOGOUT_AVATAR:
            s.portalLogout(self.portal)
            return Redirect("."), ()
        else:
            r = s.resourceForPortal(self.portal)
            if r:
                ## Delegate our getChild to the resource our portal says is the right one.
                return r[0], segments
            else:
                return self.login(request, s, Anonymous(), segments, anonymous=True)

    def getCredentials(self, request):
        username = request.args.get('username', [''])[0]
        password = request.args.get('password', [''])[0]
        return UsernamePassword(username, password)

    def login(self, request, session, credentials, segments, anonymous=False):
        mind = self.mindFactory(request, credentials)
        session.mind = mind
        return self.portal.login(credentials, mind, inevow.IResource).addCallback(
            self.cbLoginSuccess, request, segments, anonymous
        ).addErrback(
            self.ebLoginError, request, segments
        )

    def cbLoginSuccess(self, (iface, res, logout), request, segments, anonymous):
        session = request.getSession()
        session.setResourceForPortal(res, self.portal, logout)
        if not anonymous:
            session.justLoggedIn = True
            u = url.URL.fromString(request.getRootURL())
            for seg in segments:
                u = u.child(seg)
            return u, ()
        return res, segments

    def ebLoginError(self, error, request, segments):
        error.trap(UnauthorizedLogin)
        error.printTraceback()
        refr = ((request.getHeader("referer") or '/')).split('?')[0]+'?'+urllib.urlencode({'login-failure': 'Incorrect login.'})
        return Redirect(refr), ()
