#! /bin/sh
cat <<EOF
program pi;

(* +------------------------------------------------------------------------+
   |                                                                        |
   |                    Calcul de Pi, formule de Ramanujan                  |
   |                                                                        |
   +------------------------------------------------------------------------+ *)

(* M. Quercia, le 18/08/2001 *)
(* cf. "The Caml Numbers Reference Manual", Inria, RT-0141 *)
(* annexe A, pp. 115 et suivantes.                         *)

uses $MODULENAME;

                       (* +--------------------------+
                          |  Sommation dichotomique  |
                          +--------------------------+ *)

type xlist = record
		u,v,w : xint;
		pred  : ^xlist;
	     end;

procedure somme(prec : longint; var num,den : xint);
var etapes,i,j        : longint;
    pile,q            : ^xlist;
    a,b,c,un,deux,six : xint;
    p,alpha,beta,gamma,delta,eps,t,u,x1,x2,x3 : xint;
begin

   etapes := (prec+197) div 94;   (* nombre de termes  calculer *)
   pile   := nil;                 (* pile de rcursion *)

   a     := of_int(13591409);     (* constantes  *)
   b     := of_int(545140134);
   c     := of_string('10939058860032000');
   un    := of_int(1);
   deux  := of_int(2);
   six   := of_int(6);

   p     := of_int(0);            (* index srie *)
   alpha := of_int(1);            (* 2p + 1      *)
   beta  := of_int(1);            (* 6p + 1      *)
   gamma := of_int(5);            (* 6p + 5      *)
   delta := of_int(53360);        (* c*p^3       *)
   eps   := of_xint(a);           (* a + bp      *)
   t     := xnew;                 (* scratch     *)
   u     := xnew;                 (* scratch     *)
   x1    := xnew;                 (* scratch     *)
   x2    := xnew;                 (* scratch     *)
   x3    := xnew;                 (* scratch     *)

   for i:=1 to etapes do begin

      new(q);
      with q^ do begin u := xnew; v := xnew; w := xnew; pred := pile; end;
      pile := q;
    
      (* calcule et retranche les termes de rangs p et p+1 *)
      mul (t,       alpha,   beta);
      mul (pile^.u, t,       gamma);
      copy(pile^.v, delta); 
      copy(pile^.w, eps);   
                                
      add (p,       p,       un);
      add (alpha,   alpha,   deux);
      add (beta,    beta,    six);
      add (gamma,   gamma,   six);
      sqr (t,       p);
      mul (u,       c,       p);
      mul (delta,   t,       u);
      add (eps,     eps,     b);
                                
      mul (t,       delta,   pile^.w);
      mul (u,       pile^.u, eps);
      sub (pile^.w, t,       u);
      mul (t,       alpha,   beta);
      mul (u,       pile^.u, gamma);
      mul (pile^.u, t,       u);
      mul (pile^.v, pile^.v, delta);
                                
      add (p,       p,       un);
      add (alpha,   alpha,   deux);
      add (beta,    beta,    six);
      add (gamma,   gamma,   six);
      sqr (t,       p);
      mul (u,       c,       p);
      mul (delta,   t,       u);
      add (eps,     eps,     b);

      (* combine avec les calculs prcdents *)
      j := 1;
      while (j and i) = 0 do begin
	 q := pile; pile := q^.pred;

	 mul(t,       pile^.w, q^.v);
	 mul(pile^.w, pile^.u, q^.w);
	 add(pile^.w, pile^.w, t);
	 mul(pile^.u, pile^.u, q^.u);
	 mul(pile^.v, pile^.v, q^.v);

	 with q^ do begin xfree(u); xfree(v); xfree(w); end;
	 dispose(q);
	 j := 2*j;
      end;

   end;

   (* termine les calculs en instance *)
   q := pile; pile := q^.pred;
   while pile <> NIL do begin 

      mul(t,       pile^.w, q^.v);
      mul(pile^.w, pile^.u, q^.w);
      add(pile^.w, pile^.w, t);
      mul(pile^.v, pile^.v, q^.v);

      with q^ do begin xfree(u); xfree(v); xfree(w); end;
      dispose(q);
      q := pile; pile := q^.pred;

   end;

   (* nettoie les variables locales et retourne la fraction *)
   num := q^.v;
   den := q^.w;
   xfree(q^.u); dispose(q);
   
   xfree(a);
   xfree(b);
   xfree(c);
   xfree(un);
   xfree(deux);
   xfree(six);
   xfree(p);
   xfree(alpha);
   xfree(beta);
   xfree(gamma);
   xfree(delta);
   xfree(eps);
   xfree(t);
   xfree(u);
   xfree(x1);
   xfree(x2);
   xfree(x3);
   
end;


                 (* +--------------------------------------+
                    |  Calcule pi avec digits+2 dcimales  |
                    +--------------------------------------+ *)

procedure calc_pi(digits : longint; pgcd,print,skip,debug : boolean);
var prec, i,j : longint;
    cinq, d, num, den, t, u, x1, x2, x3 : xint;
    s  : string;
    ss : ansistring;
begin
  
   cinq := of_int(5);
   d    := of_int(640320);
   num  := xnew;
   den  := xnew;
   t    := xnew;
   u    := xnew;
   x1   := xnew;
   x2   := xnew;
   x3   := xnew;

   if debug then chrono('dbut');  
   pow(t, cinq, digits);
   if debug then chrono('puiss-5');
   prec := nbits(t) + digits;
   sqr   (t, t);
   mul   (t, d,   t);
   shiftl(t, t,   2*digits);
   sqrt  (t, t);
   if debug then chrono('sqrt');
   somme(prec,num,den);
   if debug then begin
      str(nbits(num),s);
      ss := 'srie lb=' + s;
      chrono(pchar(ss));
   end;
   if pgcd then begin
      cfrac(num,den,x1,x2,x3,num,den);
      if debug then begin
	 str(nbits(num),s);
	 ss := 'pgcd  lb=' + s;
	 chrono(pchar(ss));
      end;
   end;
   mul_1(t, t, 100);
   mul  (t, num, t);
   quo  (t, t, den);
   if debug then chrono('quotient');

   xfree(cinq);
   xfree(d);
   xfree(num);
   xfree(den);
   xfree(u);
   xfree(x1);
   xfree(x2);
   xfree(x3);

  if print then begin
     ss := string_of(t);
     if debug then chrono('conversion');

     writeln(ss[1],'.');
     i := 2;
     while ss[i] <> #0 do begin
	write(ss[i]);
	if      (i mod 250) = 1 then begin writeln; writeln; end
        else if (i mod 50)  = 1 then writeln
        else if (i mod 10)  = 1 then write('  ')
        else if (i mod 5)   = 1 then write(' ');
	if skip and ((i mod 50) = 1) then begin
	   j := (length(ss)-i) div 50 - 1;
	   if j > 0 then begin
	      writeln('... (',j,' lignes omises)');
	      i := i + 50*j;
	   end;
	end;
	i := i+1;
     end;
     if (i mod 50) <> 2 then writeln;
  end;
  xfree(t);

end;

                      {+-----------------------+
                       |  Programme principal  |
                       +-----------------------+}

var digits,i : longint;
    c        : word;
    pgcd, print, skip, debug, help : boolean;
begin
   
   digits := 100;
   pgcd   := false;
   print  := true;
   skip   := false;
   debug  := false;
   help   := false;
   
   for i:=1 to paramcount do begin
      if      paramstr(i) = '-h'       then help  := true
      else if paramstr(i) = '-d'       then debug := true
      else if paramstr(i) = '-noprint' then print := false
      else if paramstr(i) = '-skip'    then skip  := true
      else if paramstr(i) = '-gcd'     then pgcd  := true
      else if paramstr(i) = '-test'    then begin
	 digits := 1000;
	 print  := true;
	 skip   := true;
	 debug  := false;
	 pgcd   := false;
      end
      else val(paramstr(i),digits,c);
   end;
   
  if help then writeln('usage: pi [digits] [-d] [-noprint] [-skip] [-gcd]')
  else calc_pi(digits-2,pgcd,print,skip,debug);

end.
EOF
