/*
 *  Load Tcl/Tk scripts into the interpreter
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation. You find a copy of this
 *  license in the file COPYRIGHT in the root directory of this release.
 *
 *  THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL,
 *  BUT WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
 *  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *
 *  Description:
 *
 *    This small helper module loads all Tcl/Tk scripts into the Tcl
 *    interpreter at startup. The modules may also have a "dynamic"
 *    part which is only loaded when one of the specifically declared
 *    entry-point procedures is called.  Stubs for dynamic loading are
 *    automatically generated by the tcl2c converter.
 *
 *  Author: Tom Zoerner
 *
 *  $Id: loadtcl.c,v 1.7 2003/12/27 17:27:44 tom Exp tom $
 */

#define DEBUG_SWITCH DEBUG_SWITCH_EPGUI
#define DPRINTF_OFF

#include <string.h>
#include <tcl.h>
#include <tk.h>

#include "epgctl/mytypes.h"
#include "epgctl/debug.h"
#include "epgdb/epgblock.h"
#include "epgui/epgmain.h"
#include "epgui/loadtcl.h"

#include "epgtcl/mainwin.h"
#include "epgtcl/rcfile.h"
#include "epgtcl/helptexts.h"
#include "epgtcl/draw_stats.h"
#include "epgtcl/shortcuts.h"
#include "epgtcl/mclistbox.h"
#include "epgtcl/combobox.h"
#include "epgtcl/rnotebook.h"
#include "epgtcl/dlg_hwcfg.h"
#include "epgtcl/dlg_xawtvcf.h"
#include "epgtcl/dlg_ctxmencf.h"
#include "epgtcl/dlg_acqmode.h"
#include "epgtcl/dlg_dump.h"
#include "epgtcl/dlg_netsel.h"
#include "epgtcl/dlg_netname.h"
#include "epgtcl/dlg_udefcols.h"
#include "epgtcl/dlg_shortcuts.h"
#include "epgtcl/dlg_prov.h"
#include "epgtcl/dlg_filter.h"
#include "epgtcl/dlg_substr.h"
#include "epgtcl/dlg_remind.h"
#include "epgtcl/htree.h"
#ifdef USE_PRECOMPILED_TCL_LIBS
#include "epgtcl/tcl_libs.h"
#include "epgtcl/tk_libs.h"
#endif


// prior to 8.4 there's a SEGV when evaluating const scripts (Tcl tries to modify the string)
#if (TCL_MAJOR_VERSION > 8) || ((TCL_MAJOR_VERSION == 8) && (TCL_MINOR_VERSION >= 4))
# define TCL_EVAL_CONST(INTERP, SCRIPT) Tcl_EvalEx(INTERP, SCRIPT, -1, TCL_EVAL_GLOBAL)
#else
# define TCL_EVAL_CONST(INTERP, SCRIPT) Tcl_VarEval(INTERP, "uplevel #0 {", (char *) SCRIPT, "}", NULL)
#endif


// ----------------------------------------------------------------------------
// Table which holds all Tcl/Tk scripts required for the GUI
//
typedef struct
{
   const uchar * pStatic;
   const uchar * pDynamic;
   const uchar * pFileName;
} TCL_LOAD_TAB;

static const TCL_LOAD_TAB pLoadTab[] =
{
   #ifdef USE_PRECOMPILED_TCL_LIBS
   {tcl_libs_tcl_static, NULL, NULL},
   {tk_libs_tcl_static, NULL, NULL},
   #endif
   {mainwin_tcl_static, NULL, NULL},
   {helptexts_tcl_static, NULL, NULL},
   {dlg_hwcfg_tcl_static, dlg_hwcfg_tcl_dynamic, "dlg_hwcfg_tcl"},
   {dlg_ctxmencf_tcl_static, dlg_ctxmencf_tcl_dynamic, "dlg_ctxmencf_tcl"},
   {dlg_acqmode_tcl_static, dlg_acqmode_tcl_dynamic, "dlg_acqmode_tcl"},
   {dlg_dump_tcl_static, dlg_dump_tcl_dynamic, "dlg_dump_tcl"},
   {dlg_netsel_tcl_static, dlg_netsel_tcl_dynamic, "dlg_netsel_tcl"},
   {dlg_netname_tcl_static, dlg_netname_tcl_dynamic, "dlg_netname_tcl"},
   {dlg_xawtvcf_tcl_static, dlg_xawtvcf_tcl_dynamic, "dlg_xawtvcf_tcl"},
   {dlg_udefcols_tcl_static, dlg_udefcols_tcl_dynamic, "dlg_udefcols_tcl"},
   {shortcuts_tcl_static, shortcuts_tcl_dynamic, "shortcuts_tcl"},
   {dlg_shortcuts_tcl_static, dlg_shortcuts_tcl_dynamic, "dlg_shortcuts_tcl"},
   {draw_stats_tcl_static, draw_stats_tcl_dynamic, "draw_stats_tcl"},
   {dlg_prov_tcl_static, dlg_prov_tcl_dynamic, "dlg_prov_tcl"},
   {dlg_filter_tcl_static, dlg_filter_tcl_dynamic, "dlg_filter_tcl"},
   {dlg_substr_tcl_static, dlg_substr_tcl_dynamic, "dlg_substr_tcl"},
   {dlg_remind_tcl_static, dlg_remind_tcl_dynamic, "dlg_remind_tcl"},
   {mclistbox_tcl_static, mclistbox_tcl_dynamic, "mclistbox_tcl"},
   {combobox_tcl_static, combobox_tcl_dynamic, "combobox_tcl"},
   {rnotebook_tcl_static, rnotebook_tcl_dynamic, "rnotebook_tcl"},
   {htree_tcl_static, NULL, NULL},
   {NULL, NULL, NULL}
};

// ----------------------------------------------------------------------------
// Compile a Tcl script
// - called when a yet undefined procedure is referenced
// - loading dynamically prevents that all scripts have to be compiled at start-up
//
static int LoadDynamicTclScript( ClientData ttp, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
   const char * const pUsage = "Usage: C_LoadDynamicTclScript <script-name>";
   const TCL_LOAD_TAB * pModule;
   const char * pScriptName;
   const char * pScript;
   int  result;

   if ( (objc != 2) || ((pScriptName = Tcl_GetString(objv[1])) == NULL) )
   {  // parameter count is invalid
      Tcl_SetResult(interp, (char *)pUsage, TCL_STATIC);
      result = TCL_ERROR;
   }
   else
   {
      // search for the given script name in the table
      pScript = NULL;
      for (pModule = pLoadTab; pModule->pStatic != NULL; pModule++)
      {
         if ( (pModule->pFileName != 0) &&
              (strcmp(pScriptName, pModule->pFileName) == 0) )
         {
            // found -> load the dynamic part of this tcl script file
            pScript = pModule->pDynamic;
            break;
         }
      }

      if (pScript != NULL)
      {
         if (TCL_EVAL_CONST(interp, pScript) != TCL_OK)
         {
            debug1("LoadDynamic-TclScript: failed to compile '%s'", pScriptName);
            debugTclErr(interp, "eval");
         }
         result = TCL_OK;
      }
      else
         result = TCL_ERROR;
   }
   return result;
}

// ----------------------------------------------------------------------------
// Load static Tcl modules
//
void LoadTcl_Init( bool withGui )
{
   const TCL_LOAD_TAB * pModule;

   // the rc/ini file handling is required both for GUI and the daemon
   if (TCL_EVAL_CONST(interp, rcfile_tcl_static) != TCL_OK)
   {
      debugTclErr(interp, "eval rcfile script");
   }

   if (withGui)
   {
      // load the static part of all tcl script files
      for (pModule = pLoadTab; pModule->pStatic != NULL; pModule++)
      {
         if (TCL_EVAL_CONST(interp, pModule->pStatic) != TCL_OK)
         {
            debug1("LoadTcl-Init: failed to compile static script #%d", pModule - pLoadTab);
            debugTclErr(interp, "eval");
         }
      }
   }

   Tcl_CreateObjCommand(interp, "C_LoadTclScript", LoadDynamicTclScript, (ClientData) NULL, NULL);
}

