<?php
/* 	OpenDb - Open Lending Database Project
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

include_once("./functions/widgets.php");
include_once("./functions/utils.php");
include_once("./functions/http.php");

/**
	All we need to do is check if there is a matching theme/$theme/index.php
	script.
*/
function is_legal_site($site)
{
	if(file_exists("site/$site/index.php"))
		return TRUE;
	else
		return FALSE;
}

/**
	Return a list of all sites.  This is based on getting a directory
	listing of all directories underneath site/.

	Note: Only sites with a index.php will be returned in this array.
*/
function get_site_type_r()
{
	$handle=opendir('site');
	while ($file = readdir($handle))
    {
		if(!ereg("^[.]",$file) && is_dir("site/$file"))
		{
			if(is_legal_site($file))
			{
				$sitelist[] = $file;
			}
		}
	}
	closedir($handle);
    
	if(is_array($sitelist) && count($sitelist)>0)
		return $sitelist;
	else // empty array as last resort.
		return array();
}

/**
	Will reorder $site_type_rs based on the order
	of $site_plugin_order.   Both of these arrays should 
	use a numeric index, but nonetheless the index values
	are ignored anyway.
	
	Any values in $site_type_rs not found in $site_plugin_order
	will be at the end of the array returned from this function.
	
	If $site_plugin_order is an empty array then $site_type_rs
	will be returned unchanged.
*/
function get_ordered_site_type_rs($site_type_rs)
{
	global $CONFIG_VARS;
	
	if(is_not_empty_array($site_type_rs) && is_not_empty_array($CONFIG_VARS['item_add.site_plugin_order']))
	{
		$site_plugin_order = $CONFIG_VARS['item_add.site_plugin_order'];
		
		// Add any entries from $site_type_rs to site_plugin_order
		// that are not already there.  This way the next while
		// can go through the whole $site_plugin_order and build a
		// ordered copy of $site_plugin_order.
		reset($site_type_rs);
		while(list(,$value) = each($site_type_rs))
		{
			if(!in_array($value, $site_plugin_order))
				$site_plugin_order[] = $value;
		}

		reset($site_plugin_order);
		while(list(,$value) = each($site_plugin_order))
		{
			if(in_array($value, $site_type_rs))
				$new_list_of_sites[] = $value;
		}
		return $new_list_of_sites;
	}
	else
		return $site_type_rs;		
}

/**
	Will return an array of all configuration information
	
		title
		description
		image
		external_link
		links
		
	Note: links will not be defined if $item_r is passed in null.
*/
function get_site_config_r($site, $item_r = NULL)
{
	global $LANG_VARS;

	// So we can access the language variables.
	if(file_exists("./site/$site/config.php"))
		include("./site/$site/config.php");

	if(strlen($site_title)>0)
		$site_cfg['title'] = $site_title;
	else
		$site_cfg['title'] = ucfirst($site);
	
	// Existing site plugins should generally work as normal, at least
	// where language definition is concerned.
	if(strlen($lang_var['site'][$site]['description'])>0)
		$site_cfg['description'] = $lang_var['site'][$site]['description'];
	else if($site_cfg['description'] = $LANG_VARS['site'][$site]['description'])
		$site_cfg['description'] = $LANG_VARS['site'][$site]['description'];
	else
		$site_cfg['description'] = "";
		
	// Automatically add the location.
	if(strlen($site_image_icon) && file_exists("site/$site/$site_image_icon"))
		$site_cfg['image'] = "site/$site/$site_image_icon";
		
	$site_cfg['external_link'] = $site_external_link;
	
	$site_cfg['links'] = $site_item_link_rs;
	
	return $site_cfg;
}

/**
	This function will return a complete table of all links valid
	for this item_type.

	This will only include s_attribute_type.site_type's if any are
	found or the s_item_type.default_site_type if none.

	This is useful because it allows the use of a site plugin for
	generating links only, by specifying as the default site type.
*/
function get_site_links_table($item_r)
{
	global $CONFIG_VARS;
	
	$site_type_rs = fetch_site_type_rs($item_r['s_item_type']);
	if(!is_array($site_type_rs))//Default!
		$site_type_rs[] = fetch_item_type_sitetype($item_r['s_item_type']);	

	if(is_array($site_type_rs))
	{
		$site_type_rs = get_ordered_site_type_rs($site_type_rs);
		
		$table = "<table border=0 cellspacing=10><tr>";
		while (list(,$site) = each($site_type_rs))
		{
			$site_r = get_site_config_r($site, $item_r);

			// Get the primary link, which will be extended with details for each individual link.
			if($site_r['image'])
				$link_text = "<img src=\"".$site_r['image']."\" border=0 align=absmiddle valign=absmiddle title=\"".htmlspecialchars($site_r['title'])."\" alt=\"".htmlspecialchars($site_r['title'])."\">";
			else
				$link_text = "[".$site_r['title']."]";

			if(is_array($site_r['links']))
			{
				while (list(,$links) = each($site_r['links']))
				{
					// Only if item_link is valid.
    				if(strlen($links['url'])>0)
					{
						$table .= "<td class=\"footer\"><a href=\"external.php?url=".urlencode($links['url'])."&title=".urlencode($site_r['title'].": ".$links['link'])."\" target=\"_blank\">$link_text";
						if(strlen($links['link'])>0)
							$table .= " [".str_replace(" ", "&nbsp;", $links['link'])."]";
						$table .= "</a></td>";
					}
				}
			}
		}
		$table .= "</tr></table>";
		return $table;
	}
	else
		return "&nbsp;";
}

/**
*/
function get_site_input_fields($site)
{
	// So we can access the language variables.
	global $LANG_VARS;
	
	if(file_exists("./site/$site/config.php"))
		include("./site/$site/config.php");

	while (list(,$entry) = @each($site_input_fields))
	{
		if(is_array($entry))
		{
			// Existing site plugins should generally work as normal, at least
			// where language definition is concerned.
			if(is_array($lang_var['site'][$site]['input_fields']))
				$entry['title'] = $lang_var['site'][$site]['input_fields'][$entry['field']];
			else if(is_array($LANG_VARS['site'][$site]['input_fields']))
				$entry['title'] = $LANG_VARS['site'][$site]['input_fields'][$entry['field']];
			else
				$entry['title'] = "";
					
			// Now copy into temporary array, for return to caller.
			$entries[] = $entry;
		}
	}

	// We return temporary array, otherwise our addition of 'title' would not
	// be recognised.
	return $entries;
}

// backwards compatible
function get_item_input_submit_form($formname, $site, $s_item_type, $item_id, $instance_no, $parent_id, $parent_instance_no, $params)
{
	return get_site_plugin_submit_form(
				$formname, 
				$site,
				$params,
				array('s_item_type'=>$s_item_type,'item_id'=>$item_id,'instance_no'=>$instance_no,'parent_id'=>$parent_id,'parent_instance_no'=>$parent_instance_no));
}

// backwards compatible
function get_item_input_submit_href($formname, $linktext)
{
	return get_site_plugin_submit_href($formname, $linktext);
}

// backwards compatible
function get_site_plot_chooser($site, $entry, $HTTP_VARS, $exclude_vars_r, $english_plots=TRUE)
{
	return get_site_blurb_chooser(
			$site, 
			$entry, 
			'plot', // $blurb_key
			$HTTP_VARS, 
			$exclude_vars_r, 
			$english_plots);
}

/*
* Assumes the config.php file has been included by the root level site.php
*/
function get_site_attribute_mapping($site, $key, $s_item_type)
{
	global $site_attribute_map;
		
	// Work out whether we need to do any mapping.
	if(is_array($site_attribute_map) && is_array($site_attribute_map[$s_item_type]) && strlen($site_attribute_map[$s_item_type][$key])>0)
		return $site_attribute_map[$s_item_type][$key];
	else if(is_array($site_attribute_map) && is_array($site_attribute_map['DEFAULT']) && strlen($site_attribute_map['DEFAULT'][$key])>0)
		return $site_attribute_map['DEFAULT'][$key];
	else if(is_array($site_attribute_map) && strlen($site_attribute_map[$key])>0)
		return $site_attribute_map[$key];
	else //default - no mapping!
		return $key;
}

/**
	@param $params is an array of all parameters, there may actually be arrays in this as well, and
	they will be treated correctly.
*/
function get_site_plugin_submit_form($formname, $site, $params, $HTTP_VARS, $exclude_keys_r = NULL)
{
	$buf = "<form name=\"$formname\" action=\"item_input.php\" method=\"post\">".
			get_url_fields($HTTP_VARS, NULL, array_merge(array('site','op'),$exclude_keys_r)).
			"\n<input type=\"hidden\" name=\"site\" value=\"$site\">".
			"\n<input type=\"hidden\" name=\"op\" value=\"site\">";

	while (list ($key, $val) = each($params))
	{
		$name = get_site_attribute_mapping($site, $key, $HTTP_VARS['s_item_type']);
		if(is_array($val))
		{
			while (list(,$aval) = each($val))
			{
				if(strlen($aval)>0)
					$buf .= "\n<input type=\"hidden\" name=\"".$name."[]\" value=\"".htmlspecialchars($aval)."\">";
			}
		}
		else if(strlen($val)>0)
			$buf .= "\n<input type=\"hidden\" name=\"$name\" value=\"".htmlspecialchars($val)."\">";
	}
	$buf.="</form>";
	return $buf;
}

function get_site_plugin_submit_href($formname, $linktext)
{
	// Replace {link} with the actual HREF LINK.
	return "<a href=\"javascript:document.forms['$formname'].submit();\">$linktext</a>";
}

/*
* Designed to present either a Movie Plot, Book Review, CD Review, etc.
* Will proceed directly to the item_add.php form, if no blurbs available.
* 
* @param $blurb_key - Identifies the blurb attribute options.  The blurb_key is also used
* to get the blurb_description from the $LANG_VARS array.
* 
* @param $exclude_vars_r An array of all variables from $http_vars which should not
* be passed on.
*/
function get_site_blurb_chooser($site, $entry, $blurb_key, $HTTP_VARS, $exclude_vars_r, $english_blurbs=TRUE)
{
	// So we can access the language variables.
	global $LANG_VARS;

	$blurb_s_attribute_type = get_site_attribute_mapping($site, $blurb_key, $HTTP_VARS['s_item_type']);
	$blurb_prompt = fetch_s_item_type_attr_prompt($HTTP_VARS['s_item_type'], $blurb_s_attribute_type);

	// Either no blurbs at all, or blurb attribute mapping is in error.
	if(is_not_empty_array($entry[$blurb_key]) && $blurb_prompt !== FALSE)
	{
		$blurbs = $entry[$blurb_key];
		
		$blurbcount=0;
		while (list($key,$blurb) = each($blurbs))
		{
			// If $translate_plots_with_google is set to TRUE in the site plugin config file.
			global $CONFIG_VARS;
			
			// The function_exists bit allows us to include this plugin in older versions of OpenDb without any errors.
			if($english_plots && strcasecmp($language,"english")!==0 && $CONFIG_VARS['site_plugins.translate_plots_with_google'] === TRUE)
			{
				$entry[$blurb_key] = translate_with_google($blurb, $language);
			}
			else
			{
				$entry[$blurb_key] = $blurb;
			}

			$page .= get_site_plugin_submit_form($site.$blurbcount, $site, $entry, $HTTP_VARS, $exclude_vars_r).
					"\n\n<div class=\"plain\">".nl2br(htmlspecialchars($entry[$blurb_key]))."</div>".
					"\n<p><i>[".get_site_plugin_submit_href($site.$blurbcount, replace_lang_var('prompt', $blurb_prompt, $LANG_VARS['this_blurb_suits_me']))."]</i></p>";
			$page .= "\n<hr width=25% align=middle />";

			// Increment, so forms are unique.
			$blurbcount++;
		}
		
		// No plot this time.
		$entry[$blurb_key] = NULL;
		$page .= get_site_plugin_submit_form($site, $site, $entry, $HTTP_VARS, $exclude_vars_r).
				"\n<p><i>[". get_site_plugin_submit_href($site, replace_lang_var('prompt', $blurb_prompt, $LANG_VARS['no_blurb']))."]</i></p>";
	}
	else // If no blurbs proceed automatically.
	{			
		echo "<br />".
			get_site_plugin_submit_form($site, $site, $entry, $HTTP_VARS, $exclude_vars_r)
			."\n".
			replace_lang_var("url", "javascript:document.forms['$site'].submit();", $LANG_VARS['site_page_redirect']);
			
		// Lets move along...
		echo("\n<script language=\"JavaScript\">document.forms['$site'].submit();</script>");
	}
	return $page;
}

// Utilities

/**
	Common formatting function.

	@param $lookup_table
				Should be an array of the following format: array('code'=>'text')
				Where 'code' is the numeric code.  If an entry is found in
				the lookup table, it will be used instead of the chr(code)
				call.
*/
function convert_html_numeric_codes($text, $lookup_table=NULL)
{
    $start_of_match = 0;
    while(preg_match("/&#([0-9]+);/", substr($text,$start_of_match), $matches))
    {
        $start_of_match = strpos($text, $matches[0], $start_of_match);
        $end_of_match = $start_of_match + strlen($matches[0]);

		if(is_array($lookup_table) && strlen($lookup_table[$matches[1]])>0)
			$char = $lookup_table[$matches[1]];
		else
			$char = chr($matches[1]);

        $text = substr($text, 0,$start_of_match).
				$char.
                substr($text,$end_of_match);
    }

    return $text;
}

/*
* From php manual
*/
function unhtmlentities($string)
{
	$trans_tbl = get_html_translation_table(HTML_ENTITIES);
	$trans_tbl = array_flip($trans_tbl);
	return strtr($string, $trans_tbl);
}

//
// ###### EXPERIMENTAL ######
//

/**
	We have to convert the OpenDb language specifiers to 
	Google equivalents.
*/
$translate_with_google_lang_map
				 = array(
					'german'=>'de',
					'spanish'=>'es',
					'french'=>'fr',
					'italian'=>'it',
					'portuguese'=>'pt');

/*
	Will translate english text to $to_lang using the Google
	language site.

	@param $text
	@param $language	This value should be the same as that
				used in include/config.php $CONFIG_VARS['site.language']
				but it does not have to be, as long as the
				variable is specified in the same manner (The
				full name of the language, rather than an 
				abbreviation)
*/
function translate_with_google($text, $language)
{
	// from config.php
	global $CONFIG_VARS;
	global $translate_with_google_lang_map;

	$snoopy = new Snoopy;
	
	// if no proxy
	if ($CONFIG_VARS['proxy_server.enable']==TRUE)
	{
		$snoopy->proxy_host = $CONFIG_VARS['proxy_server.host'];
		$snoopy->proxy_port = $CONFIG_VARS['proxy_server.port'];
		$snoopy->user = $CONFIG_VARS['proxy_server.userid'];
		$snoopy->pass = $CONFIG_VARS['proxy_server.password'];
	}

	if(strlen($translate_with_google_lang_map[$language])>0)
	{
		$formvars["langpair"] = "en|".$translate_with_google_lang_map[$language];
		$formvars["text"] = $text;

		if($snoopy->submit("http://translate.google.com/translate_t", $formvars) && $snoopy->status == 200)
		{
			if(preg_match("/<textarea name=q.+?>(.+?)<\/textarea>/is", $snoopy->results, $regs))
			{
				return $regs[1];
			}
		}	
		else
		{
			opendb_log("site.php::translate_with_google() failed to open $url (".$snoopy->error.")");
		}
	}
	else
	{
		opendb_log("site.php::translate_with_google() could not translate to specified language (language=$language)");
	}
	
	//else - return original text as last resort.
	return $text;
}
?>
