<?php
/* 	OpenDb - Open Lending Database Project
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

// This must be first - includes config.php
require_once("./include/begin.inc.php");

include_once("./include/language.php");
include_once("./include/theme.php");

include_once("./functions/function.php");
include_once("./functions/item_attribute.php");
include_once("./functions/widgets.php");
include_once("./functions/utils.php");
include_once("./functions/parseutils.php");
include_once("./functions/item_type.php");
include_once("./functions/item_input.php");
include_once("./functions/item.php");
include_once("./functions/user.php");
include_once("./functions/status_type.php");

/*
 See docs/notes/csv_import_error_codes.txt for details of the 
 error codes produced by this script for undefined and usually
 unlikely error states.
*/

/**
	Check if line only contains spaces and the delimiter.  This is
	to support lines in csv converted from xls that contain blank rows.
*/
function is_line_empty($line, $delimiter)
{
	for($i=0; $i<strlen($line); $i++)
	{
		// No need to test any whitespace that ends line, as this is only a line.
		if($line[$i]!== ' ' && $line[$i]!== '\t' && $line[$i] !== $delimiter){
			return FALSE;
		}
	}

	//else
	return TRUE;
}

/**
	Get rid of any surrounding quotes.  Will check for " only!
	Must have a start and end " for the match to occur!
*/
function trim_quotes($value)
{
	if(strlen($value)>=2 && substr($value,0,1) == "\"" && substr($value,-1) == "\"")
		return substr($value,1,-1);
	else
		return $value;
}

/**
	Will return FALSE if file type invalid, or will return file
	type.
*/
function get_file_type($uploadfilename)
{
	if(eregi("[^.]+\.csv",$uploadfilename))
		return "csv";
	else if(eregi("[^.]+\.dif",$uploadfilename))
		return "dif";
	else
		return FALSE;
}

/**
	Generic functions, based on type,
	which in our case will be the extension.
*/
function read_header($type, $ufile)
{
	switch ($type)
	{
		case "csv":
			// Do not need to do anything here.
			return TRUE;
		case "dif":
			return read_dif_header($ufile);
		default://not supported.
			return FALSE;
	}
}

function read_row($type, $ufile, $delimiter)
{
	switch ($type)
	{
		case "csv":
			return read_csv_row($ufile, $delimiter);
			// Do not need to do anything here.
		case "dif":
			return read_dif_row($ufile);
		default://not supported.
			return FALSE;
	}
}

/**
	You should assume that if this function 
	returns TRUE, that the $file pointer
	is now at BOT (Beginning of Tuple)
*/
function read_dif_header($ufile)
{
	// read first line.
	while(!feof($ufile))
	{
		$topic = trim(fgets($ufile, 50));
		// Ignore next two lines - info and value
		trim(fgets($ufile, 50));
		trim(fgets($ufile, 50));

		if($topic == "DATA")
		{
			//read "0,1" and ignore!
			fgets($ufile, 50);
			return TRUE;
		}
	}

	return FALSE;
}

/**
	Will return an array corresponding to a row.
	If this function returns FALSE, the $file
	has ended.

	Limit of 4096 per CELL!

	Note:
		Excel does not convert '-' or '\'' properly.
*/
function read_dif_row($ufile)
{
	if(feof($ufile) || trim(fgets($ufile, 4096))!=="BOT")
		return FALSE;

	while(!feof($ufile))
	{   
		$line1 = trim(fgets($ufile, 4096));

		// Now lets process it!
		if($line1 == "-1,0")
			break;//break out of while
		else
		{
			$line2 = trim(fgets($ufile, 4096));
			if(substr($line1,0,2) == "0,")
			{
				if($line2 == "V")
					$data = substr($line1,2);
				else
					$data = "";//not available or error!
			}
			else if($line1 == "1,0")
			{
				$data = trim_quotes($line2);
				// Get rid of any repeated quotes!
				$data = str_replace("\"\"", "\"", $data);
			}
			else
				$data = "";//error

			$row[] = $data;
		}
	}
	return $row;
}

/**
	Will read a CSV row from $ufile.  When no
	more records left, will return FALSE.
	
	Limit 10000 per row.
*/
function read_csv_row($ufile, $delimiter)
{
	if(feof($ufile))
		return FALSE;
    else
        return fgetcsv($ufile, 10000, $delimiter);
}

/**
	We need to provide a default field for all items.
*/
function get_table_field($prompt, $field, $field_default, $initcap_field)
{
	return "\n<tr>".format_prompt($prompt).format_data(NULL, $field).format_data(NULL, $field_default).format_data(NULL, $initcap_field, "center")."</tr>";
}

/**
	Will attempt to get the value of the fieldname, via the
	$tokens array and any $fieldname_default GLOBAL.
*/
function get_field_value($HTTP_VARS, $fieldname, $s_attribute_type, $tokens)
{
	if(isset($HTTP_VARS[$fieldname]) && is_numeric($HTTP_VARS[$fieldname]) && strlen($tokens[(int)$HTTP_VARS[$fieldname]])>0)
	{
		// Only support INITCAP of actual tokens imported from CSV/DIF file!!!
		if($HTTP_VARS[$fieldname."_initcap"] == "true")
			return initcap($tokens[(int)$HTTP_VARS[$fieldname]]);
		else
			return $tokens[(int)$HTTP_VARS[$fieldname]];
	}
	else if(isset($HTTP_VARS[$fieldname."_default"]))
	{
		// If a $s_attribute_type, then we have to format in a specific way.
		if(strlen($s_attribute_type)>0)
			return get_item_attr_col_value($s_attribute_type, $HTTP_VARS[$fieldname."_default"]);
		else
			return $HTTP_VARS[$fieldname."_default"];
	}
	else // no $value to return
		return FALSE;
}

/**
	Format of single_select is:
		display, value
*/
function get_column_select_block($fieldname, $lookup_array, $selectedindex)
{
	// So we can access the language variables.
	global $LANG_VARS;

	$var="\n<select name=\"$fieldname\">";
	$var.="\n<option value=\"\">-- ".$LANG_VARS['none']." --";
	for ($i=0; $i<count($lookup_array); $i++)
		$var .= "\n<option ".(is_numeric($selectedindex) && $selectedindex==$i?"SELECTED":"")." value=\"$i\">".$lookup_array[$i];

	$var.="\n</select>";
	return $var;
}

/**
	A value of NULL for $checked indicates that the field should be disabled.
*/
function get_initcap_field($HTTP_VARS, $fieldname)
{
	return get_input_field($fieldname."_initcap",
					NULL, 
					NULL, 
					"checkbox(true,false,)",
					"N", 
					ifempty($HTTP_VARS[$fieldname."_initcap"],"true"),
					FALSE);
}

/**
	We are using this instead of the particular input types, so
	that users can clearly see what the %value% and %display%
	columns are for the lookups, so they can match them in
	their CSV file.
*/
function build_select($name, $lookup_results, $value, $include_none_option=TRUE)
{	
	global $LANG_VARS;
	
	// If at least one $lookup_r['value'] is different from $lookup_r['display'] set this variable to true.
	$display_used=FALSE;
	
	$var="\n<select name=\"$name\">";
	if($include_none_option)
		$var.="\n<option value=\"\">-- ".$LANG_VARS['none']." --";
		
	while($lookup_r = mysql_fetch_array($lookup_results))
	{
		if($lookup_r['value'] === $lookup_r['display'])
			$display = $lookup_r['value'];
		else{
			$display = $lookup_r['value']." - ".$lookup_r['display'];
			if(!$display_used)
				$display_used=TRUE;
		}

		$var .= "\n<option value=\"".$lookup_r['value']."\"";

		// Only support checked_ind, where we are not displaying a None option.  If None option, it should
		// be chosen by default.		
		if( ($include_none_option!==TRUE && strlen($value)==0 && $lookup_r['checked_ind']=="Y") || (strlen($value)>0 && strcasecmp($value,$lookup_r['value'])===0))
			$var .= " SELECTED";
		
		$var .= ">$display";
	}
	mysql_free_result($lookup_results);
	
	$var.="\n</select>";

	if($display_used)
		$var.="<br>(%value% - %display%)";
	else
		$var.="<br>(%value%)";
	return $var;
}

/**
	@param s_item_type	The item type of the insert data.

	@param tokens		Tokens for CSV, indexed by integer, which will then be used
						get the values based on the fieldname value, which will be a numeric
						index value.

	@param http_vars	Is the HTTP_POST_VARS array (But we actually pass in $HTTP_VARS which is
						copy of $HTTP_POST_VARS)

	@param trialrun 	If true, the function should only process the row and display everything
						it would do, but do not actually perform the insert.  This provides users
						with a way to check how their csv file will be processed without affecting
						the database...
*/
function insert_item_record($tokens, $HTTP_VARS, $rownum, $trialrun, $color)
{
	global $LANG_VARS;
	global $CONFIG_VARS;

	// get the title.
	$title_attr_type_r = fetch_sfieldtype_item_attribute_type_r($HTTP_VARS['s_item_type'], 'TITLE');
	$title = get_field_value($HTTP_VARS, get_field_name($title_attr_type_r['s_attribute_type'], $title_attr_type_r['order_no']), NULL, $tokens);
	if(strlen($title))
	{
		if($HTTP_VARS['title_initcap'] == "true")
			$title = initcap($title);

		echo("\n<tr><td class=\"$color\" align=center>$rownum</td><td class=\"$color\">$title</td>");
		
		// If duplicate titles allowed, then ignore title_exists
		if($HTTP_VARS['ignore_title_exists'] == 'Y' || !is_exists_title($title, $HTTP_VARS['s_item_type'], $HTTP_VARS['owner_id']))
		{
			// get the category.
			$category_attr_type_r = fetch_sfieldtype_item_attribute_type_r($HTTP_VARS['s_item_type'], 'CATEGORY');
			$category = get_field_value($HTTP_VARS, get_field_name($category_attr_type_r['s_attribute_type'], $category_attr_type_r['order_no']), $category_attr_type_r['s_attribute_type'], $tokens);
			
			// Replace the '/' or '\' separator with a space instead.
			$category = str_replace(array('/','\\'), ' ', $category);

			$status_attr_type_r = fetch_sfieldtype_item_attribute_type_r($HTTP_VARS['s_item_type'], 'STATUSTYPE');
			$s_status_type = get_field_value($HTTP_VARS, get_field_name($status_attr_type_r['s_attribute_type'], $status_attr_type_r['order_no']), $status_attr_type_r['s_attribute_type'], $tokens);
			
			$status_type_r = fetch_status_type_r($s_status_type);
			if(is_newitem_status_type_valid($HTTP_VARS['owner_id'], $status_type_r, $errors))
			{
				$new_id = 0;
				if($trialrun!="true")
				{
					$new_id = insert_item(NULL, $HTTP_VARS['s_item_type'], $title, $category, $HTTP_VARS['owner_id']);
	   				if($new_id !== FALSE)
					{
						$duration_attr_type_r = fetch_sfieldtype_item_attribute_type_r($HTTP_VARS['s_item_type'], 'DURATION');
						$duration_val = get_field_value($HTTP_VARS, get_field_name($duration_attr_type_r['s_attribute_type'], $duration_attr_type_r['order_no']), $duration_attr_type_r['s_attribute_type'], $tokens);

						$status_cmnt_attr_type_r = fetch_sfieldtype_item_attribute_type_r($HTTP_VARS['s_item_type'], 'STATUSCMNT');
						$status_comment = get_field_value($HTTP_VARS, get_field_name($status_cmnt_attr_type_r['s_attribute_type'], $status_cmnt_attr_type_r['order_no']), $status_cmnt_attr_type_r['s_attribute_type'], $tokens);
					
						$instance_no = insert_newitem_instance($new_id, $s_status_type, $status_comment, $duration_val, $HTTP_VARS['owner_id']);
						if($instance_no===FALSE)
							$new_id = FALSE;
					}
				}

				// item AND item_instance successfully added!
				if($new_id !== FALSE)
				{
					echo("\n\t<td class=\"$color\" align=center>".($trialrun!="true"?$new_id:"?")."</td><td align=center class=\"$color\">".ifempty($category,"&nbsp;")."</td><td class=\"$color\">");	
   					// We need to update all the attributes as well. Do not need full records.
					$results = fetch_item_attribute_type_rs($HTTP_VARS['s_item_type'], TRUE);
					if($results)
					{	
						$found=FALSE;
						while($item_attribute_type_r = mysql_fetch_array($results))
						{
							if($item_attribute_type_r['s_field_type'] != 'CATEGORY' && 
										$item_attribute_type_r['s_field_type'] != 'TITLE' && 
										$item_attribute_type_r['s_field_type'] != 'STATUSTYPE' && 
										$item_attribute_type_r['s_field_type'] != 'STATUSCMNT' && 
										$item_attribute_type_r['s_field_type'] != 'DURATION' &&
										$item_attribute_type_r['s_field_type'] != 'ITEM_ID')
							{
								$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']);
								$value = get_field_value($HTTP_VARS, $fieldname, $item_attribute_type_r['s_attribute_type'], $tokens);

								// Only do so, if field actually has a value.
								if(strlen($value)>0)
								{
									$found=TRUE;
									echo("<div class=\"fineprint\">* ".$item_attribute_type_r['s_attribute_type'].": \"".htmlspecialchars($value)."\"</div>");
									if($trialrun!="true")
										insert_item_attribute($new_id, $item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no'], $value, $HTTP_VARS['s_item_type']);
								}
							}
						}
						//So we avoid ugly space in Netscape 4.x
						if(!$found)	echo("&nbsp;");

						mysql_free_result($results);
					}//if($results)
					else
						echo("&nbsp;");
					echo("</td>");
				}
				else //if($new_id !== FALSE)
				{
					echo("\n\t<td class=\"$color\" colspan=3><div class=\"smerror\">".$LANG_VARS['item_not_added']." [".mysql_error()."]</div></td>");
				}
			}
			else//if(is_newitem_status_type_valid($owner_id, $new_status_type_r, $errors))
			{
				echo("\n\t<td class=\"$color\" colspan=3><div class=\"smerror\">".$LANG_VARS['item_not_added']."<i>".format_error_block($errors)."</i></div></td>");
			}
		}
		else
		{
			echo("\n\t<td class=\"$color\" colspan=3 align=center><div class=\"smerror\">".$LANG_VARS['title_exists']."</div></td>");
		}
		echo("\n</tr>");
	}
	else
		echo ("\n<tr><td class=\"$color\" align=center>$rownum</td><td class=\"$color\" colspan=4 align=center><div class=\"smerror\">".$LANG_VARS['specify_title']."</div></td></tr>");
	
	// If failure, $new_id will not be defined.
	if(isset($new_id))
  		return $new_id;
	else
		return FALSE;
}

session_start();
if (is_opendb_valid_session())
{
	// Only admin user or owner user allowed to access this.
	if( ($HTTP_VARS['owner_id'] == $HTTP_SESSION_VARS['user_id'] && is_user_normal($HTTP_SESSION_VARS['user_id'],$HTTP_SESSION_VARS['user_type']) ) || is_user_admin($HTTP_SESSION_VARS['user_id'],$HTTP_SESSION_VARS['user_type']))
	{
		if(is_file_upload_enabled())
		{
			// Set time limit to 10 minutes maximum.
			@set_time_limit(600);
		
   			$showform=FALSE;
			
			// If s_item_type defined, it should be legal.
			if(strlen($HTTP_VARS['s_item_type'])==0 || is_valid_item_type_structure($HTTP_VARS['s_item_type']))
			{
				if($HTTP_VARS['op'] == "upload" || $HTTP_VARS['op'] == "uploaded")
				{
					$previewprocess=FALSE;
			
					// No error means process succeeded.
					$error_no = 0;

					// The HTTP_POST_FILES will be empty if $op == "uploaded", because we already have
					// the file on the server.
					if($HTTP_VARS['op'] == "upload" && strlen($HTTP_POST_FILES['uploadfile']['name'])==0)
					{
						$previewprocess=TRUE;
					}				
					else
					{
						if(strlen($HTTP_VARS['s_item_type'])>0 && strlen($HTTP_VARS['delimiter'])>0)
						{
							// Get the tempname here so we can do all tests in the one if...
							// In the case of $op == "uploaded", $csvfile will be defined...
							if($HTTP_VARS['op'] == "uploaded" && strlen($HTTP_VARS['ext_type'])>0)
							{
								// we do not check that the file exists here, we will check in a minute.
								$tempname = $HTTP_VARS['csvfile'];
								$ext_type = $HTTP_VARS['ext_type'];
							}
							else 
							{
								if(is_uploaded_file($HTTP_POST_FILES['uploadfile']['tmp_name']))//only move if not already uploaded!
								{
									if($HTTP_POST_FILES['uploadfile']['size']>0)
									{
										// If the upload file does not have a correct extension, we will not move it,
										// but it should be removed automatically anyway.
										$ext_type = get_file_type($HTTP_POST_FILES['uploadfile']['name']);
										if($ext_type!==FALSE)
										{
											$tempname = tempnam(NULL, "csvfile");
											if(!move_uploaded_file($HTTP_POST_FILES['uploadfile']['tmp_name'], $tempname))
												$error_no = 3; // could not move upload file.
										}
										else
											$error_no = 2; // invalid extension.
									}
									else
										$error_no = 5; // zero length file.
								}
								else
									$error_no = 4; // no upload file.
							}
						}
						else
							$error_no = 1; // invalid s_item_type or delimiter.
					}
				
					// Now see what has happened.
					if($error_no===0 || $previewprocess)
					{
						$page_title = replace_lang_var("type", strtoupper($HTTP_VARS['type']), $LANG_VARS['type_import']);
						echo _theme_header($page_title);
						echo ("<h2>".$page_title."</h2>");
				
						if($previewprocess)
							echo("<h3>".replace_lang_var('type', 'CSV', $LANG_VARS['import_type_column_mapping_preview'])." ".get_item_image($HTTP_VARS['s_item_type'])."</h3>");
						else
						{
							echo("<h3>".replace_lang_var('type', 'CSV', $LANG_VARS['import_type_column_mapping'])." ".get_item_image($HTTP_VARS['s_item_type'])."</h3>");
					
							//Now we are going to process the file
							$fp = fopen($tempname, "r");
							if($fp)
							{
								// Get rid of any header info in file, that we do not need!
								// In this case $ext_type is actually locally defined, even if copy
								// of HTTP_VARS['ext_type']
								if(read_header($ext_type, $fp))//csv does not have any, DIF does!
								{
									// Read first line
									$tokens = read_row($ext_type, $fp, $HTTP_VARS['delimiter']);
								}
								else
									echo("\n<div class=\"error\">".$LANG_VARS['undefined_error']." (err=7)</div>");
								// Close file.
								fclose($fp);
							}
							else
								echo("\n<div class=\"error\">".$LANG_VARS['undefined_error']." (err=6)</div>");
						}

						// The first line found.
						if($previewprocess || is_not_empty_array($tokens))
						{
							echo("\n<form action=\"$PHP_SELF\" method=\"post\">");
							echo("\n<input type=\"hidden\" name=\"type\" value=\"csv\">");
							echo("\n<input type=\"hidden\" name=\"ext_type\" value=\"$ext_type\">");
							echo("\n<input type=\"hidden\" name=\"op\" value=\"process\">");
							echo("\n<input type=\"hidden\" name=\"s_item_type\" value=\"".$HTTP_VARS['s_item_type']."\">");
							echo("\n<input type=\"hidden\" name=\"delimiter\" value=\"".$HTTP_VARS['delimiter']."\">");
							echo("\n<input type=\"hidden\" name=\"csvfile\" value=\"$tempname\">");
	
							echo("\n<table border=0 frameborder=0 cellspacing=1>");
					
							// If the prompt for this field, matches any of the $tokens, then select that token.
							if(!is_numeric($HTTP_VARS['typefield']))
							{
								for($i=0; $i<count($tokens); $i++)
								{
									if($tokens[$i] == 's_item_type' || $tokens[$i] == $LANG_VARS[ 's_item_type'])
									{
										$HTTP_VARS['typefield'] = $i;
										break;
									}
								}
							}
							
							// If your data includes the type, then specify the column, so we
							// can ignore records that are not of the chosen type.
							echo(format_field($LANG_VARS['item_type_column'],
									NULL,
									get_column_select_block("typefield", $tokens, $HTTP_VARS['typefield'])));

							// No user field required if owner_id specified
							// We have already checked that $HTTP_SESSION_VARS['user_id'] is same as owner_id
							if(strlen($HTTP_VARS['owner_id'])==0)
							{
								$results = fetch_user_rs(get_owner_user_types_r());
								echo(format_field($LANG_VARS['owner'], NULL, custom_select('owner_id', $results, '%fullname% (%user_id%)', 1, NULL, 'user_id')));
							}
							else
							{
								if($HTTP_VARS['owner_id'] !== $HTTP_VARS['user_id'])
								{
									echo(format_field($LANG_VARS['owner'], NULL, "<div class=\"data\">".fetch_user_name($HTTP_VARS['owner_id'])."</div>"));
								}							
								echo("\n<input type=\"hidden\" name=\"owner_id\" value=\"".$HTTP_VARS['owner_id']."\">");
							}	
							echo(format_field($LANG_VARS['include_first_line'], NULL, "<input type=checkbox name=\"includefirstline\" value=\"Y\"".($HTTP_VARS['includefirstline']=='Y'?' CHECKED':'').">"));
							echo(format_field($LANG_VARS['ignore_title_exists'], NULL, "<input type=checkbox name=\"ignore_title_exists\" value=\"Y\"".($HTTP_VARS['ignore_title_exists']=='Y'?' CHECKED':'').">"));
  							echo("\n</table><br><br>");
	
							echo("\n<table width=100% cellspacing=1 border=0>");
							echo("\n<tr><td class=\"navbar\">&nbsp;</td><td class=\"navbar\">".$LANG_VARS['column']."</td><td class=\"navbar\">".$LANG_VARS['default']."</td><td class=\"navbar\"> ".$LANG_VARS['initcap']." </td></tr>");

							// --------- Now the main input fields start....
							$results = fetch_item_attribute_type_rs($HTTP_VARS['s_item_type'], TRUE, TRUE);
							while($item_attribute_type_r = mysql_fetch_array($results))
							{
								// purely readonly
								if($item_attribute_type_r['s_field_type'] !== 'ITEM_ID')
								{
									$widget = prc_function_spec($item_attribute_type_r['input_type']);

									$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], 
														$item_attribute_type_r['order_no']);
	
									// Need this to access http_vars array properly.
									$fieldname_default = $fieldname."_default";
										
									if($item_attribute_type_r['s_field_type'] == 'STATUSTYPE')
									{
										$lookup_results = fetch_newitem_status_type_rs(ifempty($HTTP_VARS['owner_id'], $HTTP_SESSION_VARS['user_id']));
										$default_field = build_select($fieldname_default, $lookup_results, $HTTP_VARS[$fieldname_default], FALSE); // do not include 'None' option
										// No initcap for these types of fields.
										$initcap_field = NULL;
									}
									else if($widget['type'] !== "textarea" && $widget['type'] !== "url" && $widget['type'] !== "upload_or_saveurl" && $widget['type'] !== "saveurl" && $widget['type'] !== "upload")
									{// The new upload/saveurl widgets can not work as default fields! 
									    // Also because of their nature, it does not make any sense to have a default field
										// for textarea widgets either!

										// Check if any lookup values for this field.
										$lookup_results = fetch_attribute_type_lookup_rs($item_attribute_type_r['s_attribute_type']);
										if($lookup_results)
										{
											$default_field = build_select($fieldname_default, $lookup_results, $HTTP_VARS[$fieldname_default]);
											// No initcap for these types of fields.
											$initcap_field = NULL;
										}
										else //use normal input field.
										{
											$default_field = get_input_field($fieldname_default,
														$item_attribute_type_r['s_attribute_type'], 
														$item_attribute_type_r['prompt'], 
														$item_attribute_type_r['input_type'],
														$item_attribute_type_r['compulsory_ind'], 
														$HTTP_VARS[$fieldname_default],
														FALSE);

											if($widget['type'] === "text")
 										    	$initcap_field = get_initcap_field($HTTP_VARS, $fieldname);   
											else //Only for text fields.
												$initcap_field = NULL;
										}
									}
									else
									{
										$default_field = NULL;
										$field = NULL;
										$initcap_field = NULL;
									}
									
									
									// If the prompt for this field, matches any of the $tokens, then select that token.
									if(!is_numeric($HTTP_VARS[$fieldname]))
									{
										for($i=0; $i<count($tokens); $i++)
										{
											// A direct export from CSV Export, so lets match the columns for you.
											if($tokens[$i] == $fieldname || $tokens[$i] == $item_attribute_type_r['prompt'])
											{
												$HTTP_VARS[$fieldname] = $i;
												break;
											}
										}
									}
									
									echo(get_table_field($item_attribute_type_r['prompt'],
													get_column_select_block($fieldname, $tokens, $HTTP_VARS[$fieldname]),
													$default_field,
													$initcap_field));
								}//if($item_attribute_type_r['s_field_type'] !== 'ITEM_ID')
							}
							mysql_free_result($results);

							echo("\n<tr><td colspan=3 align=center>");
							// Only display this one, if we are not on a trial run.
							if(!$previewprocess)
								echo("\n<div class=\"footer\"><input type=submit value=\"".$LANG_VARS['continue']."\">&nbsp;&nbsp;<input type=checkbox name=\"trialrun\" value=\"true\" CHECKED><i>".$LANG_VARS['trial_run']."</i></div>");
							echo("\n</td></tr>");
	
							echo("\n</form>");
							echo("\n</table>");

							echo(format_help_block($LANG_VARS['import_csv_column_mapping_help']));
						}//if(count($tokens)>0)
					}
					else//check if file uploaded correctly.
					{
						$showform=TRUE;
						$error = array('error'=>replace_lang_var('prompt', 'CSV', $LANG_VARS['file_upload_error']),'detail'=>"err=$error_no");
					}
				}
				else if ($HTTP_VARS['op'] == "process" && isset($HTTP_VARS['ext_type']) && isset($HTTP_VARS['csvfile']) && is_readable($HTTP_VARS['csvfile']) && isset($HTTP_VARS['owner_id']) && isset($HTTP_VARS['s_item_type']))
				{
					$page_title = replace_lang_var("type", strtoupper($HTTP_VARS['type']), $LANG_VARS['type_import']);
					echo _theme_header($page_title);
					echo ("<h2>".$page_title."</h2>");
				
					if($HTTP_VARS['trialrun']=="true")
						echo("\n<h3>".replace_lang_var('type','CSV', $LANG_VARS['import_type_process_file_trial'])." ".get_item_image($HTTP_VARS['s_item_type'])."</h3>");
					else
						echo("\n<h3>".replace_lang_var('type','CSV', $LANG_VARS['import_type_process_file'])." ".get_item_image($HTTP_VARS['s_item_type'])."</h3>");

					// Display the owners name if required.
					if(strlen($HTTP_VARS['owner_id'])>0 && $HTTP_VARS['owner_id'] !== $HTTP_VARS['user_id'])
					{
						echo("\n<table border=0 frameborder=0 cellspacing=1>");
						echo(format_field($LANG_VARS['owner'], NULL, fetch_user_name($HTTP_VARS['owner_id'])));
						echo("\n</table><br><br>");
					}

					echo("\n<table border=0 cellspacing=1 width=100%>");
					echo("\n<tr>
							<td class=\"navbar\">#</td>
							<td class=\"navbar\">".$LANG_VARS['title']."</td>
							<td class=\"navbar\">".$LANG_VARS['item_id']."</td>
							<td class=\"navbar\">".$LANG_VARS['category']."</td>
							<td class=\"navbar\">".$LANG_VARS['attributes']."</td>
							</tr>");
			
					$ignored = FALSE;
					$rownum = 1;
					$toggle = TRUE;
					$inserted = 0;
					$start_item_id = NULL;
			
					$fp = fopen($HTTP_VARS['csvfile'], "r");
					if($fp)
					{
						// Get rid of any header info in file, that we do not need!
						if(read_header($HTTP_VARS['ext_type'], $fp))//csv does not have any, DIF does!
						{
							while(($tokens = read_row($HTTP_VARS['ext_type'], $fp, $HTTP_VARS['delimiter']))!==FALSE)
							{
								if($toggle)
									$color="top";
								else
		 							$color="top2";
								$toggle = !$toggle;
	
								// Ignore first line if required.
								if(count($tokens)==0 || (strlen($HTTP_VARS['includefirstline'])==0 && !$ignored))
								{
									if(!$ignored)
										$ignored=TRUE;
									echo("\n<tr><td class=\"$color\" align=center>$rownum</td><td colspan=4 align=center class=\"$color\"><i>".$LANG_VARS['skipped']."</i></td></tr>");
								}	
								else
								{
									$sitemtype = get_field_value($HTTP_VARS, "typefield", NULL, $tokens);
							
									// Either no restrictions, or $type column identified by $typefield index
									// matches the http_vars['s_item_type'] we are inserting for....
									if(strlen($sitemtype)==0 || strcasecmp($sitemtype,$HTTP_VARS['s_item_type'])==0)//case insensitive match!
									{
										$new_id = insert_item_record($tokens, $HTTP_VARS, $rownum, $HTTP_VARS['trialrun'], $color);
										if($new_id !== FALSE)
										{
											// Get first item_id inserted.
											if($start_item_id===NULL)
												$start_item_id=$new_id;
											$inserted++;
										}
									}
									else
										echo("\n<tr><td class=\"$color\" align=center>$rownum</td><td colspan=4 align=center class=\"$color\"><i>".$LANG_VARS['skipped']."</i></td></tr>");
								}

								// clear out tokens.
								$tokens = array();
								$rownum++;
							}//while
						}
						else//not sure what went wrong here!!!
							echo ("\n<div class=\"error\">".$LANG_VARS['undefined_error']." (err=6)</div>");
					}
					else // cannot open file.
						echo ("\n<div class=\"error\">".$LANG_VARS['undefined_error']." (err=5)</div>");
			
					// Close file.
					fclose($fp);
	
					echo("\n<tr><td colspan=5 align=center><br>");

					if($HTTP_VARS['trialrun'] == "true")
					{
						echo("<form action=\"$PHP_SELF\" method=\"post\">");
				
						// Pass all http variables onto next instance...
						// Includes empty fields...
						echo get_url_fields($HTTP_VARS, NULL, array("op", "trialrun"));
				
						// No point going forward, unless at least one insert worked!
						// We did not specify the $op, so we could go back.
						if($inserted>0)
						{	
							echo("\n<input type=\"hidden\" name=\"op\" value=\"uploaded\">");
							echo("\n<input type=\"hidden\" name=\"trialrun\" value=\"true\">");
							echo("\n<input type=button value=\"".$LANG_VARS['back']."\" onclick=\"this.form.submit();\">&nbsp;");
							echo("\n<input type=button value=\"".$LANG_VARS['continue']."\" onclick=\"this.form.trialrun.value='false'; this.form.op.value='process'; this.form.submit();\">");
						}
						else
						{
							echo("\n<input type=\"hidden\" name=\"op\" value=\"uploaded\">");
							echo("\n<input type=\"hidden\" name=\"trialrun\" value=\"true\">");
							echo("\n<div class=\"smerror\"><br>".$LANG_VARS['no_items_added']."</div>");
							echo("\n<input type=button value=\"".$LANG_VARS['back']."\" onclick=\"this.form.submit();\">");
						}
				
						echo("</form>");
					}
					else if ($inserted>0)//Only if at least one record inserted.
					{	
						echo("\n<div class=\"footer\"><a href=\"listings.php?owner_id=".$HTTP_VARS['owner_id']."&start_item_id=$start_item_id\">[".replace_lang_var("count", $inserted, $LANG_VARS['list_imported_items'])."]</a></div>");

						//Get rid of the CSV file now!
						@unlink($HTTP_VARS['csvfile']);
					}
	
					echo("\n</td></tr></table>");
				}
				else // default
					$showform=TRUE;
			}
			else// if(is_valid_item_type_structure($HTTP_VARS['s_item_type'], $errors))
			{
				$error = array('error'=>replace_lang_var("s_item_type", $item_r['s_item_type'], $LANG_VARS['invalid_item_type_structure']),'detail'=>'');
				$showform=TRUE;
			}
		
			// Show form if no 'if test' matches.
			if($showform)
			{
				// A custom title.
				if($HTTP_VARS['owner_id'] == $HTTP_SESSION_VARS['user_id'])
					$page_title = $LANG_VARS['import_my_items'];
				else
					$page_title = $LANG_VARS['import_items'];

				echo _theme_header($page_title);
				echo("<h2>".$page_title."</h2>");
				
				if(isset($error))
					echo(format_error_block($error));
	
				echo("\n<h3>".replace_lang_var('type','CSV', $LANG_VARS['import_type_upload_file'])."</h3>");
				echo get_validation_javascript();

				echo("\n<table border=0 frameborder=0 cellspacing=1>");
				echo("\n<form name=\"main\" action=\"$PHP_SELF\" method=\"post\" enctype=\"multipart/form-data\">");
				echo("\n<input type=\"hidden\" name=\"type\" value=\"csv\">");
				echo("\n<input type=\"hidden\" name=\"owner_id\" value=\"".$HTTP_VARS['owner_id']."\">");
				echo("\n<input type=\"hidden\" name=\"op\" value=\"upload\">");
			
				echo(format_field($LANG_VARS['item_type'], NULL, single_select("s_item_type", fetch_item_type_rs(TRUE), "%value% - %display%", NULL, $HTTP_VARS['s_item_type'])));
				echo(get_input_field("uploadfile", NULL, $LANG_VARS['file'], "upload(25,*,\"csv,dif\")"));
				echo(get_input_field("delimiter", NULL, $LANG_VARS['delimiter'], "text(1,1)", "Y", ","));
		
				if($CONFIG_VARS['widgets.enable_javascript_validation']!==FALSE)
					$onclick="if(!checkForm(this.form)){return false;}else{this.form.submit();}";
				else
					$onclick="this.form.submit()";
				
				echo("\n<tr><td colspan=2><input type=button onclick=\"$onclick\" value=\"".$LANG_VARS['continue']."\">");
				echo("\n</td></tr>");
				echo("\n</form>");
				echo("\n</table>");

				echo(format_help_block($LANG_VARS['import_csv_help']));
			}
		}
		else// File upload not supported on server
		{
			echo _theme_header($LANG_VARS['file_upload_not_available']);
			echo _theme_error($LANG_VARS['file_upload_not_available']);
		}
	}
	else//not an administrator or own user.
	{
		echo _theme_header($LANG_VARS['not_authorized_to_page']);
		echo _theme_error($LANG_VARS['not_authorized_to_page']);
	}
	echo _theme_footer();
}
else//invalid session
{
	include("./include/invalidsession.php");
}
	
// Cleanup after begin.inc.php
require_once("./include/end.inc.php");
?>
