/*
 * console.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <windows.h>
#include <commctrl.h>
#include <io.h>
#include <fcntl.h>
#include <stdio.h>
#include <process.h>
#include <errno.h>
#include "console.h"


#define M(str) MessageBox(NULL, str, NULL, MB_OK);
#define MN(str, num) \
do { \
	char buf[80]; \
	sprintf(buf, "%s%d", str, num); \
	M(buf); \
} while (0)




#define MC_OUTPUT WM_USER+1

class ConsoleThread {
public:
	ConsoleThread(int handle, HWND hwnd)
		: outputReadHandle_(handle), consoleHwnd_(hwnd) { }
	~ConsoleThread() { }
	void Main();

private:
	int outputReadHandle_;
	HWND consoleHwnd_;
};


MashConsole::MashConsole(int bufferSize)
	: hInstance_(NULL), toplevel_(NULL), statusBar_(NULL), edit_(NULL),
	  buffer_(NULL), len_(0), maxLen_(0)
{
	SetBufferSize(bufferSize);
}


MashConsole::~MashConsole()
{
	if (toplevel_) Destroy();
	if (buffer_) free(buffer_);
}


BOOL
MashConsole::SetBufferSize(int size)
{
	if (size < maxLen_ && len_ > size) {
		// we need to clip some initial data
		CopyTo(buffer_, size);
	}

	char *newBuffer = (char*)realloc(buffer_, (size+1) * sizeof(char));
	if (!newBuffer) return FALSE;
	buffer_ = newBuffer;
	maxLen_ = size;
	return TRUE;
}


void
MashConsole::Init(HINSTANCE hInstance)
{
	WNDCLASSEX consoleClass;
	hInstance_ = hInstance;

	consoleClass.cbSize        = sizeof(consoleClass);
	consoleClass.style         = CS_HREDRAW | CS_VREDRAW | CS_CLASSDC;
	consoleClass.lpfnWndProc   = MainWindowProc_Static;
	consoleClass.cbClsExtra    = 0;
	consoleClass.cbWndExtra    = 0;
	consoleClass.hInstance     = hInstance_;
	consoleClass.hIcon         = LoadIcon(hInstance_, "MashIcon");
	consoleClass.hIconSm       = LoadIcon(hInstance_, "MashIcon");
	consoleClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
	consoleClass.hbrBackground = NULL;
	consoleClass.lpszMenuName  = "MashConsoleMenu";
	consoleClass.lpszClassName = MashConsoleWindow_Class;
	RegisterClassEx(&consoleClass);

	InitCommonControls();
}


void
MashConsole::Create(int x, int y, int width, int height)
{
	toplevel_ = CreateWindow(MashConsoleWindow_Class, "Mash Console",
				 WS_OVERLAPPEDWINDOW, x, y, width, height,
				 NULL, NULL, hInstance_, this);
	ShowWindow(toplevel_, SW_HIDE) ;
	UpdateWindow(toplevel_);
}


void
MashConsole::Show(BOOL show)
{
	if (!toplevel_) return;

	if (show) {
		/* auto scroll to the end */
		ShowWindow(toplevel_, SW_SHOW);
		SendMessage(edit_, EM_SETSEL, len_, len_);
		SendMessage(edit_, EM_SCROLLCARET, 0, 0);
	} else {
		ShowWindow(toplevel_, SW_HIDE);
	}
}

void
MashConsole::DoModal()
{
	MSG message;
	while (GetMessage(&message, NULL, 0, 0)) {
		TranslateMessage(&message);
		DispatchMessage(&message);
	}
}


BOOL
MashConsole::Toggle()
{
	if (IsVisible()) {
		Show(FALSE);
		return FALSE;
	} else {
		Show(TRUE);
		return TRUE;
	}
}


static void ShowError(const char *msg, BOOL isWindowsError)
{
	char buffer[256];
	if (isWindowsError) {
		sprintf(buffer, "%s:\n", msg);
		int len = sizeof(buffer) - strlen(buffer) - 1;
		FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM |
			      FORMAT_MESSAGE_IGNORE_INSERTS,
			      NULL,
			      GetLastError(),
			      MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
			      // Default language
			      buffer, len, NULL);
	} else {
		sprintf(buffer, "%s:\n%s", msg, strerror(errno));
	}

	// Display the string.
	MessageBox(NULL, msg, "Console Error", MB_OK | MB_ICONERROR);
}


BOOL
MashConsole::SavedFileHandle::Set(DWORD nStdHandle, int newHandle, FILE *file)
{
	int fileno = _fileno(stdout);

	FILE *fp = _fdopen(newHandle, "w");
	if (fp==NULL || setvbuf(fp, NULL, _IONBF, 0)!=0) {
		ShowError("Could not create FILE object for output "
			  "handle", FALSE);
		if (fp) fclose(fp);
		else close(newHandle);
		return FALSE;
	}

	if (SetStdHandle(nStdHandle,
			 (HANDLE)_get_osfhandle(newHandle))==FALSE) {
		fclose(fp);
		ShowError("Could not set the system standard handle", TRUE);
		return FALSE;
	}

	file_ = *file;
	*file = *fp;
	osf_handle_ = GetStdHandle(nStdHandle);
	return TRUE;
}


void
MashConsole::SavedFileHandle::Restore(DWORD nStdHandle, FILE *file)
{
	if (osf_handle_ != INVALID_HANDLE_VALUE) {

		// for some reason, the other end of the pipe does not
		// seem to detect EOF unless I invoke both fclose and close!
		// may have something to do with the way _fd_open is
		// implemented on Windows.

		int fileno = _fileno(file);
		fclose(file);
		close(fileno);
		SetStdHandle(nStdHandle, osf_handle_);
		*file = file_;

		osf_handle_ = INVALID_HANDLE_VALUE;
	}
}


BOOL
MashConsole::CreateOutputPipe(int &outputReadHandle)
{
	enum { READ_HANDLE, WRITE_HANDLE };
	int hOutputPipe[2], fileno;

	// Create the pipe
	if(_pipe(hOutputPipe, 512, O_BINARY) == -1) {
		ShowError("Could not create output pipe", FALSE);
		return FALSE;
		// don't goto error here, coz there is nothing to cleanup!
	}

	fileno = _dup(hOutputPipe[WRITE_HANDLE]);
	if (fileno < 0) {
		ShowError("Could not duplicate output file handle for stdout",
			  FALSE);
		goto error;
	}
	if (savedStdOut_.Set(STD_OUTPUT_HANDLE, fileno, stdout)==FALSE)
		goto error;

	fileno = _dup(hOutputPipe[WRITE_HANDLE]);
	if (fileno < 0) {
		ShowError("Could not duplicate output file handle for stderr",
			  FALSE);
		goto error;
	}
	if (savedStdErr_.Set(STD_ERROR_HANDLE, fileno, stderr)==FALSE)
		goto error;

	// Close original write end of pipe
	close(hOutputPipe[WRITE_HANDLE]);

	outputReadHandle = hOutputPipe[READ_HANDLE];
	return TRUE;

error:
	savedStdOut_.Restore(STD_OUTPUT_HANDLE, stdout);
	savedStdErr_.Restore(STD_ERROR_HANDLE,  stderr);
	close(hOutputPipe[WRITE_HANDLE]);
	close(hOutputPipe[READ_HANDLE]);
	return FALSE;
}


void
MashConsole::CloseOutputPipe()
{
	savedStdOut_.Restore(STD_OUTPUT_HANDLE, stdout);
	savedStdErr_.Restore(STD_ERROR_HANDLE,  stderr);
}



#if 0
{
	fileno = _fileno(stdout);
	if (fileno < 0) {
		// we don't currently have a stdout
		fp = _fdopen(hOutputPipe[WRITE_HANDLE], "w");
		if (fp==NULL || setvbuf(fp, NULL, _IONBF, 0)!=0) {
			ShowError("Could not create FILE object for output "
				  "handle");
			if (fp) fclose(fp);
			return FALSE;
		}

		savedStdOut_.Save(stdout);
		*stdout = *fp;
		SetStdHandle(_get_osfhandle(hOutputPipe[WRITE_HANDLE]));
	}
	else {
		// we do have a stdout already; do a complicated
		// dup procedure to map it to a new file descriptor

		savedStdOut_.Save(fileno);

		// duplicate the write end of the pipe to the stdout handle

	}

	outputReadHandle = hOutputPipe[READ_HANDLE];
	return TRUE;

#if 0
	// Duplicate stdout handle (next line will close original)
	savedStdOut_ = _dup(_fileno(stdout));

	MN("stdout is ", _fileno(stdout));
	// Duplicate write end of pipe to stdout handle
	if(_dup2(hOutputPipe[WRITE_HANDLE], _fileno(stdout)) != 0) {
		M("stdout dup2 failed");
		return FALSE;
	}

	// Duplicate stderr handle (next line will close original)
	savedStdErr_ = _dup(_fileno(stderr));

	// Duplicate write end of pipe to stderr handle
	if(_dup2(hOutputPipe[WRITE_HANDLE], _fileno(stderr)) != 0)
		return FALSE;

	// Close original write end of pipe
	close(hOutputPipe[WRITE_HANDLE]);

	outputReadHandle = hOutputPipe[READ_HANDLE];
	return TRUE;
#endif
}
#endif



#if 0
BOOL
MashConsole::CloseOutputPipe()
{
#if 0
	// Duplicate copy of original stdout back into stdout
	if (_dup2(savedStdOut_, _fileno(stdout)) != 0) return FALSE;
	// Close duplicate copy of original stdout
	close(savedStdOut_);

	// Duplicate copy of original stderr back into stderr
	if (_dup2(savedStdErr_, _fileno(stderr)) != 0) return FALSE;
	// Close duplicate copy of original stderr
	close(savedStdErr_);

	savedStdOut_ = savedStdErr_ = -1;
#endif
	return TRUE;
}
#endif


void
MashConsole::Destroy()
{
	CloseOutputPipe();
	DestroyWindow(toplevel_);
	toplevel_ = statusBar_ = edit_ = NULL;
}


LRESULT CALLBACK MashConsole::MainWindowProc_Static(HWND hwnd, UINT msg,
						    WPARAM wParam,
						    LPARAM lParam)
{
	MashConsole *This=NULL;
	CREATESTRUCT *info;

	switch (msg) {
	case WM_CREATE:
		info = (CREATESTRUCT *) lParam;
		This = (MashConsole*) info->lpCreateParams;
		SetWindowLong(hwnd, GWL_USERDATA, (LONG) This);
		//MessageBox(NULL, "Building window", NULL, MB_OK);
		This->BuildWindow(hwnd);
		return 0;

	default:
		This = (MashConsole*)GetWindowLong(hwnd, GWL_USERDATA);
		if (This)
			return This->MainWindowProc(hwnd, msg, wParam, lParam);
		else return DefWindowProc(hwnd, msg, wParam, lParam);
	}
}


LRESULT MashConsole::MainWindowProc(HWND hwnd, UINT msg,
				    WPARAM wParam, LPARAM lParam)
{
	BOOL  isSysMenu;
	HMENU hmenu;
	UINT  uPos;

	switch(msg)
	{
	case WM_SIZE:
		if (edit_ && statusBar_) {
			int width, height, statusHeight;
			RECT rect;
			width = LOWORD(lParam);
			height= HIWORD(lParam);
			GetWindowRect(statusBar_, &rect);
			statusHeight = rect.bottom - rect.top + 1;
			MoveWindow(edit_, 0, 0, width, height-statusHeight+1,
				   TRUE);
			MoveWindow(statusBar_, 0, height-statusHeight,
				   width, statusHeight, TRUE);
			return 0;
		}
		break;

	case MC_OUTPUT:
		// got more data to write
	{
		char *buffer;
		int len;
		buffer = (char*)lParam;
		len    = (int)wParam;
		Output(buffer, len);
		delete [] buffer;
		break;
	}

	case WM_INITMENUPOPUP:
		isSysMenu = (BOOL) HIWORD(lParam); // window menu flag
		hmenu = (HMENU) wParam;            // handle of submenu
		uPos  = (UINT) LOWORD(lParam);     // submenu item position
		if (!isSysMenu && uPos==1) {
			int select, enable;

			select = SendMessage(edit_, EM_GETSEL, 0, 0);
			if (HIWORD(select)==LOWORD(select))
				enable = MF_GRAYED;
			else
				enable = MF_ENABLED;
			EnableMenuItem(hmenu, IDM_COPY, enable);
		}
		break;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDM_HIDE:
			Show(FALSE);
			return 0;

		case IDM_EXIT:
			exit(0);
			return 0;

		case IDM_COPY:
			SendMessage(edit_, WM_COPY, 0, 0);
			return 0;

		case IDM_SELECTALL:
			SendMessage(edit_, EM_SETSEL, 0, -1);
			return 0;

		case IDM_ABOUT:
			DialogBox(hInstance_, "MashConsoleAboutDialog",
				  toplevel_, (DLGPROC) AboutDialogProc);
			break;
		}

		break;

	case WM_CLOSE:
		// just hide the window; don't actually close it!
		Show(FALSE);
		return 0;
	}

	return DefWindowProc(hwnd, msg, wParam, lParam);
}


BOOL CALLBACK
MashConsole::AboutDialogProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	case WM_COMMAND:
		if (LOWORD(wParam)==IDOK) {
			EndDialog(hdlg, 0);
			return TRUE;
		}
		break;
	}

	return FALSE;
}


void
MashConsole::BuildWindow(HWND toplevel)
{
	toplevel_ = toplevel;
	edit_ = CreateWindow("edit", NULL, WS_CHILD | WS_VISIBLE | WS_BORDER |
			     WS_EX_CLIENTEDGE | WS_HSCROLL | WS_VSCROLL |
			     ES_LEFT | ES_MULTILINE | ES_AUTOVSCROLL |
			     ES_AUTOHSCROLL | ES_READONLY, 0, 0, 0, 0,toplevel,
			     (HMENU) ID_EDIT, hInstance_, NULL);
	statusBar_ = CreateStatusWindow(WS_CHILD | WS_VISIBLE |
					WS_CLIPSIBLINGS | CCS_BOTTOM,
					"Application running...",
					toplevel, ID_STATUS);
	SendMessage(edit_, WM_SETFONT,
		    (WPARAM)GetStockObject(DEFAULT_GUI_FONT),
		    MAKELPARAM(TRUE, 0));

	HDC hdc = GetDC(edit_);
	SIZE size;
	GetTextExtentPoint32(hdc, "a", 1, &size);
	ReleaseDC(edit_, hdc);
	size.cx *= 60;
	size.cy *= 25;

	// add something for the statusbar, menu, title, borders
	RECT rect;
	GetClientRect(toplevel_, &rect);
	size.cx += rect.left * 2;
	size.cy += rect.top * 2;
	GetWindowRect(toplevel_, &rect);
	MoveWindow(toplevel_, rect.left, rect.top, size.cx, size.cy, TRUE);

	int outputReadHandle;
	if (CreateOutputPipe(outputReadHandle)==FALSE) {
		MessageBox(NULL, "error in creating output pipe", NULL, MB_OK);
		return;
	}

	ConsoleThread *thread = new ConsoleThread(outputReadHandle, toplevel_);
	_beginthread(ThreadMain, 0, (void*)thread);
}


void
MashConsole::Output(const char *bytes, int len)
{
	// first count any "\n" characters that are not preceded by "\r"
	const char *ptr;
	char *dst;
	int extra=0;
	for (ptr=bytes; ptr < (bytes+len); ptr++) {
		if (*ptr=='\n') {
			if ( ! (ptr > bytes && *(ptr-1)=='\r') ) extra++;
		}
	}

	// make sure we have enough space in the buffer
	if ((len+extra) > maxLen_) {
		int needToCut = len - maxLen_;
		bytes += needToCut;
		len   -= needToCut;
	}

	if (len_ + len + extra > maxLen_) {
		CreateSpace(len_ + len + extra);
	}

	// now copy the bytes over
	for (ptr=bytes, dst=buffer_+len_; ptr < (bytes+len); ptr++) {
		if (*ptr=='\n') {
			if ( ! (ptr > bytes && *(ptr-1)=='\r') ) *dst++ = '\r';
		}
		*dst++ = *ptr;
	}

	*dst = '\0';
	len_ = (dst-buffer_);

	if (edit_) {
		SetWindowText(edit_, buffer_);
		if (IsVisible()) {
			/* auto scroll to the end */
			SendMessage(edit_, EM_SETSEL, len_, len_);
			SendMessage(edit_, EM_SCROLLCARET, 0, 0);
		}
	}
}


void
MashConsole::CreateSpace(int need)
{
	int size = maxLen_ - (need - maxLen_);

	if (size < len_) {
		// we have too much data; we should throw away some...
		// shrink the buffer to size
		CopyTo(buffer_, size);
		len_ = size;
	}
}


void
MashConsole::CopyTo(char *newBuffer, int &size)
{
	int curSize = len_;
	char *curBuf= buffer_, *newline;

	while (curSize > size) {
		newline = strchr(curBuf, '\n');
		if (newline==NULL) {
			curSize = 0;
			curBuf  = buffer_ + maxLen_;
			break;
		} else {
			curSize -= (newline+1 - curBuf);
			curBuf   = newline+1;
		}
	}

	if (curSize <= 0) {
		size = 0;
		*newBuffer = '\0';
	} else {
		// remember to copy the trailing '\0'
		memmove(newBuffer, curBuf, curSize+1);
		size = curSize;
	}
}


void
MashConsole::ThreadMain(void *args)
{
	ConsoleThread *thread = (ConsoleThread*)args;
	thread->Main();
	delete thread;

	// implicit _endThread
}


void
ConsoleThread::Main()
{
	char *buffer;
	const int  bytesAtATime=80;
	int bytesRead;

	while (1) {
		buffer = new char[bytesAtATime+1];
		bytesRead = _read(outputReadHandle_, buffer, bytesAtATime);
		if (bytesRead <= 0) {
			if (bytesRead < 0) {
				MessageBox(NULL, strerror(errno),
					   "Console error",
					   MB_ICONERROR | MB_OK);
			}
			close(outputReadHandle_);
			break;
		}

		PostMessage(consoleHwnd_, MC_OUTPUT, (WPARAM)bytesRead,
			    (LPARAM)buffer);
	}
}

