/*
 * video-dali.cc --
 *
 *      Creates a Dali pseudo-device for transmitting video.
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/video-dali.cc,v 1.1 2002/02/07 04:17:01 chema Exp $ */


#include <stdio.h>
#include <stdlib.h>
#include <fstream.h>
#include <sys/stat.h>

#include "Tcl.h"
#include "module.h"
#include "video/video-device.h"
#include "rgb-converter.h"
#include "fx/vidreps.h"



//
//	DaliVideoCapture
//
// this is the C++ object that implements the virtual, source module device 
//	that takes Dali images and acts as a VideoCapture device plugging them 
//	into an encoder
//
//	Question: Why a SourceModule instead of the more-common VideoCapture?
//
//	Answer: We don't use almost anything from VideoCapture, and we don't
//	want the tcl class to inherit from VideoCapture because that will
//	mean it will appear in vic as one of the "Device..."s. Actually we 
//	should even be careful with the Timer-related code in VideoCapture
//	because in our case frame-capturing is triggered by the arriving of
//	an FX-processed frame, while in the VideoCapture case is triggered
//	by a Timer. FIXME: More on this decision coming...
//
//
//
class DaliVideoCapture : public VideoCapture {
public:
	DaliVideoCapture(int csss);
	virtual ~DaliVideoCapture();

	virtual int command(int argc, const char*const* argv);
	inline int sameframe(const Uncompressed* uf) {
		return ((uf->w_ == inw_) && (uf->h_ == inh_) && (uf->bp_ == frame_));
	}
	void init_frame(Uncompressed* input);

	void recv(Uncompressed* input);
	void recv(Buffer* bp) {
		recv((Uncompressed *)bp);
	};

	virtual void start();
	virtual void stop();
	virtual void grab();
	inline Module* target() { return (target_); }

protected:
	int capture();
	void format();
	void setsize();
//	virtual void set_frame_buffer(VidRep *fb) {
//		if (input_ != 0) {
//			input_->dealloc();
//		};
//		input_ = (Uncompressed *) fb;
//	};

	Uncompressed *input_;
	Module* target_;
	EncoderModule* encoder_;

	int ts_;

	int fd_;
	int csss_;
  
//	int decimate_;

	// true if capture object okay
	int status_;
};



//
//	DaliVideoCapture::DaliVideoCapture
//
//	Inputs:
//		int csss: color subsampling scheme. It may be 422, 420, or 411
//
DaliVideoCapture::DaliVideoCapture (int csss)
	: VideoCapture(), input_(0), target_(0), encoder_(0), ts_(0), status_(0)
{
//	// create the new Uncompressed image
//	output_ = new Uncompressed; 
//	// initialize it as a 0-pixel x 0-pixel, 4:4:4 color, no-buffers image
//	output_->init(width_, height_, 1, 1, width_, height_, 0, 0, 0, 0, 0);


	if ((csss != 444) && (csss != 422) && (csss != 420) && (csss != 411)) {
		fprintf (stderr, "DaliVideoCapture: csss %d unsupported\n", csss);
	}
	csss_ = csss;


//	decimate_  = 2;

	// create the VideoCapture registering methods (the tcl class won't inherit 
	//	from VideoCapture!)
	Tcl& tcl = Tcl::instance();
	char *classname_ = "DaliVideoCapture/Uncompressed";
	char *nickname_ = "dalisource";
	char *attributes_ = "format { 411 422 420 } size { small large cif }";
	tcl.evalf("%s set nickname_ {%s}", classname_, nickname_);
	tcl.evalf("%s set attributes_ {%s}", classname_, attributes_);

	// register as a Device so VideoTap is happy
	tcl.evalf("import Device");
	tcl.evalf("Device register_class %s", classname_);

	format();
}


//
//	DaliVideoCapture::~DaliVideoCapture
//
DaliVideoCapture::~DaliVideoCapture()
{
}


//
//	DaliVideoCapture::command
//
// This method is called from tcl to change the video-capture device 
//	settings
//
int DaliVideoCapture::command(int argc, const char*const* argv)
{

	if (argc == 3) {
//		if (strcmp(argv[1], "decimate") == 0) {
//			decimate_ = atoi(argv[2]);
//			// FIXME
//			format();
//			return (TCL_OK);
//		}

		if (strcmp(argv[1], "encoder") == 0) {
			encoder_ = (EncoderModule*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}

		if (strcmp(argv[1], "target") == 0) {
			target_ = (Module*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}

		if (strcmp(argv[1], "set-frame-buffer") == 0) {
			VidRep *fb;
			fb = (VidRep *)TclObject::lookup(argv[2]);
//			set_frame_buffer(fb);
			return TCL_OK;
		}

		if (strcmp(argv[1], "port") == 0) {
			return (TCL_OK);
		}

		if (strcmp(argv[1], "fps") == 0) {
			// what goes here?
			return (TCL_OK);
		}

		if (strcmp(argv[1], "bps") == 0) {
			// what goes here?
			return (TCL_OK);
		}

		if (strcmp(argv[1], "send") == 0) {
			// what goes here?
			return TCL_OK;
		}

	} else if (argc == 2) {
		Tcl& tcl = Tcl::instance();

		if (strcmp(argv[1], "status") == 0) {
			sprintf(tcl.buffer(), "%d", status_);
			tcl.result(tcl.buffer());
			return TCL_OK;
		}

		if (strcmp(argv[1], "need-capwin") == 0) {
			tcl.result("0");
			return (TCL_OK);
		}

		if (strcmp(argv[1], "recv") == 0) {
			grab();
			return TCL_OK;
		}
	}
  
	return (VideoCapture::command(argc, argv));
}

//
//	DaliVideoCapture::start
//
void DaliVideoCapture::start()
{
	format();
	//VideoCapture::start();
}

//
//	DaliVideoCapture::stop
//
void DaliVideoCapture::stop()
{   
	//VideoCapture::stop();
}


//
//	DaliVideoCapture::capture
//
// This method is called every time the effect implementation calls 
//	recv from tcl. 
//
int DaliVideoCapture::capture()
{
	if (input_ == 0) {
		// input frame still unmapped
		return 0;
	}

	frame_ = (u_char *)(input_->bp_);
	return (1);
}



void DaliVideoCapture::init_frame(Uncompressed* input)
{
	switch (csss_) {
	case 422:
		set_size_422(input->w_, input->h_);
		break;

	case 420:
		// set_size_420 is really set_size_420_and_check_cif_dimensions
		//	As the only thing this does is to set the chroma sizes, 411 
		//	produces the same results than 420
		//set_size_420(input->w_, input->h_);
		set_size_411(input->w_, input->h_);
		break;

	case 411:
		set_size_411(input->w_, input->h_);
		break;

	default:
		fprintf(stderr, "DaliVideoCapture::init_frame -> bad geometry %dx%d\n",
				input->w_, input->h_);
    abort();
	}

	// set_size_xxx creates a new frame, but we prefer to use the 
	//	one obtained from the EffectModule
	delete[] framebase_;
	frame_ = framebase_ = (u_char *)(input->bp_);

	// allocate the ref_ frame
	allocref();
	return;
}

void DaliVideoCapture::recv(Uncompressed* input)
{
	if (!sameframe(input)) {
		init_frame(input);
	}

	suppress(frame_);
	saveblks(frame_);

	//YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	YuvFrame f(input->ts_, frame_, crvec_, outw_, outh_);
	target_->recv(&f);
}

void DaliVideoCapture::grab()
{
	if (capture() == 0)
			return;
	suppress(frame_);
	saveblks(frame_);

	for (int i=0; i<1584; i++) {
		crvec_[i] = 0xff;
	};
	//YuvFrame f(media_ts(), frame_, crvec_, width_, height_);
	ts_++;
	YuvFrame f(ts_, frame_, crvec_, outw_, outh_);
	target_->recv(&f);
}

void DaliVideoCapture::format()
{
	return;
}


static class DaliVideoCaptureClass : public TclClass {
	public:
		DaliVideoCaptureClass() : TclClass("DaliVideoCapture/Uncompressed") {}
		TclObject* create(int argc, const char*const* argv) {
			if (argc != 5) {
				printf("Invalid arguments passed to DaliVideoCaptureClass\n");
				return (0);
			}
			if (strcmp(argv[4], "422") == 0) {
				return (new DaliVideoCapture(422));
			} else if (strcmp(argv[4], "411") == 0) {
				return (new DaliVideoCapture(411));
			} else if (strcmp(argv[4], "420") == 0) {
				return (new DaliVideoCapture(420));
			} else if (strcmp(argv[4], "cif") == 0) {
				return (new DaliVideoCapture(420));
			} else {
				printf("Invalid type passed to DaliVideoCaptureClass (%s)\n", argv[4]);
			}
			return (0);
		}
} dalisourcemodule_class;

