/*
 * proc.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <osfcn.h>
#include <sys/param.h>
#include <unistd.h>
#include <stdio.h>
#if defined(ultrix) || defined(AIX)
#include <fcntl.h>
#else
#include <sys/fcntl.h>
#endif
#include <string.h>

#include <sys/wait.h>
#include <sys/resource.h>
#if defined(hpux) || defined(AIX) || defined(linux)
#include <signal.h>
#endif

#include "proc.h"

extern "C" char** environ;

Process::Process()
{
	nenv = 0;
	const char** p = (const char**)::environ;
	while (*p++ != 0)
		++nenv;

	maxenv = 2 * nenv;
	environ = new char*[maxenv];
	envbase = nenv;

	for (int i = 0; i < nenv; ++i)
		environ[i] = ::environ[i];

	pid = -1;
	/* Make sure nbio() does not work until after fork. */
	fd_out = -1;
}

Process::~Process()
{
	if (pid > 0)
		kill();

	for (int i = envbase; i < nenv; ++i)
		delete environ[i];
	delete environ;
}

int Process::envslot(const char* var)
{
	int n = strlen(var);
	for (int i = 0; i < nenv; ++i) {
		if (strncmp(var, environ[i], n) == 0 && environ[i][n] == '=')
			return (i);
	}
	return (-1);
}

const char* Process::getenv(const char* var)
{
	int s = envslot(var);
	if (s < 0)
		return (0);
	const char* cp = environ[s];
	while (*cp != '=')
		if (*cp++ == 0)
			return (0);
	return (cp + 1);
}

void Process::setenv(const char* var, const char* value)
{
	if (nenv + 1 >= maxenv)
		/* FIXME too many setenv calls */
		abort();

	int s = envslot(var);
	if (s < 0)
		s = nenv++;
	else if (s >= envbase)
		delete environ[s];

	int n = strlen(var) + strlen(value) + 2;
	char* cp = new char[n];
	sprintf(cp, "%s=%s", var, value);
	environ[s] = cp;
}

void Process::nonblock(int fd)
{
	int flags = fcntl(fd, F_GETFL, 0);
	flags |= O_NONBLOCK;
	if (fcntl(fd, F_SETFL, flags) == -1) {
		perror("fcntl: F_SETFL");
		exit(1);
	}
}

void Process::fork(const char* exec, int, char** argv)
{
	/*
	 * Pipes for communicating between the parent and child.
	 * The naming convention is from the perspective of the child.
	 */
	int pin[2], pout[2], perr[2];
	register char** envsave = ::environ;

#ifdef notyet
	if (pipe(pin) < 0 || pipe(pout) < 0 || pipe(perr) < 0) {
		perror("pipe");
		exit(1);
	}
#else
	if (pipe(pin) < 0) {
		perror("pipe");
		exit(1);
	}
#endif
#ifndef sgi
	pid = ::vfork();
#else
	pid = ::fork();
#endif
	if (pid < 0) {
		perror("vfork");
		exit(1);
	}
	if (pid == 0) {
		/*
		 * The child -- exec the new process.
		 * Bind pipe descriptors to stdio.
		 */
		::environ = environ;
		environ[nenv] = 0;

		close(pin[1]);
		dup2(pin[0], 0);
#ifdef notyet
		close(pout[0]);
		dup2(pout[1], 1);
		close(perr[0]);
		dup2(perr[1], 2);
#endif
		for (int i = 3; i < 32; ++i)
			close(i);

		if (argv == 0) {
			char* av[2];
			av[0] = (char*)exec;
			av[1] = 0;
#ifdef __osf__
			execvp(exec, (char *const*)av);
#else
			execvp(exec, av);
#endif

		} else
			execvp(exec, argv);
		perror(exec);
		_exit(1);
	}
	/*
	 * The parent -- set up non-blocking I/O and
	 * remember our sides of the pipes.
	 */
	::environ = envsave;
	close(pin[0]);
	fd_out = pin[1];
#ifdef notyet
	close(pout[1]);
	fd_in = pout[0];
	close(perr[1]);
	fd_err = perr[0];
#endif
}

void Process::kill()
{
	if (pid <= 0)
		return;
	/*
	 * If the process has not exited, kill it.  We really do not want
	 * to block here, but once we kill it, waitpid() should return soon.
	 */
	close(fd_out);
	fd_out = -1;
#ifdef notyet
	close(fd_in);
	fd_in = -1;
	close(fd_err);
	fd_err = -1;
#endif
	int s;
	::kill(pid, SIGKILL);
	waitpid(pid, &s, 0);
	pid = -1;
}
