/*
 * video-lmljpg.cc --
 *
 *      Support for Linux Media Labs LML33 video capture/playback MJPEG card.
 */

/* =========================================================================

     Copyright (c) 1997 Regents of Koji OKAMURA, oka@kobe-u.ac.jp
     All rights reserved.

     largely rewritten for new bttv/video4linux interface
     by Gerd Knorr <kraxel@cs.tu-berlin.de>

     Adapted for the Mash environment by Hank Magnuski <hankm@netvideo.com>

     Modified for Linux Media Labs LML33 board with jpeg capture
     (i.e., /dev/mvideo/stream*).

   ========================================================================= */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/mman.h>

#include <linux/types.h>

extern "C" {
#include <linux/videodev.h>
}

#include "video-device.h"
#include "Tcl.h"
#include "module.h"
#include "lml33.h"

/* here you can tune the device names */
static const char *devlist[] = {
	"/dev/mvideo/stream0",
	"/dev/mvideo/stream1",
	"/dev/mvideo/stream2",
	"/dev/mvideo/stream3",
	NULL };

static const char *nickName = "LML33 JPEG";

/* display memory */
unsigned char* lml33jpg_memory;
unsigned int lml33jpg_offset;
unsigned int lml33jpg_width;
unsigned int lml33jpg_height;

#define DEBUG(x)
//#define DEBUG(x) (x)
int dropframe;

#define NTSC	1
#define PAL	2
#define	SECAM	3

#define NTSC_WIDTH  640
#define NTSC_HEIGHT 480
#define PAL_WIDTH   768
#define PAL_HEIGHT  576
#define CIF_WIDTH   352
#define CIF_HEIGHT  288

#define CF_422 0
#define CF_411 1
#define CF_CIF 2

#define LML_D1 0
#define LML_CIF 3
#define LML_QCIF 4

class VideoCaptureLMLJPEG : public VideoCapture {
public:
    VideoCaptureLMLJPEG(const char * cformat, const char *dev);
    virtual ~VideoCaptureLMLJPEG();

    virtual int  command(int argc, const char*const* argv);
    virtual void start();
    virtual void stop();
    virtual void grab();
protected:
    void format();
    void setsize();
    u_int8_t* parse_markers(u_int8_t* start, u_int8_t* end, u_int8_t** argv);

    // video buffer with compressed frames
    CodecFrame cf;
    int grab_count;

    int video_format_;      /* video input format: NTSC, PAL or SECAM */
    int fd_;
    int format_;
    int cformat_;
    int port_;

    unsigned char *tm_;
    int width_;
    int height_;
    int max_width_;
    int max_height_;
    int decimate_;
    int rtp_type_;
    int rtp_type_specific_;
};

/* ----------------------------------------------------------------- */

class LMLJPEGDevice : public VideoDevice {
public:
        LMLJPEGDevice(const char* clsname, const char* nickname,
                  const char* devname, const char* attr, int free);
        TclObject* create(int argc, const char*const* argv) {
                if (argc != 5)
                        abort();/*FIXME*/
		if ((argv[4][0]=='4')&&(argv[4][1]=='2')&&(argv[4][2]=='2'))
                  return (new VideoCaptureLMLJPEG(argv[4],name_));
                else if (strcmp(argv[4], "cif") == 0)
                  return (new VideoCaptureLMLJPEG("cif",name_));
                return (0);
                }
protected:
        const char* name_;
};

LMLJPEGDevice::LMLJPEGDevice(const char* clsname, const char* nickname,
                     const char *devname, const char* attr, int free)
  : VideoDevice(clsname, nickname), name_(devname)
{
        if (free)
	 attributes_ = attr;
        else
         attributes_ = "disabled";

        DEBUG(fprintf(stderr,"LMLJPEG: ==> %s, %s, %s\n",
                      clsname,nickname,name_));
	DEBUG(fprintf(stderr,"LMLJPEG: ==> %s\n",attributes_));
}

/* ----------------------------------------------------------------- */

class LMLJPEGScanner {
public:
    LMLJPEGScanner(const char **dev);
};

static LMLJPEGScanner find_lmljpeg_devices(devlist);

LMLJPEGScanner::LMLJPEGScanner(const char **dev)
{
    int  i,fd;
    const char *lmlJPEGclassnames[] = {
	"VideoCapture/LMLJPEG0",
	"VideoCapture/LMLJPEG1",
	"VideoCapture/LMLJPEG2",
	"VideoCapture/LMLJPEG3"
    };

    char *attr;

    for (i = 0; dev[i] != NULL; i++) {
	DEBUG(fprintf(stderr,"LMLJPEG: trying %s... ",dev[i]));
	if (-1 == (fd = open(dev[i],O_RDONLY))) {
	    DEBUG(perror("open"));
	    continue;
	}

	DEBUG(fprintf(stderr,"ok, %s\n", nickName));

	attr = new char[512];
	strcpy(attr,"format { jpeg } ");

	strcat(attr,"size { large } ");

	strcat(attr,"port { Composite } ");

	new LMLJPEGDevice(lmlJPEGclassnames[i],nickName,dev[i],attr,1);

	close(fd);
    }
}

/* ----------------------------------------------------------------- */

VideoCaptureLMLJPEG::VideoCaptureLMLJPEG(const char *cformat, const char *dev)
{

    DEBUG(fprintf(stderr,"LMLJPEG: constructor (fmt=%s, dev=%s)\n",
                  cformat, dev));

    fd_ = open(dev, O_RDWR);
    if (fd_ < 0) {
	perror("open");
	exit(1);
    }
    video_format_ = NTSC; /* FIXME hack for now - should set in constructor? */
	/* default: 422 progressive - no line doubling */
    cformat_ =CF_422;
    rtp_type_ = 0;
    rtp_type_specific_ = 0;

    /* fill in defaults */
    if(!strcmp(cformat, "422p")) { // 4:2:2 progressive
	rtp_type_specific_ = 3; // first field only - double it.
    }
    if(!strcmp(cformat, "422i")) { // 4:2:2 interlaced
	rtp_type_specific_ = 1; // first field is odd frame
    }
    if(!strcmp(cformat, "411")) {
	cformat_ = CF_411;
	rtp_type_ = 1;
	rtp_type_specific_ = 3; // first field - double it.
    }
    if(!strcmp(cformat, "cif")) {
	perror("CIF format not supported by this device");
	close(fd_);
	return;
    }

    port_      = 0;
    decimate_  = 1;
}

VideoCaptureLMLJPEG::~VideoCaptureLMLJPEG()
{
    DEBUG(fprintf(stderr,"LMLJPEG: destructor\n"));

    close(fd_);
}

int VideoCaptureLMLJPEG::command(int argc, const char*const* argv)
{
    Tcl& tcl = Tcl::instance();

    if (argc == 2) {
        if (strcmp(argv[1], "is-hardware-encode") == 0) {
            tcl.result("1");
            return (TCL_OK);
        }

        if (strcmp(argv[1], "rtp_type") == 0) {
		sprintf(tcl.buffer(), "%d", rtp_type_);
		tcl.result(tcl.buffer());
		return (TCL_OK);
	}
    }
    if (argc == 3) {
	if (strcmp(argv[1], "decimate") == 0) {
	    decimate_ = atoi(argv[2]);
	    if (running_)
		format();
	}

	if (strcmp(argv[1], "port") == 0) {
    	    return (TCL_OK);
	}

        if (strcmp(argv[1], "rtp_type") == 0) {
		rtp_type_ = atoi(argv[2]);
		return (TCL_OK);
	}
    }

    return (VideoCapture::command(argc, argv));
}

void VideoCaptureLMLJPEG::start()
{
    DEBUG(fprintf(stderr,"LMLJPEG: start\n"));

    format();
    grab_count = 0;
    VideoCapture::start();
}

void VideoCaptureLMLJPEG::stop()
{
    DEBUG(fprintf(stderr,"LMLJPEG: stop (grab_count=%d)\n", grab_count));

    VideoCapture::stop();
}

void VideoCaptureLMLJPEG::grab()
{
    int retryCount=0;
    int i, n, len, field;
    u_int8_t *p;
    u_int8_t *bp;
    u_int8_t *argv[100];

    // Read frame header, block if no frame ready
    while ( read(fd_, (char*)&cf, sizeof(FrameHeader)) == -1 ) {
    	sleep(1);
        if ( ++retryCount > 100 ) {
        	perror("CodecFrame read retry count exceeded");
               exit(1);
        }
    }

    // Read the rest of the frame
    n = read(fd_, cf.data,
             cf.header.frameSize - sizeof(FrameHeader));
    if ( n == -1 ) {
    	perror("CodecFrame.data read error");
        exit(1);
    }

    grab_count++;

	/* grab the odd field */
    p = parse_markers((u_int8_t *) &cf, (u_int8_t *) &cf+cf.header.frameSize,
                      (u_int8_t **) argv);
    //printf("returned from parse_markers odd field\n");

    // get start and end of image data
    u_int8_t *sosp = NULL, *eoip = NULL;
    for (i=0; argv[i] != 0; i++) {
	if (((char) argv[i][1]) == ((char) 0xda)) { // SOS marker
		sosp = argv[i];
		//printf("...found SOS marker\n");
	}
	if (((char) argv[i][1]) == ((char) 0xd9)) { // EOI marker
		eoip = argv[i];
		//printf("...found EOI marker\n");
	}
    }
    //printf("finished scanning argv\n");
    if (sosp == NULL) {
	printf("Missing SOS marker frame# %d odd field\n", grab_count);
	return;
    }
    if (eoip == NULL) {
	printf("Missing EOI marker frame# %d odd field\n", grab_count);
	return;
    }

        /* create jpeg frame and send it */
    sosp += 2;			// get length of scan header
    bp = (u_int8_t *)(((u_int) sosp) + ((u_int) *sosp)); // skip scan header
    len = (int) (eoip - bp);		// compute length of scan data
	/* create jpeg frame and pass to framer */
    JpegFrame f(media_ts(), bp, len, 50, rtp_type_, rtp_type_specific_,
                720, 240);
    target_->recv(&f);
printf("send odd field frame %d\n", grab_count);

    switch (rtp_type_specific_)  {
      case 1:
	break; // send second field
      case 0:
      case 3:
	return; // send only first field
      default:
	printf("Illegal rtp_type_specific_ field setting\n");
	return;
    }
	/* grab the even field */
    if ((eoip != NULL) && (cformat_ == CF_422)) {
        p = parse_markers(p, (u_int8_t *) &cf+cf.header.frameSize, argv);
    	//printf("returned from parse_markers even field\n");

    	// get start and end of image data
    	sosp = NULL; eoip = NULL;
    	for (i=0; argv[i] != 0; i++) {
        	if (((char) argv[i][1]) == ((char) 0xda)) { // SOS marker
                	sosp = argv[i];
                	//printf("...found SOS marker\n");
        	}
        	if (((char) argv[i][1]) == ((char) 0xd9)) { // EOI marker
                	eoip = argv[i];
                	//printf("...found EOI marker\n");
        	}
    	}
    	//printf("finished scanning argv\n");
        if (sosp == NULL) {
            printf("Missing SOS marker frame# %d even field\n", grab_count);
            return;
        }
        if (eoip == NULL) {
            printf("Missing EOI marker frame# %d even field\n", grab_count);
            return;
        }
            /* create jpeg frame and send it */
	sosp += 2;                  // get length of scan header
	bp = (u_int8_t *)(((u_int) sosp) + ((u_int) *sosp)); // skip scan header
	len = (int) (eoip - bp);            // compute length of scan data
            /* create jpeg frame and pass to framer, must be even field */
	JpegFrame f(media_ts(), bp, len, 50, rtp_type_, 2, 720, 240);
	target_->recv(&f);
printf("send even field frame %d\n", grab_count);
    } else {
	printf("Skipped even field...eoip=%u cformat_=%d\n", eoip, cformat_);
    }

}

/*
* parse_markers - return index array to all jpeg markers in the data
*
* returns pointer to byte beyond EOI - so that two images can be parsed
*/
u_int8_t*
VideoCaptureLMLJPEG::parse_markers(
	u_int8_t* start, u_int8_t* end, u_int8_t** argv) {
  int endImage = 0;
  int i = 0;
  u_int8_t *p;

  //printf("parse_markers: start=0x%u, end=0x%u\n",
  //       (unsigned) start, (unsigned) end);
  for ( p=start; ((!endImage) && (p < end)); p++) {
	if (((char) *p) == ((char)0xff)) {
		if ((*(p+1) == 0) || (((char) *(p+1)) == ((char) 0xff))) {
			p++;
			continue;
		}
		//printf("found marker %4.4x at 0x%u\n",
		//       ((((unsigned int) *p) & 0xff) << 8) |
                //          (((unsigned int) *(p+1)) & 0xff),
		//       (unsigned int) (p - start));
		argv[i++] = p;
		endImage = (((char) *(p+1)) == ((char) 0xd9));
		p++;
		if (i >99) {
			perror("parse_markers: overflow argv array");
			return (p);
		}
	}
  }
  argv[i] = (u_int8_t *) 0;
  //printf("...found %d markers\n", i);
  return (p);
}

void VideoCaptureLMLJPEG::format()
{
    //DEBUG(fprintf(stderr,"LMLJPEG: format"));

    switch (decimate_) {
    case 1: // full-sized (LMLJPEG can only grab this size)
      width_ = 720;
      height_ = 240;
      break;

    case 2: // CIF-sized
    case 4: // QCIF - may not work
      width_ = 352;
      height_ = 240;
      break;

    }

    DEBUG(fprintf(stderr,"LMLJPEG: format width_=%d, height_=%d, decimate=%d\n",
	width_,height_,decimate_));

    switch (cformat_) {
    case CF_CIF:
	set_size_411(width_, height_);
	DEBUG(fprintf(stderr,"...format: set_size_411\n"));
	break;
    case CF_411:
	set_size_411(width_, height_);
	DEBUG(fprintf(stderr,"...format: set_size_411\n"));
	break;
    case CF_422:
	set_size_422(width_, height_);
	DEBUG(fprintf(stderr,"...format: set_size_422\n"));
	break;
    }

}
