/*
 * Copyright (c) 1999,2000,2004 Damien Miller <djm@mindrot.org>
 * Copyright (c) 2004 Darren Tucker <dtucker at zip.com.au>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"
#include "ntpd.h"

/* RCSID("$Id: bsd-arc4random.c,v 1.8 2004/09/04 13:23:26 dtucker Exp $"); */

#ifndef HAVE_ARC4RANDOM

#ifdef HAVE_OPENSSL
# include <openssl/rand.h>
# include <openssl/rc4.h>
# include <openssl/err.h>
#else

/*
 * arcfour routines from "nanocrypt" also by Damien Miller, included with
 * permission under the license above.
 * Converted to OpenSSL API by Darren Tucker.
 */

#define _PATH_DEVURANDOM "/dev/urandom"

static int randfd = -1;

typedef struct {
	unsigned int s[256];
	int i;
	int j;
} RC4_KEY;

static void
RC4_set_key(RC4_KEY *r, int len, unsigned char *key)
{
	int t;
	
	for(r->i = 0; r->i < 256; r->i++)
		r->s[r->i] = r->i;

	r->j = 0;
	for(r->i = 0; r->i < 256; r->i++)
	{
		r->j = (r->j + r->s[r->i] + key[r->i % len]) % 256;
		t = r->s[r->i];
		r->s[r->i] = r->s[r->j];
		r->s[r->j] = t;
	}
	r->i = r->j = 0;
}

static void
RC4(RC4_KEY *r, unsigned long len, const unsigned char *plaintext,
    unsigned char *ciphertext )
{
	int t;
	unsigned long c;

	c = 0;
	while(c < len)
	{
		r->i = (r->i + 1) % 256;
		r->j = (r->j + r->s[r->i]) % 256;
		t = r->s[r->i];
		r->s[r->i] = r->s[r->j];
		r->s[r->j] = t;

		t = (r->s[r->i] + r->s[r->j]) % 256;

		ciphertext[c] = plaintext[c] ^ r->s[t];
		c++;
	}
}

static int
RAND_status(void)
{
	if (randfd >= 0)
		return 1;
	randfd = open(_PATH_DEVURANDOM, O_RDONLY);
	return (randfd >= 0 ? 1 : 0);
}

static int
RAND_bytes(unsigned char *buf, size_t len)
{
	if (read(randfd, buf, len) != (ssize_t)len)
		return 0;
	return 1;
}
#endif

/* Size of key to use */
#define SEED_SIZE 20

/* Number of bytes to reseed after */
#define REKEY_BYTES	(1 << 24)

static int rc4_ready = 0;
static RC4_KEY rc4;

void
seed_rng(void)
{
	if (RAND_status() != 1)
		fatal("PRNG is not seeded");
}

unsigned int arc4random(void)
{
	unsigned int r = 0;
	static int first_time = 1;

	if (rc4_ready <= 0) {
		if (first_time)
			seed_rng();
		first_time = 0;
		arc4random_stir();
	}

	RC4(&rc4, sizeof(r), (unsigned char *)&r, (unsigned char *)&r);

	rc4_ready -= sizeof(r);
	
	return(r);
}

void arc4random_stir(void)
{
	unsigned char rand_buf[SEED_SIZE];
	int i;

	memset(&rc4, 0, sizeof(rc4));
	if (RAND_bytes(rand_buf, sizeof(rand_buf)) <= 0)
		fatal("Couldn't obtain random bytes");
	RC4_set_key(&rc4, sizeof(rand_buf), rand_buf);

	/*
	 * Discard early keystream, as per recommendations in:
	 * http://www.wisdom.weizmann.ac.il/~itsik/RC4/Papers/Rc4_ksa.ps
	 */
	for(i = 0; i <= 256; i += sizeof(rand_buf))
		RC4(&rc4, sizeof(rand_buf), rand_buf, rand_buf);

	RC4(&rc4, sizeof(rand_buf), rand_buf, rand_buf);
	memset(rand_buf, 0, sizeof(rand_buf));

	rc4_ready = REKEY_BYTES;
}
#endif /* !HAVE_ARC4RANDOM */
