/* ----------------------------------------------- */
/* company : pentamedia
 * author  : elcomski 
 * file    : inface.c 
 * date    : 2001-06-09
 * modify  : 2001-12-13 by elcomski
 */
/* ----------------------------------------------- */
#if 0 /* if you want to debug , set to 1 */
#define INFACE_DEBUG
#define TUNER_DEBUG
#endif

#include <net/if.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "icedef.h"
#include "../../driver/share/pentadef.h"
#include "../../driver/share/hwpacket.h"
#include "../../driver/share/gatedef.h"
#include "../../share/xptype.h"
#include "../../share/ifdef.h"
#include "pentapi.h"

/* tuner part */
#include "nimdef.h"
#include "nim_drv.h"
#include "nim_drvd.h"



/*--------------------------------*/
/* Funtion prototypes */
/*--------------------------------*/
static aint sm_tuner_start_tunning_action(achar *dev_name, str_trans_data trans_info);
static aint sm_tuner_stop_tunning_action(achar *dev_name);


/*--------------------------------*/
/* Constant definitions */
/*--------------------------------*/
#define MAX_IF_LIST 10
#define HDM8513_ADDR 	0xe0
#define HDM8513A_ADDR 	0xe8


/*--------------------------------*/
/* Structure definitions */
/*--------------------------------*/
struct str_if_list{
	int socket_num;
	char device_name[100];
};


/*--------------------------------*/
/* Variable definitions */
/*--------------------------------*/
struct str_if_list g_dif_list[MAX_IF_LIST] =
{
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"},
    {-1,"pentanet0"}
};

achar gs_dev_name[100];
extern unsigned short dmx_ucode_img[];
extern aint dmx_ucode_size;



/*--------------------------------*/
/* Func definitions */
/*--------------------------------*/

/*
 * Func : sm_device_io_control
 * Context :
 */
int sm_device_io_control(achar *dev_name, str_io_data *sio)
{
	struct ifreq ifr;
	int fd;
	int iret;
	    
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if(fd < 0) return ERR_OPEN_SOCKET;
	strcpy(ifr.ifr_name, dev_name);
	ifr.ifr_data = (void*) sio;

	iret = ioctl(fd, SM_IO_CONTROL, &ifr);
	if(iret < 0){
		close(fd);
		return -ERR_NOT_DEVICE;
	}
	close(fd);
	return 0;
}


/*
 * Func : sm_device_route_control
 * Context :
 */
int sm_device_route_control(achar *dev_name, str_io_data *sio)
{
	struct ifreq ifr;
	int fd;
	int iret;
	    
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if(fd < 0) return ERR_OPEN_SOCKET;
	strcpy(ifr.ifr_name, dev_name);
	ifr.ifr_data = (void*) sio;

	iret = ioctl(fd, SM_ROUTE_FUNC, &ifr);
	if(iret < 0){
		close(fd);
		return -ERR_NOT_DEVICE;
	}
	close(fd);

	return 0;
}


/*
 * Func : sm_get_device_status
 * Context :
 *	 x < 0 error
 *	 x == 0 - driver is
 *	 x > 0 - driver initialize
 */
aint sm_get_device_status(achar *dev_name)
{
	aint iret;
	str_io_data sio;

	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_GET_DEVICE_STATUS; 
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);

	if(iret < 0) return iret;
	if(sio.base_data.para0 <= 0){
		return ERR_NOT_DEVICE_OPEN;
	}
	return 1;
}

/*
 * Func : sm_init_device
 * Context :
 */
aint sm_init_device(achar *dev_name)
{
	aint iret;
	str_io_data sio;
    
	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_INIT_DMX; 
	sio.base_data.para0 = (aulong)dmx_ucode_img;
	sio.base_data.para1 = dmx_ucode_size;
	sio.base_data.para2 = PAL_mode;
	sio.base_data.para3 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;

	sio.base_data.code = CM_INIT_COMMON_DEVICE; 
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;

	sio.base_data.code = CM_GET_QUEUE_DATA; 
	sio.base_data.para0 = 0; 		
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;

	sio.base_data.code = CM_SET_INITIAL_FLAG;
	sio.base_data.para0 = 1;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return iret;
}

/*
 * Func : sm_exit_device
 * Context : 
 */
aint sm_exit_device(achar *dev_name)
{
    	aint iret;
	str_io_data sio;
    
	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_DEINIT_DMX; 
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;

	sio.base_data.code = CM_CONTROL_ISR; 
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;
	
	return 0;
}
    

/*
* Func : sm_verify_tuner_device 
* Context :
* 	Verify device (HDM8513, HDM8513A )
* 	return value -> 0xe0 , 0xe8
*/

byte sm_verify_tuner_device( achar *dev_name )
{
	byte test_byte;
	byte qpsk_addr = 0x00;

	strcpy(gs_dev_name, dev_name); 

	/* Check device and set qpsk address */ 
	test_byte = 0xaa ;
	if( !sm_read_with_offset( dev_name, HDM8513A_ADDR, 0x07, &test_byte, 1) )
	{
		if( test_byte != 0xaa ){
			test_byte = 0xaa ;
			if( !sm_read_with_offset( dev_name, HDM8513A_ADDR, 0x1f, &test_byte, 1) ){
				if( test_byte == 0x93 ){
#ifdef TUNER_DEBUG
					printf(" This device is HDM8513A\n");
#endif
					qpsk_addr = HDM8513A_ADDR;
				}else{
#ifdef TUNER_DEBUG
					printf(" This device is HDM8513\n");
#endif
					qpsk_addr = HDM8513_ADDR;
				}
			}
		}else{
#ifdef TUNER_DEBUG
			printf(" This device is HDM8513\n");
#endif
			qpsk_addr = HDM8513_ADDR;
		}
	}

	return qpsk_addr;
}


/*
 * Func : sm_get_device_version
 * Context :
 */
aulong sm_get_device_version(achar *dev_name)
{
    	aulong ultemp;
	aint iret;
	str_io_data sio;

	strcpy(gs_dev_name, dev_name); 
    
	ultemp = 0;
	sio.base_data.code = CM_GET_VERSION_CODE;
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	ultemp = sio.base_data.para0;

    	return ultemp;	
    
}


/*
 * Func : sm_get_mac_address
 * Context :
 */
aint sm_get_mac_address(achar *dev_name, achar *addr)
{
    	aint iret,i;
	str_io_data sio;

	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_GET_MAC; 
	sio.base_data.para0 = 0;
	sio.base_data.length = 6;
	iret = sm_device_io_control(dev_name,&sio);
	if(iret < 0) return iret;

	for(i = 0; i < 6 ; i++){
		addr[i] = sio.spec.byte_buf[i];
	}

	return 0;
}



/*
 * Func : sm_tuner_get_status
 * Context : check only qpsk lock status
 * 	+ lock ok 
 * 	0 lock fail 
 * 	- not tunning
 */
aint sm_tuner_get_status(achar *dev_name)
{
	qpsk_lock_t enQlock;
	aint ferror;
	byte qpsk_addr;
	aint iret = 0;

	strcpy(gs_dev_name, dev_name);

	qpsk_addr = sm_verify_tuner_device( dev_name );

	ferror = nim_get_qpsk_lock_status( dev_name, qpsk_addr, &enQlock );
	if(enQlock != ALL_LOCKED) ferror |= 1;
	if(!ferror) iret = 1;
	else iret = 0;

	return iret;
}



/*
 * Func : sm_get_received_packets
 * Context :
 */
aint sm_get_received_packets(achar *dev_name)
{
	aint iret;
	str_io_data sio;
	aint iRxPackets;

	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_GET_RECEIVED_PACKETS;
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	iRxPackets = sio.base_data.para0;

	return iRxPackets;
}


/*
 * Func : sm_get_down_speed
 * Context :
 */
aint sm_get_down_speed(achar *dev_name)
{
	aint iret;
	str_io_data sio;
	aint iBps;

	strcpy(gs_dev_name, dev_name);

	sio.base_data.code = CM_GET_DATABPS;
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	iBps = sio.base_data.para0;
	
	return iBps;
}


/*
 * Func : sm_get_bit_error_rate
 * Context :
 * 	return value presents the error number of the 2^20 data bits.
 * 	if return value is -1 ,it means not support this func.
 *	support only HDM8513A device. 	
 */
aint sm_get_bit_error_rate(achar *dev_name)
{
	int ber;	
	byte qpsk_addr,ber_hbyte,ber_lbyte;

	strcpy(gs_dev_name, dev_name);

	qpsk_addr = sm_verify_tuner_device( dev_name );

	if( qpsk_addr != HDM8513A_ADDR ){
		ber = -1;
	}else{
		ber = 0;
		/* get high byte of BER */
		sm_read_with_offset( dev_name, qpsk_addr, 0x0e, &ber_hbyte, 1);
		/* get low byte of BER */
		sm_read_with_offset( dev_name, qpsk_addr, 0x0f, &ber_lbyte, 1);
		ber = ber_hbyte << 8 | ber_lbyte ;
	}

	return ber;
}


/*
 * Func : sm_get_signal_quality
 * Context :
 *	it provides a measure of quality of the signal processed 
 *	by the Viterbi decoder.
 *	its return value is just the SNR[dB]  
 */
double sm_get_signal_quality(achar *dev_name)
{
	byte qpsk_addr;

	strcpy(gs_dev_name, dev_name);

	qpsk_addr = sm_verify_tuner_device( dev_name );

	return nim_get_quality( dev_name , qpsk_addr , 2 );
}


/*
 * Func : sm_get_uncorrected_rs_error
 * Context :
 *	it provides a count of the uncorrected words since it was last reset.
 *	When it reaches its MAX count(255) , it rolls back to zero.
 */
aint sm_get_uncorrected_rs_error(achar *dev_name)
{
	byte qpsk_addr;
	byte rserr = 0;
	int retval = 0;

	strcpy(gs_dev_name, dev_name);

	qpsk_addr = sm_verify_tuner_device( dev_name );

	sm_read_with_offset( dev_name, qpsk_addr, 0x1e, &rserr, 1 );
	retval = (aint)rserr;

	return retval;
}


/*
 * Func : sm_get_corrected_rs_error
 * Context :
 *	it provides a count of the uncorrected words since it was last reset.
 *	Bit 7 of address 0x1c is the MSB of the 16 bits number and bit 0 of
 * 	address 0x1d is the LSB.	
 */
aint sm_get_corrected_rs_error( achar *dev_name )
{
	byte qpsk_addr,rserr_hbyte,rserr_lbyte;
	int retval  = 0;

	strcpy(gs_dev_name, dev_name);

	qpsk_addr = sm_verify_tuner_device( dev_name );

	sm_read_with_offset( dev_name, qpsk_addr, 0x1c, &rserr_hbyte, 1 );
	sm_read_with_offset( dev_name, qpsk_addr, 0x1d, &rserr_lbyte, 1 );
	retval = rserr_hbyte << 8 | rserr_lbyte;

	return retval;
}


/*
 * Func : sm_control_reset
 * Context :
 */
aint sm_control_reset(achar *dev_name)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_CONTROL_RESET;
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}



/*-----------------------------------------------------------*/
//
//						
//	PID-queue control functions 
//
//
/*-----------------------------------------------------------*/
//-------------------------------------//
// PID With CW Control API
//-------------------------------------//

/*
 * Func : sm_queue_pid_clear
 * Context :
 */
aint sm_queue_pid_clear(achar *dev_name)
{
	auint iret;
    	str_io_data sio;

	iret = 0;
	strcpy(gs_dev_name, dev_name);
	
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_CLEAR;
	sio.base_data.para1 = 0;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return iret;
}
    

/*
 * Func : sm_queue_pid_add
 * Context :
 */
aint sm_queue_pid_add(achar *dev_name, auint pid)
{
	auint iret;
    	str_io_data sio;

	iret = 0;
	strcpy(gs_dev_name, dev_name);
	
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_ADD;
	sio.base_data.para1 = pid;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_ACTIVE;
	sio.base_data.para1 = 0;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return iret;
}


/*
 * Func : sm_queue_pid_delete
 * Context :
 */
aint sm_queue_pid_delete(achar *dev_name, auint pid)
{
	auint iret;
    	str_io_data sio;

	iret = 0;
	strcpy(gs_dev_name, dev_name);
	
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_CLEAR;
	sio.base_data.para1 = pid;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
    	return 0;
}


/* Begin Change for Fixed Key */
/*
 * Func : sm_queue_pid_add_with_cw
 * Context:
 * 	evencw : control word (8 bytes)
 * 	oddcw : control word (8 bytes)
 */
auint sm_queue_pid_add_with_cw(achar *dev_name, auint pid, auchar *evencw, auchar *oddcw )
{
	auint iret,i;
    	str_io_data sio;

	iret = 0;
	strcpy(gs_dev_name, dev_name);
	
#ifdef INFACE_DEBUG
	printf("sm_queue_pid_add_with_cw:Pid=%d\n",pid);
#endif
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_ADD_WITH_CW;
	sio.base_data.para1 = pid;
	sio.base_data.length = 8;
	for( i = 0; i < 8; i++){
		sio.spec.pid_data.evencw[i] = evencw[i];
		sio.spec.pid_data.oddcw[i] = oddcw[i];
	}

#ifdef INFACE_DEBUG 
	printf("Even CW: ");
	for( i = 0; i < 8; i++){
		printf("0x%02x ",sio.spec.pid_data.evencw[i]);
	}
	printf("\n");
	printf("Odd CW: ");
	for( i = 0; i < 8; i++){
		printf("0x%02x ",sio.spec.pid_data.oddcw[i]);
	}
	printf("\n");
#endif

	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	/* Cw buffer Clear */
	for( i = 0; i < 8; i++){
		sio.spec.pid_data.evencw[i] = 0x00;
		sio.spec.pid_data.oddcw[i] = 0x00;
	}
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_ACTIVE;
	sio.base_data.para1 = 0;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return iret;
	
}

/*
 * Func : sm_queue_pid_set_cw
 * Context :
 * 	evencw : control word (8 bytes)
 * 	oddcw : control word (8 bytes)
 *
 * 	return -1 -> Invalid Pid
 */
auint sm_queue_pid_set_cw(achar *dev_name, auint pid, auchar *evencw, auchar *oddcw )
{
	auint iret,i;
    	str_io_data sio;

	iret = 0;
	strcpy(gs_dev_name, dev_name);
	
#ifdef INFACE_DEBUG
	printf("sm_queue_pid_set_cw:Pid=%d\n",pid);
#endif
    	sio.base_data.code = CM_CONTROL_PID;
	sio.base_data.para0 = PID_OPT_SET_CW;
	sio.base_data.para1 = pid;
	sio.base_data.length = 8;
	for( i = 0; i < 8; i++){
		sio.spec.pid_data.evencw[i] = evencw[i];
		sio.spec.pid_data.oddcw[i] = oddcw[i];
	}

#ifdef INFACE_DEBUG
	printf("Even CW: ");
	for( i = 0; i < 8; i++){
		printf("0x%02x ",sio.spec.pid_data.evencw[i]);
	}
	printf("\n");
	printf("Odd CW:  ");
	for( i = 0; i < 8; i++){
		printf("0x%02x ",sio.spec.pid_data.oddcw[i]);
	}
	printf("\n");
#endif
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	/* Cw buffer Clear */
	for( i = 0; i < 8; i++){
		sio.spec.pid_data.evencw[i] = 0x00;
		sio.spec.pid_data.oddcw[i] = 0x00;
	}

	iret = sio.base_data.para0;
	return iret;
}
/* End Change for Fixed Key */





/*-----------------------------------------------------------*/
//
//						
//	LNB control functions 
//
//
/*-----------------------------------------------------------*/
/*
 * Func : sm_lnb_set_power
 * Context :
 */
aint sm_lnb_set_power(achar *dev_name, auint sw)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_CONTROL_LNB;
    	if( sw ){
		sio.base_data.para0 = LNB_POWER_ON ;
	}else{
		sio.base_data.para0 = LNB_POWER_OFF ;
	}
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_lnb_set_voltage
 * Context :
 */
aint sm_lnb_set_voltage(achar *dev_name, auint voltage)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_CONTROL_LNB;
    	if( voltage ){
		sio.base_data.para0 = LNB_18V_ON ;
	}else{
		sio.base_data.para0 = LNB_13V_ON ;
	}
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}

/*
 * Func : sm_lnb_set_22khz
 * Context :
 */
aint sm_lnb_set_22khz(achar *dev_name, auint sw)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_CONTROL_LNB;
    	if( sw ){
		sio.base_data.para0 = LNB_22K_ON;
	}else{
		sio.base_data.para0 = LNB_22K_OFF;
	}
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}




/*-----------------------------------------------------------*/
//
//						
//	IP-filter control functions 
//
//
/*-----------------------------------------------------------*/
/*
 * Func : sm_filter_ip_clear
 * Context :
 */
aint sm_filter_ip_clear(achar *dev_name)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_FILTER_IP;
	sio.base_data.para0 = IPADDR_OPT_CLEAR;
	sio.base_data.para1 = 0; // IP address
	sio.base_data.para2 = 0; // PID
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}

/*
 * Func : sm_filter_ip_add
 * Context :
 */
aint sm_filter_ip_add(achar *dev_name, aulong fip)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_FILTER_IP;
	sio.base_data.para0 = IPADDR_OPT_ADD;
	sio.base_data.para1 = fip; // IP address
	sio.base_data.para2 = 0; // PID
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_filter_ip_active
 * Context :
 */
aint sm_filter_ip_active(achar *dev_name)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_FILTER_IP;
	sio.base_data.para0 = IPADDR_OPT_ACTIVE;
	sio.base_data.para1 = 0; // IP address
	sio.base_data.para2 = 0; // PID
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_filter_ip_del
 * Context :
 */
aint sm_filter_ip_del(achar *dev_name, aulong fip)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_FILTER_IP;
	sio.base_data.para0 = IPADDR_OPT_DEL;
	sio.base_data.para1 = fip; // IP address
	sio.base_data.para2 = 0; // PID
	sio.base_data.length = 0;
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}

    
  
/*-----------------------------------------------------------*/
//
//						
//	Tuner control functions 
//
//
/*-----------------------------------------------------------*/
/* 
 * Func : sm_tuner_start_tunning
 * Context :
 */
aint sm_tuner_start_tunning(achar *dev_name, str_trans_data trans)
{
	strcpy(gs_dev_name, dev_name);	

	return sm_tuner_start_tunning_action( dev_name, trans );
	
}


/*
 * Func : sm_tuner_start_tunning_action
 * Context : 
 */
aint sm_tuner_start_tunning_action( achar *dev_name, str_trans_data trans_info )
{    
   	aint iret; 
    	tuning_t tuning;
	aint lnb_high_voltage;
	tuner_lock_t enTlock;
	qpsk_lock_t enQlock;
	aint ferror;
	byte qpsk_addr;
	float fLnbValue ;
	
	/* Initialize some valriables */
	fLnbValue = 0.0;

	strcpy(gs_dev_name, dev_name);

	/* Check device and set qpsk address */ 
	qpsk_addr = sm_verify_tuner_device( dev_name );
	nim_set_values( qpsk_addr );

	nim_stop( dev_name );

	/* reset NIM tuner */
	sm_control_reset( dev_name );

	/* Reset 8513(A) Registers */
	nim_initialize( dev_name );

	/* Compute local frequency as band */
	if(trans_info.en_lnb_band == BAND_KU){
		switch( trans_info.en_lnb_type )
		{	
			case 0:fLnbValue = 0.0; break;
			case 1:fLnbValue = 9300.0 ; break; /* 9.3 GHz */
			case 2:fLnbValue = 9750.0 ; break; /* 9.75 GHz */
			case 3:fLnbValue = 10000.0 ; break; /* 10.0 GHz */
			case 4:fLnbValue = 10600.0 ; break; /* 10.6 GHz */
			case 5:fLnbValue = 10750.0 ; break; /* 10.75 GHz */
			case 6:fLnbValue = 11300.0 ; break; /* 11.30 GHz */ 
			case 7: /* Universal : KU band only */
			{
				if( trans_info.down_frequency >= 11700.0 ){
					fLnbValue = 10600.0;
					trans_info.lnb_22khz_on_off = 1;
				}else{
					fLnbValue = 9750.0;
					trans_info.lnb_22khz_on_off = 0;
				}
			}
			break; 
		}
		trans_info.local_frequency = fLnbValue;	
	}else{	/* C band */
		trans_info.local_frequency =  5150.0;	
	}

#if 0
	printf("\n-------------------------------------\n");
	printf("down Freq : %f\n", trans_info.down_frequency);
	printf("local Freq : %f\n", trans_info.local_frequency);
	printf("Symbol   : %f\n", trans_info.symbol_rate);
	printf("Polar    : %d\n", trans_info.en_lnb_polar );
	printf("Band     : %d\n", trans_info.en_lnb_band );
	printf("22K      : %d\n", trans_info.lnb_22khz_on_off );
	printf("Viterbi  : %d\n", trans_info.en_viterbi );
	printf("Lnb type : %d\n", trans_info.en_lnb_type );
	printf("Lnb Power: %d\n", trans_info.lnb_power_on_off );
	printf("--------------------------------------\n");
#endif

	lnb_high_voltage = 0;
	if(trans_info.en_lnb_polar == LNB_POL_VERTICAL){
		lnb_high_voltage = 0; /* 13V */
	}else{
		lnb_high_voltage = 1; /* 18V */
	}

	if(lnb_high_voltage){
		sm_lnb_set_voltage( dev_name, POWER_18V);
	}else{
		sm_lnb_set_voltage( dev_name, POWER_13V);
	}

	if(trans_info.lnb_power_on_off) {
		sm_lnb_set_power( dev_name, ON);
	}else{
		sm_lnb_set_power( dev_name, OFF);
	}
	
	if(trans_info.lnb_22khz_on_off) {
		sm_lnb_set_22khz( dev_name, ON);
	}else{
		sm_lnb_set_22khz( dev_name, OFF);
	}

	/* Set drift frequency */
	tuning.lLnbDriftFrequency = 4000000;

	/* set frequency */
	if(trans_info.down_frequency >= trans_info.local_frequency) {
		tuning.tuner_param.lwFrequency = (trans_info.down_frequency - trans_info.local_frequency ) * 1000000;
	} else {
	    	tuning.tuner_param.lwFrequency = (trans_info.local_frequency - trans_info.down_frequency ) * 1000000;
	}

  	tuning.tuner_param.enSawBW = TSAW_SINGLE;
        tuning.qpsk_param.lwSymbol = trans_info.symbol_rate * 1000000;
	tuning.qpsk_param.enViterbi = trans_info.en_viterbi;

	iret = nim_start( dev_name,&tuning);

	if(iret == 0){
		ferror = nim_get_lock_status( dev_name, qpsk_addr, &enTlock, &enQlock );
		if(enTlock != TUNER_PLL_LOCKED) ferror |= 1;
		if(enQlock != ALL_LOCKED) ferror |= 1;
		if(!ferror) iret = 1;
	}
	
	return iret;
}


/*
 * Func : sm_tuner_stop_tunning
 * Context :
 */
aint sm_tuner_stop_tunning(achar *dev_name)
{
	strcpy( gs_dev_name, dev_name );

	return sm_tuner_stop_tunning_action( dev_name );
}



/*
 * Func : sm_tuner_stop_tunning_action 
 * Context :
 */
aint sm_tuner_stop_tunning_action(achar *dev_name)
{
	strcpy(gs_dev_name, dev_name);

	if( !nim_stop( dev_name ) ) return 0;
	else return -1;
}





/*-----------------------------------------------------------*/
//
//						
//	Multicast-route control functions 
//
//
/*-----------------------------------------------------------*/
/*
 * Func : sm_multi_route_device
 * Context :
 */
aint sm_multi_route_device(achar *dev_name,achar *gate_dev_name)
{
    	aint iret,i;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_SET_ROUTE_DEVICE;
	sio.base_data.para0 = 0;
	sio.base_data.length = strlen(gate_dev_name);
	/* Copy gate_dev_name */
	for( i = 0; i < sio.base_data.length ; i++ ){
		sio.spec.byte_buf[i] = gate_dev_name[i];
	}
	sio.spec.byte_buf[i] = 0;
	
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_multi_ip_all
 * Context :
 */
aint sm_multi_ip_all(achar *dev_name,abool flag)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_SET_IP_ALL;
	sio.base_data.para0 = flag;
	sio.base_data.length = 0;
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_multi_ip_add
 * Context :
 */
aint sm_multi_ip_add(achar *dev_name,aulong mip)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_ADD_IP;
	sio.base_data.para0 = mip;
	sio.base_data.length = 0;
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_multi_ip_del
 * Context :
 */
aint sm_multi_ip_del(achar *dev_name,aulong mip)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_DEL_IP;
	sio.base_data.para0 = mip;
	sio.base_data.length = 0;
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}


/*
 * Func : sm_multi_ip_clear
 * Context :
 */
aint sm_multi_ip_clear(achar *dev_name)
{
    	aint iret;
    	str_io_data sio;

	strcpy(gs_dev_name, dev_name);	

    	sio.base_data.code = CM_CLEAR_IP;
	sio.base_data.para0 = 0;
	sio.base_data.length = 0;
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;
	
	return 0;
}



/*
 * Func : sm_multi_get_mac_info
 * Context :
 */
aint sm_multi_get_mac_info( achar *dev_name, str_mac_data *mmac_data )
{
    	str_io_data sio;
	aint i,j;
	aint iret;
	aint mmac_count,mmac_flags;

	/* initailize the variables */
	mmac_count = 0;
	mmac_flags = 0;

	strcpy(gs_dev_name, dev_name);

	for( i = 0; i < MAX_MAC_LISTS; i++ ){
		for( j =0; j < 6; j++){
			sio.spec.mac_data.mac[i][j] = 0;;
			mmac_data->mac[i][j] = 0;;
		}
	}

    	sio.base_data.code = CM_GET_MIP;
	sio.base_data.para0 = 0; /* command: get the count of mmac*/
	sio.base_data.para1 = 0;
	sio.base_data.para2 = 0;
	sio.base_data.length = 0;
	iret = sm_device_route_control(dev_name, &sio);
	if(iret < 0) return iret;

	mmac_count = sio.base_data.para1;
	mmac_flags = sio.base_data.para2;

	if( (mmac_flags == 1) || (mmac_flags == 2) || ( mmac_count == 0 )){
		mmac_data->count = mmac_count;
		mmac_data->filterflags = mmac_flags;

	}else{	
		sio.base_data.code = CM_GET_MIP;
		sio.base_data.para0 = 1; /* command: get the mmac */
		sio.base_data.para1 = mmac_count;
		sio.base_data.length = 6;
		iret = sm_device_route_control(dev_name, &sio);
		if(iret < 0) return iret;

		mmac_data->count = mmac_count;
		mmac_data->filterflags = mmac_flags;

		for( i = 0; i < mmac_count ; i++ ){
			memcpy( mmac_data->mac[i], sio.spec.mac_data.mac[i], 6 );
		}
	}

	return 0;
}





/*-----------------------------------------------------------*/
//
//						
//	I2C (qpsk/tuner) control functions 
//
//
/*-----------------------------------------------------------*/
/*
 * Func : sm_send_without_offset
 * Context : write function for tuner 
 */ 
aint sm_send_without_offset(achar *dev_name, aint chip_addr, abyte *ba_data, auint usize)
{
    	str_io_data sio;
	aint iret;
    	aint Count,i;
	dword dbuf[100];

    	strcpy(gs_dev_name, dev_name);

	for(i = 0; i < usize; i++){
		dbuf[i] = (dword)ba_data[i];
	}		

    	Count = usize;
    	sio.base_data.code = CM_CONTROL_TUNERIO;
	sio.base_data.para0 = IO_WriteData; /* command */
	sio.base_data.para1 = chip_addr; /* chip address */
	sio.base_data.para2 = Count; /* size */ 
	sio.base_data.length = Count * 4;
    	memcpy( sio.spec.dword_buf, dbuf, Count * 4);
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return 0;
}


/*
 * Func : sm_read_without_offset
 * Context : read function for tuner 
 */ 
aint sm_read_without_offset(achar *dev_name, aint chip_addr, abyte *ba_data, auint usize)
{    
    	str_io_data sio;
	aint iret;
    	aint Count,i;
	dword dbuf[150];

    	strcpy(gs_dev_name, dev_name);

	for(i = 0; i < usize; i++){
		dbuf[i] = (dword)ba_data[i];
	}		

    	Count = usize;
    	sio.base_data.code = CM_CONTROL_TUNERIO;
	sio.base_data.para0 = IO_ReadData; /* command */
	sio.base_data.para1 = chip_addr; /* chip address */
	sio.base_data.para2 = Count; /* size */ 
	sio.base_data.length = Count * 4;
    	memcpy( sio.spec.dword_buf, dbuf, Count * 4);
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
    	memcpy(dbuf, sio.spec.dword_buf, Count * 4);

	for(i = 0;i < usize; i++){
		ba_data[i] = (abyte)dbuf[i];
	}		
	return 0;
}


/*
 * Func : sm_send_with_offset
 * Context : write function for qpsk 
 */ 
aint sm_send_with_offset(achar *dev_name, aint chip_addr, abyte offset, abyte *ba_data, auint usize)
{
    	str_io_data sio;
	aint iret;
    	aint Count,i;
	dword dbuf[150];

    	strcpy(gs_dev_name, dev_name);

	for(i = 0; i < usize; i++){
		dbuf[i] = (dword)ba_data[i];
	}		

    	Count = usize;
    	sio.base_data.code = CM_CONTROL_QPSKIO;
	sio.base_data.para0 = IO_WriteData; /* command */
	sio.base_data.para1 = chip_addr; /* chip address */
	sio.base_data.para2 = offset; /* offset */ 
	sio.base_data.para3 = Count; /* size */ 
	sio.base_data.length = Count * 4;
    	memcpy( sio.spec.dword_buf, dbuf, Count * 4);
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;

	return 0;
}


/*
 * Func : sm_read_with_offset
 * Context : read function for qpsk 
 */ 
aint sm_read_with_offset(achar *dev_name, aint chip_addr, abyte offset, abyte *ba_data, auint usize)
{
    	str_io_data sio;
	aint iret;
    	aint Count,i;
	dword dbuf[150];

    	strcpy(gs_dev_name, dev_name);

	for(i = 0; i < usize; i++){
		dbuf[i] = (dword)ba_data[i];
	}		

    	Count = usize;
    	sio.base_data.code = CM_CONTROL_QPSKIO;
	sio.base_data.para0 = IO_ReadData; /* command */
	sio.base_data.para1 = chip_addr; /* chip address */
	sio.base_data.para2 = offset; /* offset */ 
	sio.base_data.para3 = Count; /* size */ 
	sio.base_data.length = Count * 4;
    	memcpy( sio.spec.dword_buf, dbuf, Count * 4);
	iret = sm_device_io_control(dev_name, &sio);
	if(iret < 0) return iret;
    	memcpy(dbuf, sio.spec.dword_buf, Count * 4);

	for(i = 0;i < usize; i++){
		ba_data[i] = (abyte)dbuf[i];
	}		
	return 0;
}
