/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996-2000 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    support@pftp.de
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#if defined unicos || defined SunOS
#include <fcntl.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#endif
#ifdef unicos
#include <sys/types.h>
#endif
#include <sys/wait.h>
#include <setjmp.h>
#include "main.h"

extern void sending(client_type *);
extern void receive_data(void *);
extern short get_var_from_pftprc(FILE *, const char *, char *, int);
extern int init_user_env(int);
extern char *time_string(void);
extern int my_setenv(const char *, const char *, int);
extern void free_vec(char **);
extern void send_oob_error(int, char *);
int set_filter(short, int, char **);
int set_env_vars(char *);

void client_filter_programm(void *cstruct)
{
   client_type *clientstr = (client_type *)cstruct;
   char *filter[SONAME]; /* holds the name of the filter program and its arguments */
   char str[SONAME], filvar[SONAME];
   int pfds[2];
   int pid;

   /*
    * Set client filter if '-f' was given on cmdline.
    */
   sprintf(filvar, "PFTPCFVAR%d", (*statstr)->usefilter);
   if (!set_filter(0, 0, filter)) {
      fprintf(stderr, "** Couldn't set the filter variable.\n");
      exit(PFTP_FILTER_ERR);
   }

   /*
    * Start a child process for the filter of the client.
    */
   if (pipe(pfds) < 0) {
      if (slfp) fprintf(slfp, "** pipe: %s\n", _PFTP_ERROR_ARRAY_);
      exit(PFTP_PIPE_ERR);
   }
   if ((pid = fork()) < 0) {
      if (slfp) fprintf(slfp, "** fork: %s\n", _PFTP_ERROR_ARRAY_);
      exit(PFTP_FORK_ERR);
   }
   /*
    * Here goes the child 'sending'.
    */
   if (pid) {
      int status=1;
      close(*pfds);
      (*statstr)->fd = *(pfds+1);
      free_vec(filter);
      sending(clientstr);
      if ((*statstr)->version && !(*statstr)->_STANDARD_INPUT_) write((*statstr)->fd, "0 -1 0\nB", 8);
      close((*statstr)->fd);
      while (wait((int *)&status) != pid);
      shutdown((*statstr)->ws, 1);
   }
   else {
      /*
       * Set descriptors for the filter of the client.
       */
      dup2(*pfds, 0);
      close(*(pfds+1));
      dup2((*statstr)->ws, 1);
      close((*statstr)->ws);

      /*
       * Before we start the filter sending
       * to the daemon we close stderr.
      if (slfp && (*statstr)->_PFTP_DAEMON_) close(2);
       */

      /*
       * Set variables from PFTPCFILVAR.
       */
      if (get_var_from_pftprc((FILE *)NULL, filvar, str, 0) \
       || get_var_from_pftprc((FILE *)NULL, "PFTPCFVAR", str, 0)) {
         if (!set_env_vars(str)) {
            if (slfp) fprintf(slfp, "\n** putenv: insufficient space in the environment\n");
            exit(PFTP_ENVIRON_ERR);
         }
      }

      /*
       * Execute the filter programm specified
       * by the environment variable PFTPCFILTER.
       */
      execvp(*filter, filter);
      if (slfp) fprintf(slfp, "\n** exec: %s: %s\n", *filter, _PFTP_ERROR_ARRAY_);
      free_vec(filter);
      exit(PFTP_PROG_ERR);
   }
}

jmp_buf fenv;

void server_filter_programm(void *snum)
{
   int strnum = (int)*(int *)snum;
   char *filter[SONAME]; /* holds the name of the filter program and its arguments */
   char str[SONAME], filvar[SONAME];
   int pfds[2];
   int pid;

   /*
    * Set server filter if '-f' was given
    * by the client on cmdline.
    */
   sprintf(filvar, "PFTPSFVAR%d", (*(statstr+strnum))->usefilter);
   if (!set_filter(1, strnum, filter)) {
      send((*(statstr+strnum))->ws, PFTP_WRONG_FILTER, PFTP_WRONG_FILTER_LEN, MSG_OOB);
      if (slfp) {
         if ((*statstr)->_PFTP_DAEMON_) {
            fprintf(slfp, "%s %s filter not found in list.\n",
            time_string(), (*(statstr+strnum))->rlogin);
         }
         else fprintf(slfp, "** Filter not found in list.\n");
      }
      return;
   }

   /*
    * Start a child for the filter of the server.
    */
   if (pipe(pfds) < 0) {
      fprintf(stderr, "** pipe: %s\n", _PFTP_ERROR_ARRAY_);
      return;
   }
   if ((pid = fork()) < 0) {
      fprintf(stderr, "** fork: %s\n", _PFTP_ERROR_ARRAY_);
      return;
   }
   /*
    * Here goes the child 'receive_data'.
    */
   if (pid) {
      if (slfp) fflush(slfp);
      close(*(pfds+1));
      free_vec(filter);
      (*(statstr+strnum))->fd = *pfds;
      if (!setjmp(fenv)) {
         int status=1;
         receive_data((void *)&strnum);
         close((*(statstr+strnum))->fd);
         while (wait((int *)&status) != pid);
         (*(statstr+strnum))->fd = 0;
      }
      if ((*(statstr+strnum))->fd) close((*(statstr+strnum))->fd);
      if (*((*(statstr+strnum))->incoming_lock)) {
         unlink((*(statstr+strnum))->incoming_lock);
         *((*(statstr+strnum))->incoming_lock) = '\0';
      }
      if (*((*(statstr+strnum))->uplimit_lock)) {
         unlink((*(statstr+strnum))->uplimit_lock);
         *((*(statstr+strnum))->uplimit_lock) = '\0';
      }
   }
   else {
      /*
       * Set descriptors for the filter of the server.
       */
      close(*pfds);
      dup2((*(statstr+strnum))->ns, 0);
      close((*(statstr+strnum))->ns);
      dup2(*(pfds+1), 1);
      close(*(pfds+1));
#ifdef _PFTP_DEBUG_
      dup2(fileno(slfp), 2);
#else
      /*
       * Before we start the filter for the
       * daemon we close stderr.
       */
      if ((*statstr)->_PFTP_DAEMON_) close(2);
#endif

      /*
       * Change to user's home directory.
       */
      if ((*statstr)->_PFTP_DAEMON_) {
         if (chdir((*(statstr+strnum))->pw_dir) < 0) {
            send_oob_error(strnum, (*(statstr+strnum))->pw_dir);
            exit(PFTP_CHDIR_ERR);
         }
      }

      /*
       * Set user's environment variables USER and HOME.
       */
      if (!init_user_env(strnum)) {
         if (slfp) fprintf(slfp, "** putenv: insufficient space in the environment\n");
         close((*(statstr+strnum))->ns);
         free_vec(filter);
         exit(PFTP_ENVIRON_ERR);
      }

      /*
       * Set variables from PFTPSFILVAR.
       */
      if (get_var_from_pftprc((FILE *)NULL, filvar, str, strnum) \
       || get_var_from_pftprc((FILE *)NULL, "PFTPSFVAR", str, strnum)) {
         if (!set_env_vars(str)) {
            if (slfp) fprintf(slfp, "** putenv: insufficient space in the environment\n");
            exit(PFTP_ENVIRON_ERR);
         }
      }

      /*
       * Execute the filter programm specified
       * by the environment variable PFTPSFILTER.
       */
      execvp(*filter, filter);
      if (slfp) {
         fprintf(slfp, "** execvp: %s: %s\n", *filter, _PFTP_ERROR_ARRAY_);
      }
      longjmp(fenv, 1);
      exit(PFTP_PROG_ERR);
   }
}


int set_filter(short isserver, int strnum, char **filter)
{
   int j=0, h=(*(statstr+strnum))->usefilter;
   char last = '\0', *str=NULL, *tmp=NULL;

   if (!h) return 0;
   if (isserver) {
      if ((*statstr)->_PFTP_DAEMON_ || !(str = getenv("PFTPSFILTER"))) {
         MEM_CHECK((str = (char *) calloc(SONAME, sizeof(char))));
         if (!get_var_from_pftprc((FILE *)NULL, "PFTPSFILTER", str, strnum)) {
            if ((*statstr)->_PFTP_DAEMON_) {
               if (str) free(str);
               return 0;
            }
            else {
               fprintf(stderr, "** Please set the variable PFTPSFILTER!\n");
               exit(PFTP_FILTER_ERR);
            }
         }
      }
   }
   else {
      if (!(str = getenv("PFTPCFILTER"))) {
         MEM_CHECK((str = (char *) calloc(SONAME, sizeof(char))));
         if (!get_var_from_pftprc((FILE *)NULL, "PFTPCFILTER", str, strnum)) {
            fprintf(stderr, "** Please set the variable PFTPCFILTER!\n");
            exit(PFTP_FILTER_ERR);
         }
      }
   }
   /*
    * Get memory.
    */
   MEM_CHECK((*filter = (char *)calloc(SONAME, sizeof(char))));

   /*
    * Set vector filter.
    */
   for (--((*(statstr+strnum))->usefilter),tmp=str; (*(statstr+strnum))->usefilter && *tmp; tmp++) {
      if (*tmp == ':') ((*(statstr+strnum))->usefilter)--;
   }
   if ((*(statstr+strnum))->usefilter) {
      if ((*statstr)->_PFTP_DAEMON_) {
         if (str) free(str);
         return 0;
      }
      else {
         fprintf(stderr, "** Filter %d not found. Check your filter variable.\n\n", h);
         exit(PFTP_FILTER_ERR);
      }
   }
   for (;*tmp == ' ' && *tmp; tmp++);
   for (j=h=0; *tmp && *tmp != ':'; tmp++) {
      if (*tmp != ' ') {
         *(*(filter+j)+h) = *tmp;
         h++;
      }
      else if (last != ' ') {
         *(*(filter+j)+h) = '\0';
         j++;
         h = 0;
         /*
          * Get memory.
          */
         MEM_CHECK((*(filter+j) = (char *)calloc(SONAME, sizeof(char))));
      }
      last = *tmp;
   }
   *(*(filter+j)+h) = '\0';
   if (!getenv("PFTPSFILTER") && !getenv("PFTPCFILTER")) free(str);

   /*
    * Terminating the list.
    */
   if (!**(filter+j)) *(filter+j) = (char *)NULL;
   else *(filter+j+1) = (char *)NULL;

   /*
    * Check if the filter contains something at all.
    */
   if (!*filter) return 0;

   /*
    * Print out filter type.
    * root must not know that!
    * Think twice before changing this.
    */
   if ((*statstr)->verbose && ((slfp && !((*statstr)->_PFTP_DAEMON_)) || !isserver)) {
      fprintf(slfp, "* Using `");
      for (j=0; *(filter+j); j++)
      fprintf(slfp, "%s ", *(filter+j));
      fprintf(slfp, "\b' as filter.\n");
   }
   return 1;
}

int set_env_vars(char *vars)
{
   char *stra, *strv, *stre;

   for (stra=vars; *stra; stra=stre+1) {
      for (; *stra && *stra == ' '; stra++);
      if (!*stra) return 0;
      for (strv=stra; *strv && *strv != ' ' && *strv != ':'; strv++);
      if (!*strv || *strv == ':') return 0;
      *strv = '\0';
      for (stre=++strv; *stre && *stre != ':'; stre++);
      if (!*stre) {
         *stre = '\0';
         stre--;
      }
      else *stre = '\0';
      if (my_setenv(stra, strv, 1) < 0) return 0;
   }
   return 1;
}
