package PSP::FieldSpace::Propagation;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Propagation.pm,v 1.2 2000/12/03 15:18:43 muaddib Exp $

use strict;

=head1 NAME

NAME - PSP::FieldSpace::Ephemeral

=head1 SYNOPSIS

=head1 DESCRIPTION

FIXME: this documentation is bogus.
A slightly less abstract class that inherits form PSP::FieldSpace and
defines some data propagation functions. Still relies on PSP to
compile in the default value and setup functions. This class is
directly inheritable by PSP, however.

=cut

use Exporter;
use Set::IntSpan;
use Error;
use HTMLIO::Utils qw(&html_tag);

@PSP::FieldSpace::Propagation::ISA = qw(Exporter);
@PSP::FieldSpace::Propagation::EXPORT_OK = 
  qw(&hide_in_html &get_from_cgi &encode_list &decode_list);

# A FieldSpace::Propagation object should inherit from this base class.

sub new {
  my ($proto) = @_;

  my $this = {};
  bless $this, ref($proto)||$proto;

  return $this;
}

=head2 propagate

 (string $html) propagate (Fieldspace $fs)

DESCRIPTION:

(From FS to HIDDEN HTML (+/- Storage))

Takes in the data object, C<$cgi> and returns C<$html> which contains
all the information necessary for state to be propagated across a
submitted form.

=cut

sub propagate {
  my ($this,$fs,$no_prop) = @_;
  $fs or throw Error::Simple("propagate() requires a fieldspace");
  $no_prop ||= {};

  (my $fs_class = ref($fs)) =~ s/^(FieldSpace|Pile):://;

  my (@values,@empty_current,@poss_changed);
  my $set = Set::IntSpan->new();

  my $prop_sub = sub {
    my ($field,$field_name,$gname,$index) = @_;
    my $out = "";

    # map the field name to a CGI parameter name.
    my $cgi_name = $fs->cgi_param_name($field_name,$gname,$index);

    if ($field->poss_changed_p()) {
      #print STDERR "$field_name($index) is poss_changed..\n";
      push @poss_changed, $cgi_name;
      return $out;
    } #else { print STDERR "$field_name($index) is not poss_changed..\n"; }

    # Don't propagate these hidden fields which haven't changed.
    if (! $field->changed_p()) {
      return $out;
    }

    # Propagate current values.
    $index and $set->insert($index);
    @values = $field->canonical_value();
    if (@values) {
      $out .= $this->put_keyval($fs_class,"_value::$cgi_name",@values);
    } else {
      push @empty_current, $cgi_name;
    }

    return $out;
  };
  my $html = $fs->scan_fields($prop_sub);

  for my $gname ($fs->groups()) {
    my $group = $fs->group($gname);
    my @controls = 
      ($group->first_item_n(),
       $group->n_items_per_page(),
       $group->n_pages_at_a_time());
    $html .= $this->put_keyval($group->name(),"_control",@controls);
    $html .= $this->put_keyval($group->name(),"_control_names",
			       sort keys %{$group->{export_controls}});
  }

  # store empty fields.
  @empty_current and
    $html .= $this->put_keyval($fs_class,"_value_empty",@empty_current);
  @poss_changed and
    $html .= $this->put_keyval($fs_class,"_poss_changed",@poss_changed);
  $set->empty() or
    $html .= $this->put_keyval($fs_class,"_indices",$set->run_list());

  return $html;
}

=head2 retrieve_state 

 instance
 () retrieve_state (CGI $cgi)

DESCRIPTION:

(From CGI to FS (+/- storage))

Takes the data object, $cgi in this case, and extracts all the
necessary state information, remembering it in member data. This
function does not retrieve field values, that is done in the various
init calls. It does, however, call the request_obj and rejuest_group
functions in order to ensure that the fieldspace will be properly
setup.

=cut

sub retrieve_state {
  my ($this,$fs,$cgi,$no_prop) = @_;
  $fs and $cgi or throw
    Error::Simple("retrieve() requires a fieldspace and a CGI object.");
  $no_prop ||= {};

  (my $fs_class = ref($fs)) =~ s/^(FieldSpace|Pile):://;

  # get the indicies.
  my ($indices_str) = $this->get_keyval($cgi,$fs_class,"_indices");
  my $set = Set::IntSpan->new($indices_str||"");

  # Iterate through all possible fields.
  my (@values,$cgi_name,$field_def,%seen_groups);
  for my $field_name ($fs->field_names()) {

    $field_def = $fs->field_def($field_name);

    my (@over_these,$gname);
    if ($gname = $field_def->{group}) {
      $seen_groups{$gname}++;
      @over_these = $set->elements();
    } else {
      @over_these = (undef);
    }

    for my $index (@over_these) {

      $index and $fs->set_cursor($gname,$index);
      # map the field name to a CGI parameter name.
      $cgi_name = $fs->cgi_param_name($field_name,$gname,$index);

      # Retrieve current values.
      @values = $this->get_keyval($cgi,$fs_class,"_value::$cgi_name");
      @values or next;
      my $field = $fs->put_field($field_name,@values);
    }
  }#for field_names

  for my $gname ($fs->group_names()) {
    my $group = $fs->group($gname);
    my @pairs = $this->get_keyval($cgi,$group->name(),"_control_names");
    @values = $this->get_keyval($cgi,$group->name(),"_control");
    @pairs or @values or next;
    $group->{import_controls} = [];
    for my $pair (@pairs) {
      my ($key,$value) = ($pair =~ /^([^=]+)=(.*)$/);
      push @{$group->{import_controls}}, $key => $value;
    }
    my $n;
    $n = shift @values and $group->first_item_n($n);
    $n = shift @values and $group->n_items_per_page($n);
    $n = shift @values and $group->n_pages_at_a_time($n);
    $group->{propagated_controls}++;
  }

  my ($field,$field_name,$gname,$index);

  # set current empty fields if specified.
  for my $param_name ($this->get_keyval($cgi,$fs_class,"_value_empty")) {
    ($field_name,$gname,$index) = $fs->from_cgi_param_name($param_name);
    $gname and $seen_groups{$gname}++;
    $index and $fs->set_cursor($gname,$index);
    $field = $fs->get_field($field_name,1);
    $field->set_value();
  }

  # set poss changed fields if specified.
  for my $param_name ($this->get_keyval($cgi,$fs_class,"_poss_changed")) {
    ($field_name,$gname,$index) = $fs->from_cgi_param_name($param_name);
    $gname and $seen_groups{$gname}++;
    $index and $fs->set_cursor($gname,$index);
    $field = $fs->get_field($field_name,1);
    $field->poss_changed_p(1);
  }

  # reset any seen cursors.
  for my $gname (keys %seen_groups) {
    $fs->set_cursor($gname,0);
  }
}

sub put_keyval { return; }
sub get_keyval { return; }

sub free_internals {
  my ($this) = @_;
  return;
}


# helper routines.
#
sub encode_list {
  my ($delim,@values) = @_;
  grep s/$delim/\\$delim/gso, @values;
  return join($delim,@values);
}

sub decode_list {
  my ($delim,$text) = @_;
  my @values;
  while ($text =~ s/^(.*?[^\\])$delim//s) {
    push @values, $1;
  }
  push @values, $text;
  grep s/\\$delim/$delim/gso, @values;
  return @values;
}

sub hide_in_html {
  my ($prefix,$name,@values) = @_;

  my $attr = 
    { type  => "hidden",
      name  => $prefix."::".$name,
      value => encode_list('~~',@values) };

  return html_tag("input", $attr, [qw(type name value)])."\n";
}

sub get_from_cgi {
  my ($cgi,$prefix,$name,$delim) = @_;

  my $value = $cgi->param($prefix."::".$name);
  return unless defined $value;

  return decode_list('~~', $value);
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
