package PSP::Parser::FieldSpace;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: FieldSpace.pm,v 1.11 2000/12/27 07:57:43 muaddib Exp $

use strict;

use PSP::Parser;
@PSP::Parser::FieldSpace::ISA = qw(Exporter PSP::Parser);

use Exporter;
use Error qw(:try);
use PSP::Utils;

@PSP::Parser::FieldSpace::EXPORT_OK = qw(&process_fieldspace);

use vars qw(@handled @handled_no_end @stacks @current @propagatable $ERR_STAR);
BEGIN {
  @handled = qw(fieldspace use_fieldspace reinitializefs
		define verify field);
  @handled_no_end = qw(alias import display input hidden
		       vfield vcurrent vinstantiated vverify);
  @stacks        = qw(fsname);
  @current       = qw(define verify vname);
  @propagatable = ((map { "stack_$_" } @stacks),
		   (map { "current_$_" } @current),
		   'fieldspaces');
  $ERR_STAR = '<font color="#ff0000" size="+3">*</font>'; 
};

# map lowercased words to some internalized capitalization scheme.
# unspecified classes will use a standard scheme of capitalizing 1st char.
# defined below.
use vars qw(%type_map %data_map %container_map);

=head2 fsname

 [private] instance
 () fsname (string $tag, \%attributes)

PARAMETERS
 $full_p:
   false - return the right-most segment of name.
   true  - return the entire name, concated with "."
 $no_throw:
   false - throw when we are not in a fieldspace context.
   true  - return undef when we are not in a fieldspace context.

DESCRIPTION:

=cut

sub fsname {
  my ($this,$full_p,$no_throw) = @_;

  if (!@{$this->{stack_fsname}}) {
    $no_throw or throw 
      Error::Simple("FieldSpace name expected but not found.");
    return;
  }

  if ($full_p) {
    return join(".",@{$this->{stack_fsname}});
  } else {
    return $this->{stack_fsname}->[-1];
  }
}

=head2 push_fsname and pop_fsname

 [private] instance
 () push_fsname (string $fsname)

 [private] instance
 () pop_fsname ()

DESCRIPTION:

=cut

sub push_fsname {
  my ($this,$fsname) = @_;
  push @{$this->{stack_fsname}}, $fsname;
}
sub pop_fsname {
  my ($this) = @_;
  @{$this->{stack_fsname}} or throw 
    Error::Simple("Stack underflow on fsname."."\n".backtrace());
  pop @{$this->{stack_fsname}};
}

=head2 fieldspace

 [private] instance
 () fieldspace ()

PARAMETERS

 $expect_not: (default: undef)
  undef - fail if we are not in a fieldspace context.
  0     - create fieldspace unless we already have it.
  1     - fail if we are already in a fieldspace context.

 $no_throw: (default: undef)
  false - throw on fail.
  true  - return undef on fail.

DESCRIPTION:

=cut

sub fieldspace {
  my ($this,$expect_not,$no_throw) = @_;

  # get the current name from parser state.
  my $fullfsname = $this->fsname(1,$no_throw) or return;

  # get the fieldspace if it exists.
  my $fieldspace = $this->{fieldspaces}->{$fullfsname};

  # do we or didn't we expect this fieldspace?
  if ($expect_not and $fieldspace) {
    $no_throw or throw
      Error::Simple("FieldSpace $fullfsname was already defined.");
    return;
  } elsif (!defined($expect_not) and !$fieldspace) {
    $no_throw or throw
      Error::Simple("Reference to undefined fieldspace: $fullfsname");
    return;
  }

  # if we don't already have a fieldspace, create it.
  $fieldspace ||= ($this->{fieldspaces}->{$fullfsname} = { });

  # choose the base portion of the package name.
  my $basepkg = "FieldSpace";
  $basepkg .= "::$this->{pile_name}" if $this->{pile_name};

  # define these attributes
  $fieldspace->{field_defs} ||= { };
  $fieldspace->{fullfsname} ||= $fullfsname;
  $fieldspace->{package}    ||= $basepkg."::$fullfsname";

  return $fieldspace;
}

=head2 begin_pspfieldspace

 [private] instance
 () begin_pspfieldspace (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspfieldspace {
  my($this,$tag,$attr,$tag_seq,$orig) = @_;
  $this->debug_line($orig);

  my $fsname = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  $this->push_fsname($fsname);
  my $fs = $this->fieldspace(1);

  # initialize the fieldspace.
  %$fs = (%$fs,%$attr);

  # the text is accumulated as fieldspace setup code.
  $fs->{setup} ||= "";
  $this->script_mode();
  $this->push_code_sub(\&code_pspfieldspace);
  $this->push_decl();
}

=head2 code_pspfieldspace

 [private] instance
 () code_pspfieldspace (string @code)

DESCRIPTION:

See PSP specification.

=cut

sub code_pspfieldspace {
  my ($this,$text) = @_;
  $this->debug_line($text);
  my $fieldspace = $this->fieldspace();
  $this->append_code($text,\$fieldspace->{setup});
}

=head2 end_pspfieldspace

 [private] instance
 () end_pspfieldspace (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspfieldspace {
  my ($this,$orig) = @_;

  # note which fieldspace we have.
  my $fieldspace = $this->fieldspace();

  # forget which fieldspace we have.
  $this->script_mode(0);
  $this->pop_code_sub();
  $this->pop_fsname();

  # transfer any declarations into the fieldspace.
  if (my $decl = $this->pop_decl()) {
    $fieldspace->{declaration} = $decl;
  }

  # clean up the setup code.
  $fieldspace->{setup} =~ s/\n\s*\n+/\n\n/g;

  # possibly print verbose message.
  $this->{verbose} and print " FieldSpace '$fieldspace->{name}' defined\n";
}

%type_map =
  (checkbox      => "CheckBox",
   textarea      => "TextArea",
   checkboxgroup => "CheckBoxGroup",
   possiblesets  => "PossibleSets");

%data_map =
  (anything      => "AnyThing",
   h24time       => "H24Time",
   pam           => "PAN",
   rtnumber      => "RTNumber",
   taxid         => "TaxID",
   url           => "URL",
   uscurrency    => "USCurrency",
   usphone       => "USPhone",
   uszip         => "USZip",
   y4date        => "Y4Date");

%container_map = ();

sub map_to_class {
  my ($this,$input,$package,$map) = @_;
  $input ||= 'text';
  my $class;

  # if the input contains "::", assume a real class is specified.
  if ($input =~ /::/) {
    # remove any leading :: to allow the user to indicate a package without ::
    # assume proper capitalization already.
    ($class = $input) =~ s/^:://;
  } else {
    # otherwise, if there is no capitalization..
    if ($map->{"\L$input"}) {
      # either map the input name to one with internal capitalization..
      $class = $map->{"\L$input"};
    } elsif ($input !~ /^[A-Z]/) {
      # or if there are no capitals use some standard capitalization scheme..
      $class = "\u$input";
    } else {
      # otherwise, use the input type.
      $class = $input;
    }
    # assume properly capitalized, and perpend AtomicData:: to this.
    $class = "${package}::${class}" if $package;
  }
  # return the class.
  return $class;
}

=head2 begin_pspdefine

 [private] instance
 () begin_pspdefine (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspdefine {
  my($this,$tag,$attr,$tagseq,$orig) = @_;
  $this->debug_line($orig);

  # close and flush any remaining submits
  $this->{current_define} and $this->end_pspdefine("define");
  #$this->{current_define} or throw
  #  Error::Simple("nested <$tag>s not allowed.");

  my $fieldspace = $this->fieldspace();
  my $name = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  $this->{current_define} = $name;

  # Interpret type and data attributes.
  my $presentation_type = $attr->{type};
  my $data_type         = $attr->{data};

  # assign presentation class and data class for this field.
  my $type = $this->map_to_class( $presentation_type, "HTMLIO", \%type_map );
  my $data = $this->map_to_class( $data_type,     "AtomicData", \%data_map );
  my $container;
  if ($attr->{container}) {
    $container =
      $this->map_to_class($attr->{container}, "AtomicData", \%container_map );
  }

  # Create field definition.
  my $field_def = $fieldspace->{field_defs}->{$name} ||= {};
  $field_def->{name}      = $name;
  $field_def->{type}      = $type;
  $field_def->{container} = $container;
  $field_def->{data}      = $data;
  $field_def->{valuecode} = $attr->{value};
  $field_def->{blank_ok}  = $attr->{blankok};

  $field_def->{setup}  = "";
  $this->script_mode();
  $this->push_code_sub(\&code_pspdefine);
}

=head2 code_pspdefine

 [private] instance
 () code_pspdefine (string @code)

DESCRIPTION:

See PSP specification.

=cut

sub code_pspdefine {
  my ($this,$text) = @_;
  $this->debug_line($text);
  my $fieldspace = $this->fieldspace();
  my $name = $this->{current_define} or throw
    Error::Simple("DEFINE text outside of DEFINE context?!?");
  $this->append_code($text,\$fieldspace->{field_defs}->{$name}->{setup});
}

=head2 end_pspdefine

 [private] instance
 () end_pspdefine (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspdefine {
  my($this,$tag) = @_;

  my $name = delete $this->{current_define} or throw
    throw Error::Simple("</$tag> used outside of DEFINE context.");
  my $fieldspace = $this->fieldspace();
  my $field_def = $fieldspace->{field_defs}->{$name} ||= {};

  # clean whitespace of setup.
  $field_def->{setup} =~ s/\n\s*\n/\n/g;
  $field_def->{setup} =~ s/\s+$//;
  $field_def->{setup} =~ s/^\s+//;

  # if there is a ';;', interpret leading text as initial value.
  if ($field_def->{setup} =~ s/^(.*?)\s*;;\s*//s) {
    defined $field_def->{valuecode} and throw
      Error::Simple("value as attribute and as text before ';;' in \U$tag\E");
    $field_def->{valuecode} = $1;
  }

  $this->script_mode(0);
  $this->pop_code_sub();
}

=head2 begin_pspuse_fieldspace

 [private] instance
 () begin_pspuse_fieldspace (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspuse_fieldspace {
  my($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  # acquire fault-intolerant parameters.
  my $fsname = (delete $attr->{fieldspace} or 
		delete $attr->{name}) or throw
		  Error::Simple("<$tag> requires NAME attribute.");
  $this->push_fsname($fsname);

  my $fieldspace = $this->fieldspace();

  my $form;
  if ($this->can("form") and ($form = $this->form())) {
    # if we are in a form that didn't specify its own error_page,
    # attempt to inherit an error_page of the fieldspace.
    $form->{error_page} ||= $fieldspace->{error_page};
  }

  # initialize field space code.
  $this->code("\n");
  $this->code('$fs = $pile->fieldspace("'.$fsname.'");');
}

=head2 end_form

 [private] instance
 () end_form (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspuse_fieldspace {
  my($this,$orig_txt) = @_;

  my $fieldspace = $this->fieldspace();

  my $form;
  if ($this->can("form") and ($form = $this->form())) {
    $form->{fieldspace} = $this->fsname(1);
    $this->code('$out->put($fs->propagate($_no_prop));');
  }

  $this->pop_fsname();

  delete $this->{current_sname} and throw
    Error::Simple("Unclosed SUBMIT.");
}


=head2 begin_pspfield

 [private] instance
 () begin_pspfield (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspfield {
  my ($this,$tag,$attr,$attr_seq,$orig_text) = @_;

  # Ensure fieldspace context.
  my $fs = $this->fieldspace();

  my $name = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  $name = quote_bareword($name);

  # The main purpose of this tag:
  $this->code("\n");
  $this->code("\$_field = \$fs->get_field($name);");

  # parse common attributes.
  my $change    = (defined($attr->{change}) 
		   ? quote_bareword($attr->{change}) : undef);
  my $value     = (defined($attr->{value}) 
		   ? quote_bareword($attr->{value}) : $change);
  my $noformat  = bool_att($attr->{'no-format'}) ? 1 : 0;
  my $index     = quote_bareword($attr->{index});
  my $delimiter = quote_bareword($attr->{delimiter});

  # perform common functions.
  defined $change and $this->code("\$_field->poss_changed_p(1);");
  defined $value  and $this->code("\$_field->set_value($value);");

  return ($name,$index,$value,$change,$noformat,$delimiter);
}


=head2 end_pspfield

 [private] instance
 () end_pspfield (string $tag, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspfield {
  my ($this,$orig_txt) = @_;
  # this could close a scope.. but shouldn't?
}

=head2 begin_pspdisplay

 [private] instance
 () begin_pspdisplay (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspdisplay {
  my($this,$tag,$attr,$attr_seq,$orig_text) = @_;

  my ($field_name,$index_name,$value,$change,$noformat,$delimiter) =
    $this->begin_pspfield($tag,$attr,$attr_seq,$orig_text);

  my $method = $attr->{raw} ? "raw_view" : "html_view";

  my $out = '$_field->'.$method."($noformat,$index_name,$delimiter)";

  $this->code("\$out->put($out);");
}

=head2 begin_pspalias

 [private] instance
 () begin_pspalias (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

See C<handle_fields()> for functionality.

=cut

sub begin_pspalias {
  my($this,$tag,$attr,$attr_seq,$orig_text) = @_;

  my ($field_name) = 
    $this->begin_pspfield($tag,$attr,$attr_seq,$orig_text);

  $this->code('$out->put($_field->alias());');
}

=head2 begin_pspinput

 [private] instance
 () begin_pspinput (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

See C<handle_fields()> for functionality.

=cut

sub begin_pspinput {
  my($this,@args) = @_;
  my $attr = $args[1];

  my ($field_name,$index_name,$value,$change,$noformat,$delimiter) =
    $this->begin_pspfield(@args);

  if (defined $attr->{edit_test}) {
    $this->code("if ($attr->{edit_test}) {\n");
    $this->code_add_indent("  ");
  }

  # accumulate code here.
  my @code;

  my $dyndisp;
  if (my $gname = $this->{stack_ddisplay}->[-1]) {
    $dyndisp = $this->{dyndisp}->{$gname};
  }
  my $numcode = $dyndisp->{numvar}
    ? ",".quote_bareword($dyndisp->{numvar}) : "";

  # check for errors and possibly insert a star.
  unless (bool_att($attr->{nomark})) {
    push @code,
	("if (\$fs->in_error('field',$field_name$numcode)) {",
	 "  \$out->put('$ERR_STAR');",
	 "}");
  }

  push @code,
	('$out->put($_field->html_input('."$index_name,$delimiter));",
	 "\$_field->poss_changed_p(1);",
	 "for my \$val (\$_field->value($index_name)) {",
	 "  \$_no_prop->{$field_name}->{\$val}++;",
	 "}");

  map { $this->code($_) } @code;

  if (defined $attr->{edit_test}) {
    $this->code_del_indent("  ");
    $this->code("} else {\n");
    $this->code_add_indent("  ");
    $this->begin_pspdisplay(@args);
    $this->code_del_indent("  ");
    $this->code("}\n");
  }
}

=head2 begin_psphidden

 [private] instance
 () begin_psphidden (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

See C<handle_fields()> for functionality.

=cut

sub begin_psphidden {
  my($this,$tag,$attr,$attr_seq,$orig_text) = @_;

  my ($field_name,$index_name,$value,$change,$noformat,$delimiter) =
    $this->begin_pspfield($tag,$attr,$attr_seq,$orig_text);

  my @code = 
    ("\$out->put(\$_field->html_hidden($index_name));",
     "for my \$val (\$_field->value($index_name)) {",
     "  \$_no_prop->{$field_name}->{\$val}++;",
     "}");

  map { $this->code($_) } @code;
}

####
#### SUBMIT TAGS
####

=head2 begin_pspvfield

 [private] instance
 () begin_pspvfield (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspvfield {
  my($this,$tag,$attr,$attr_seq,$orig_txt) = @_;

  my $fsname = $this->fsname(1);
  my $fieldspace = $this->fieldspace();
  my $name = $attr->{name} or throw
    Error::Simple("<$tag> requires a NAME attribute.");
  $fieldspace->{field_defs}->{$name} or throw
    Error::Simple("Undefined reference to $fsname field, $name");
  my $sname = $this->{current_sname};
  $fieldspace->{was_ver}->{$sname}++;

  $name = quote_bareword($name);

  if (bool_att($attr->{ignore})) {
    $this->push_submit_code("    \$fs->add_verify('rfield',$name);");
  } else {
    $this->push_submit_code("    \$fs->add_verify('field',$name);");
    $this->push_submit_code('    $errors_to_check++;');
  }
}

=head2 begin_pspvcurrent

 [private] instance
 () begin_pspvcurrent (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspvcurrent {
  my ($this, $tag, $attr) = @_;

  my $fieldspace = $this->fieldspace();
  my $sname = $this->{current_sname};
  $fieldspace->{was_ver}->{$sname}++;

  if (bool_att($attr->{ignore})) {
    $this->push_submit_code('    $fs->remove_poss_changed_errors();');
  } else {
    $this->push_submit_code('    $fs->add_poss_changed_to_verify();'); 
    $this->push_submit_code('    $errors_to_check++;');
  }
}

=head2 begin_pspvinstantiated

 [private] instance
 () begin_pspvinstantiated (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspvinstantiated {
  my ($this, $tag, $attr) = @_;

  my $fieldspace = $this->fieldspace();
  my $sname = $this->{current_sname};
  $fieldspace->{was_ver}->{$sname}++;

  if (bool_att($attr->{ignore})) {
    $this->push_submit_code('    $fs->remove_instantiated_errors();');
  } else {
    $this->push_submit_code('    $fs->add_instantiated_to_verify();'); 
    $this->push_submit_code('    $errors_to_check++;');
  }
}

=head2 begin_pspvverify

 [private] instance
 () begin_pspvverify (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspvverify {
  my($this,$tag,$attr,$attr_seq,$orig_txt) = @_;

  my $fieldspace = $this->fieldspace();
  my $sname = $this->{current_sname} or throw
    Error::Simple("$tag tag outside of SUBMIT context.");
  my $name = $attr->{name} or throw
    Error::Simple("<$tag> requires a NAME attribute.");
  if (!$fieldspace->{verify_defs}->{$name}) {
    my $fsname = $this->fsname(1);
    throw Error::Simple("<$tag> refers to an unknown $fsname verify: '$name'");
  }
  $fieldspace->{was_ver}->{$sname}++;

  $name = quote_bareword($name);

  if (bool_att($attr->{ignore})) {
    $this->push_submit_code("    \$fs->add_verify('rverify',$name);");
  } else {
    $this->push_submit_code("    \$fs->add_verify('verify',$name);");
    $this->push_submit_code('    $errors_to_check++;');
  }
}

=head2 begin_pspverify

 [private] instance
 () begin_pspverify (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspverify {
  my ($this,$tag,$attr) = @_;

  my $fieldspace = $this->fieldspace();
  delete $this->{current_vname} and throw
    Error::Simple("<$tag> called in previous VERIFY context");

  my $name = $this->{current_vname} = $attr->{name};
  my $test = $attr->{test};
  (defined $name and defined $test) or throw
    Error::Simple("<$tag> requires NAME and TEST attributes.");

  my $verify = $fieldspace->{verify_defs}->{$name} ||= {};
  $verify->{test} = $test;
  $verify->{groups} = $attr->{groups};

  $this->{current_verify} = "";
  $this->text_mode();
  $this->push_code_sub(\&code_pspverify);

  my $fsname = $this->fsname(1);
}

=head2 end_pspverify

 [private] instance
 () end_pspverify (string $tag)

DESCRIPTION:

See PSP specification.

This will set the C<text_sub> reference to C<text_pspfieldspace> since we
are still necessarily in the fieldspace.

=cut

sub end_pspverify {
  my ($this,$tag) = @_;

  my $vname = delete $this->{current_vname} or throw
    Error::Simple("<$tag> used outside of VERIFY context.");

  my $fieldspace = $this->fieldspace();
  my $verify = $fieldspace->{verify_defs}->{$vname} ||= {};

  # flush the HTML into current_verify.  pop the code_sub.
  $this->flush_text();
  $this->pop_code_sub();
  $this->text_mode(0);

  # Assign the verify code.
  $verify->{name} = $vname;
  $verify->{code} = delete $this->{current_verify};
}

sub code_pspverify {
  my ($this,$text) = @_;
  $this->debug_line($text);
  $this->append_code($text,\$this->{current_verify});
}

=head2 begin_pspreinitializefs

 [private] instance
 () begin_pspreinitializefs (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspreinitializefs {
  my ($this, $tag, $attr, $attr_seq, $orig_txt) = @_;

  my $fieldspace = $this->fieldspace();

  defined $attr->{test} and
    $this->begin_pspif($tag,$attr,$attr_seq,$orig_txt);

  $this->code('  $fs->initialize($cgi,ref($fs));');

  defined $attr->{test} and 
    $this->end_pspif($tag);
}

=head2 end_pspreinitializefs

 [private] instance
 () end_pspreinitializefs (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspreinitializefs {
  my ($this, $orig_txt) = @_;
  # we don't scope this anymore.
}

=head2 begin_pspimport

 [private] instance
 () begin_pspimport (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspimport {
  my($this,$tag,$attr,$attr_seq,$orig_txt) = @_;

  my $fs = $this->fieldspace();
  my $group = $this->{stack_group}->[-1];

  if ($attr->{vars}) {
    $fs->{share_vars} = $fs->{share_vars} ? "$fs->{share_vars}," : "";
    $fs->{share_vars} .= $attr->{vars};
  }
  if ($attr->{modules}) {
    $fs->{use} = $fs->{use} ? "$fs->{use}," : "";
    $fs->{use} .= $attr->{modules};
  }
}

##
## MISC.
##

#sub fs_init_code {
#  my ($this, $clean) = @_;
#  $clean ||= 0;
#
#  my $fsname = $this->fsname(1) or throw
#    Error::Simple("fs_init_code called outside of FIELDSPACE.");
#  my $fieldspace = $this->fieldspace() or throw
#    Error::Simple("Internal error: fsname=$fsname but no FIELDSPACE?!?");
#
#  $this->code('$fs = $pile->fieldspace("'.$fsname.'",$cgi);');
#
#  $this->group_init_code($submit_p);
#
#  #setup change flags
#  my $fields = $this->{pre_parsed}->{change_flag_data} || {};
#  for my $name (keys %$fields) {
#    my $var_name = $fields->{$name};
#    $name = quote_bareword($name);
#    $this->code("$var_name = \$fs->changed_p($name,1);");
#  }
#}

=head2 process_fieldspace

 [private] instance
 () process_fieldspace (string @code)

DESCRIPTION:

See PSP specification.

=cut

sub process_fieldspace {
  my ($fieldspace) = @_;

  my $fsname      = $fieldspace->{fullfsname};
  my $package     = $fieldspace->{package};
  my $field_defs  = $fieldspace->{field_defs}  || {};
  my $verify_defs = $fieldspace->{verify_defs} || {};
  my $group_defs  = $fieldspace->{group_defs}  || {};

  my $out = "";

  # identify a separate package for each fieldspace.
  $out .= ("#" x 78)."\n";
  $out .= "package $package;\n\n";

  # determine the propagation class.
  my $prop_class = $fieldspace->{propagation} || 'PSP::FieldSpace::Ephemeral';

  # collect any modules we need to use.
  my $isa = $fieldspace->{isa} || "PSP::FieldSpace";
  my @modules = ("strict","Error=:try",$isa,$prop_class,"PSP::Output");
  push @modules, $fieldspace->{use} if $fieldspace->{use};
  my %field_classes;
  for my $field_name (keys %$field_defs) {
    my $field_def = $field_defs->{$field_name};
    $field_classes{$field_def->{data}}++ if $field_def->{data};
    $field_classes{$field_def->{type}}++ if $field_def->{type};
    $field_classes{$field_def->{container}}++ if $field_def->{container};
  }
  $field_classes{'AtomicData::AnyThing'}++;
  push @modules, (sort keys %field_classes);

  # use those modules, and define inheritance relationships.
  for my $str1 (@modules) {
    for my $str2 (split /,+/, $str1) {
      my @s = split /=/, $str2 or next;
      $out .= "use ".(shift @s);
      $out .= " qw(@s)" if @s;
      $out .= ";\n";
    }
  }
  $out .= "use vars qw(\@ISA);\n";
  $out .= "BEGIN { \@ISA = qw($isa) }\n\n";

  # initialize the list of shared varables.
  (my $share_vars = $fieldspace->{share_vars}||"") =~ s/,/ /g;
  $share_vars and $share_vars .= " ";
  $share_vars .= '$cgi';
  $out .= "# shared variables:\n";
  $out .= "use vars qw($share_vars);\n\n";

  # print any declaration code.
  if ($fieldspace->{declaration}) {
    $out .= ("#" x 78)."\n";
    $out .= $fieldspace->{declaration}."\n\n";
  }

  # define field definition methods by name
  $out .= ("#" x 78)."\n";
  for my $def (sort keys %$field_defs) {
    $out .= "sub f_$def { return shift->get_field('$def') }\n";
  }
  $out .= "\n";

  # Begin the constructor.
  $out .= join("\n",
	(("#" x 78),
	 'use vars qw(%field_defs %group_defs %verify_defs);',
	 "sub new {",
	 '  my ($proto,$propagation) = @_;',
	 "  \$propagation ||= '$prop_class';",
	 '  my $this = $proto->SUPER::new($propagation);',
	))."\n\n";

  $out .= join("\n",
	('  %{$this->{field_defs}} = (',
	 '    %{$this->{field_defs}},',
	 '    %'.$package.'::field_defs',
	 '  );',
	 '  %{$this->{group_defs}} = (',
	 '    %{$this->{group_defs}},',
	 '    %'.$package.'::group_defs',
	 '  );',
	 '  %{$this->{verify_defs}} = (',
	 '    %{$this->{verify_defs}},',
	 '    %'.$package.'::verify_defs',
	 '  );'
	))."\n";

  # close constructor.
  $out .= "  return \$this;\n";
  $out .= "}\n\n";

  my (@defs);

  # Assign field definitions.
  @defs = ();
  $out .= "\%field_defs = (";
  if (%$field_defs) {
    for my $field_name (sort keys %$field_defs) {
      my $field_def = $field_defs->{$field_name};
      (my $value = quote_bareword($field_def->{valuecode})) =~ s/\'/\\\'/g;
      $field_def->{setup_sub} = "${package}::field_setup_$field_name";
      $field_def->{valuecode} and
        $field_def->{set_sub} = "${package}::field_set_$field_name";

      push @defs, join("\n",
	("  \"$field_name\" => {",
	 "     'type'      => '$field_def->{type}',",
	 "     'data'      => '$field_def->{data}',",
	 ($field_def->{container} ?
	  "     'container' => '$field_def->{container}',\n" : "").
	 (defined $field_def->{blank_ok} ?
	  "     'blank_ok'  => '$field_def->{blank_ok}',\n" : "").
	 ($field_def->{group} ?
	  "     'group'     => '$field_def->{group}',\n" : "").
	 ($field_def->{set_sub} ?
	  "     'set_sub'   => '$field_def->{set_sub}'," : "").
	 "     'setup_sub' => '$field_def->{setup_sub}',",
	 "  }"));
    }
    $out .= "\n".join(",\n",@defs);
  }
  $out .= ");\n\n";

  # Assign group definitions.
  @defs = ();
  $out .= "\%group_defs = (";
  if (%$group_defs) {
    for my $gname (sort keys %$group_defs) {
      my $group = $group_defs->{$gname};
      (my $setup = $group->{setup} || "") =~ s/\'/\\\'/g;

      push @defs, join("\n",
	("  \"$gname\" => {",
	 "    'name'     => '$gname',",
	 "    'grpvar'   => '$group->{grpvar}',",
	 "    'objvar'   => '$group->{objvar}',",
	 "    'numvar'   => '$group->{numvar}',",
	 "    'number'   => ".($group->{number}||0).",",
	 "    'maxnum'   => ".($group->{maxnum}||0).",",
	 "    'field_names'=>[qw(".join(" ",@{$group->{field_names}}).")],",
	 "    'package'  => '$group->{package}',",
	 "    'dummy_ok' => ".($group->{dummy_ok}||0).",",
	 "    'setup_sub'=> '$group->{package}::setup',",
	 "  }"));
    }
    $out .= "\n".join(",\n",@defs);
  }
  $out .= ");\n\n";

  # Assign verify definitions
  @defs = ();
  $out .= "\%verify_defs = (";
  if (%$verify_defs) {
    for my $verify_name (sort keys %$verify_defs) {
      my $verify_def = $verify_defs->{$verify_name};
      (my $test = $verify_def->{test})   =~ s/\'/\\\'/g;
      $verify_def->{disp_method} = "${package}::vdisp__$verify_name";
      push @defs, join("\n",
	("  '$verify_name' => {",
	 "    'test' => '$test',",
	 "    'disp_method' => '$verify_def->{disp_method}'",
	 "  }"
	));
    }
    $out .= "\n".join(",\n",@defs);
  }
  $out .= ");\n\n";

  # first, generate the fieldspace setup.
  $out .= join("\n",
	(("#" x 78),
	 "sub setup {",
	 '  my ($fs) = @_;',
	 '  $fs->share([qw('.$share_vars.')]);',
	 $fieldspace->{setup},
	 '}'
	))."\n\n";

  #next, generate the individual field setups.
  for my $field_name (sort keys %$field_defs) {
    my $field_def = $field_defs->{$field_name};
    $out .= (("#" x 78)."\n".
	     "sub $field_def->{setup_sub} {\n".
	     '  my ($fs,$field) = @_;'."\n");

    $field_def->{setup} =~ /\$field/ and
      $out .= "  \$field ||= \$fs->get_field('$field_name',1);\n";

    if (my $gname = $field_def->{group}) {
      my $group_def = $group_defs->{$gname} || {};
      my $grpvar = $group_def->{grpvar} || '$group';
      my $numvar = $group_def->{numvar} || '$index';
      my $objvar = $group_def->{objvar} || '$obj';
      my $regexp;
      ($regexp = "($grpvar|$numvar|$objvar)") =~ s/\$/\\\$/g;
      $field_def->{setup} =~ /$regexp/ and
	$out .= "  my $grpvar = \$fs->group('$gname') or return;\n";
      ($regexp = "($numvar|$objvar)") =~ s/\$/\\\$/g;
      $field_def->{setup} =~ /$regexp/ and
	$out .= "  my $numvar = $grpvar->cursor();\n";
      ($regexp = $objvar) =~ s/\$/\\\$/g;
      $field_def->{setup} =~ /$regexp/ and
	$out .= "  my $objvar = $grpvar->object($numvar);\n";
    }
    $field_def->{setup} and
      $out .= "  ".$field_def->{setup}."\n";
    $out .= "}\n\n";

    $field_def->{set_sub} or next;

    $out .= (("#" x 78)."\n".
	     "sub $field_def->{set_sub} {\n".
	     '  my ($fs,$field) = @_;'."\n".
	     "  \$field ||= \$fs->get_field('$field_name',1);\n");

    if (my $gname = $field_def->{group}) {
      my $group_def = $group_defs->{$gname} || {};
      my $grpvar = $group_def->{grpvar} || '$group';
      my $numvar = $group_def->{numvar} || '$index';
      my $objvar = $group_def->{objvar} || '$obj';
      my $regexp;
      ($regexp = "($grpvar|$numvar|$objvar)") =~ s/\$/\\\$/g;
      $field_def->{valuecode} =~ /$regexp/ and
	$out .= "  my $grpvar = \$fs->group('$gname') or return;\n";
      ($regexp = "($numvar|$objvar)") =~ s/\$/\\\$/g;
      $field_def->{valuecode} =~ /$regexp/ and
	$out .= "  my $numvar = $grpvar->cursor();\n";
      ($regexp = $objvar) =~ s/\$/\\\$/g;
      $field_def->{valuecode} =~ /$regexp/ and
	$out .= "  my $objvar = $grpvar->object($numvar);\n";
    }
    $field_def->{valuecode} and
      $out .= '  $field->set_value('.$field_def->{valuecode}.");\n";
    $out .= "}\n\n";
  }

  # now, generate the verify display routines.
  for my $verify_name (sort keys %$verify_defs) {
    my $verify_def = $verify_defs->{$verify_name};
    $out .= join("\n",
	(("#" x 78),
	 "sub $verify_def->{disp_method} {",
	 '  my ($fs, @args) = @_;',
	 '  my $out = PSP::Output->new();',
	 $verify_def->{code},
	 '  $out or',
	 "   \$out->set(\"VERIFY '$verify_name' failed without reason.\");",
	 '  return $out->get();',
	 '}',
	 ""))."\n";
  }

  return $out;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
