package PSP::Parser::Message;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Message.pm,v 1.2 2001/01/21 12:24:19 muaddib Exp $

use strict;

use Error qw(:try);

use PSP::Parser;
@PSP::Parser::Message::ISA = qw(PSP::Parser);

use vars qw(@handled @propagatable);
BEGIN {
  @handled = qw(message);
  @propagatable = qw(current_message);
};

=head2 begin_pspmessage

 [private] instance
 () begin_pspmessage (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspmessage_def {
  my ($this, $tag, $attr, $attrseq, $orig) = @_;
  throw Error::Simple("$tag not supported yet.");
}

sub end_pspmessage_def {
  my ($this, $tag) = @_;
}

sub begin_pspmessage {
  my ($this, $tag, $attr) = @_;

  $this->{current_message} and throw
    Error::Simple("Nested <$tag>s not allowed.");

  my $matches = 0;
  map { $matches++ if $attr->{$_} } qw(sub method throw);
  if ($matches != 1) {
    throw Error::Simple("<$tag> requires one attribute of: ".
			"SUB, METHOD, or THROW.");
  }

  # clear out any previous text.
  $this->flush_text();

  # transfer attributes to message struct.
  my $message = $this->{current_message} = { %$attr };
}

=head2 end_pspmessage

 [private] instance
 () end_pspmessage (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspmessage {
  my ($this, $tag, $attr) = @_;

  my $message = delete $this->{current_message} or throw
    Error::Simple("Nested <$tag>'s not allowed.");

  # populate the text field.
  $this->flush_message($message);

  # if a test was specified, perform it.
  my $test;
  if ($test = delete $message->{test}) {
    $this->code("if ($test) {");
    $this->begin_pspblock('message');
  }

  my $text = $message->{text} || "''";

  if (my $type = delete $message->{throw}) {
    my $args = "";
    $args .= "-text=>$text";
    for my $dticked (qw(line file)) {
      next unless defined $message->{$dticked};
      $args and $args .= ",\n".$this->code_indent()."    ";
      $args .= "-$dticked=>".add_dticks($message->{$dticked});
    }
    $this->code("throw $type->new($args);");
  } elsif (my $sub = delete $message->{sub}) {
    $this->code("$sub($text)\n");
  } elsif (my $method = delete $message->{method}) {
    $this->code("\$pile->$method($text)\n");
  } else {
    throw Error::Simple("SUB, MESSAGE, or THROW required in MESSAGE context.");
  }

  # if we're in a test block, exit the block.
  $test and $this->end_pspblock('message');
}

sub flush_message {
  my ($this,$message) = @_;

  $message or throw
    Error::Simple("flush_message() called outside of MESSAGE context.");

  my $text = $this->{text_to_flush};
  $this->{text_to_flush} = "";
  $this->debug_line($text);
 
  if ($message->{text}) {
    $message->{text} .= ".\n";
  } elsif (! defined $message->{text}) {
    $message->{text} = "";
  }
  $message->{text} .= join(".\n",$this->flush_text_chunks($text));
}

=head2 begin_pspthrow

  [private] instance
  () begin_pspthrow (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspthrow {
  my ($this,$tag,$attr) = @_;

  $this->{current_throw} and throw
    Error::Simple("Nested THROWs not allowed.");

  my $type = delete $attr->{type} || 'Error::Simple';
  $attr->{throw} = $type;

  $this->begin_pspmessage($tag,$attr);
}

=head2 end_pspthrow

 [private] instance
 () end_pspthrow (string $tag);

DESCRIPTION:

See PSP specification.

=cut

sub end_pspthrow {
  my ($this,$tag) = @_;
  $this->end_pspmessage($tag);
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
