/* *************************************************************************
  Module:        ptp.c 
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Prints the hard-coded test pages. The test pages (or parts
                 of them) are compressed with zlib and base64 encoded. These
                 are found in the include files. The routines in this file
                 decode and uncompresses them during output. For compressing
                 and encoding something into what these functions can
                 decode and uncompress, see the procedure outlined in the
                 file mydeflate.c

                 zlib is a free library,
                 Copyright (C) 1995-1998 Jean-loup Gailly and Mark Adler

                 The base64 decoding is done with included source code
                 from the uudecode program. See NOTICE in the read_base64()
                 function header below.
  Changes:

****************************************************************************
                 COPYRIGHT (C) 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "lexgui.h"
#include "t_image.h"
#include "ttftest.h"

/* -------------------------------------------------------------------------
   For inout_struct, defined in lexgui.h:

   Decoding and decompressing on the fly (a small chunk at a time):
   ebuf[] stores a UBUFSIZE of encoded characters. This is decoded and
   stored into in_buf[], but the decoded characters are still deflated
   (compressed). These are inflated (decompressed) and stored into
   out_buf[], at which point this chunk is ready for writing out. Doing
   it this way saves memory while at the same time allowing the storage
   of the compressed item in the .h file in ASCII friendly format. The
   ASCII friendly format is not really necessary to generate the final
   product, but I did this to have an example I might need later for
   something else.

   Note in_buf[] takes 3/4 the number of characters as ebuf[] because
   base64 encoding stores 4 encoded characters for every 3 decoded
   characters. Having said this, in_buf[] will not even take the entire
   3/4 of ebuf[] in 1 pass because read_base64() breaks out of its inner
   loop when it encounters a carriage return (that's not part of the
   encoded data but rather part of the array uuestr). This happens after
   every 60 characters (the length of a base64 encoded line). So in_buf
   will be 45 characters max on each pass of read_base64()'s inner loop.
   So call_inflate() will process 45 characters max of input each time
   it is called (which will itself loop if out_buf[] is too small on
   1 pass to store the inflated data).
   ------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------
        check_err()
   ------------------------------------------------------------------------- */
int check_err(int err, char *msg, msgbox_struct *msgbox)
{
  char message[50];

  if(err != Z_OK)
  {
    sprintf(message, "Zlib %s error: %d", msg, err);
    put_msg(msgbox, message, RED, 0);
    return(1);
  }
  return(0);
}
/* -------------------------------------------------------------------------
        inflate_init()
   ------------------------------------------------------------------------- */
int inflate_init(inout_struct *inout, msgbox_struct *msgbox)
{
  int i;
  int err;

  for(i = 0; i < (2 * UBUFSIZE); i++)
    inout->ebuf[i] = 0;

  for(i = 0; i < UBUFSIZE; i++)
  {
    inout->in_buf[i] = 0;
    inout->out_buf[i] = 0;
  }
  inout->z.zalloc = (alloc_func)0;
  inout->z.zfree = (free_func)0;
  inout->z.opaque = (voidpf)0;
  inout->in_buf_len = 0;

  err = inflateInit(&(inout->z));
  return(check_err(err, "inflateInit", msgbox));
}
/* -------------------------------------------------------------------------
        inflate_end()
   ------------------------------------------------------------------------- */
void inflate_end(inout_struct *inout, msgbox_struct *msgbox)
{ 
  int err;
  err = inflateEnd(&(inout->z));
  check_err(err, "inflateEnd", msgbox);
}
/* -------------------------------------------------------------------------
        call_inflate()
   ------------------------------------------------------------------------- */
int call_inflate(inout_struct *inout, msgbox_struct *msgbox)
{
  int count, err;

  inout->z.next_out = inout->out_buf;
  inout->z.avail_out = UBUFSIZE;

  inout->z.next_in = inout->in_buf;
  inout->z.avail_in = inout->in_buf_len; 

  for ( ; ; )
  {
    if (inout->z.avail_in == 0 )  
      break;

    err = inflate(&(inout->z), Z_NO_FLUSH);
    count = UBUFSIZE - inout->z.avail_out;
    if (count)
      fwrite(inout->out_buf, 1, count, inout->out);

    if (err == Z_STREAM_END) break;
    if(check_err(err, "inflate", msgbox))
      return(1);
    inout->z.next_out = inout->out_buf;
    inout->z.avail_out = UBUFSIZE;
  }
  return(0);
}
/* -------------------------------------------------------------------------
        read_base64_header() Search for header line, then increment index
                             to beginning of next line and return index.
                             This is just an error check and index find;
                             the header is disregarded.
   ------------------------------------------------------------------------- */
int read_base64_header(char *uuestr, msgbox_struct *msgbox)
{
  int i, hi, len, start, nextstart, end;
  char hbuf[MAX_U_LINELENGTH + 1];
  int mode;
  char message[50];

  nextstart = 0;
  start = 0;
  len = strlen(uuestr);
  hi = 0;
  while (1)
  {
    if(len < start + MAX_U_LINELENGTH)
      end = len;
    else
      end = start + MAX_U_LINELENGTH;

    for(i = start; i < end; i++)
    {
      if(uuestr[i] == '\n')
      {
        nextstart = i + 1; 
        break;
      }
      hbuf[hi++] = uuestr[i];
    }
    hbuf[hi] = 0;
    hi = 0;
    if (strncmp (hbuf, "begin", 5) == 0)
    {
      if (sscanf (hbuf, "begin-base64 %o %s", &mode, hbuf) == 2)
        break;
      else
      {
        sprintf(message, "UUE error: No 'begin-base64' line."); 
        put_msg(msgbox, message, RED, 0);
        return(0);
      }
    }
    start = nextstart;
    if (memcmp (hbuf, "====", 4) == 0)
    {
      sprintf(message, "UUE error: No 'begin-base64' line anywhere.");
      put_msg(msgbox, message, RED, 0);
      return(0);
    }
  }
  return(nextstart);
}
/* -------------------------------------------------------------------------
        read_base64() Decode a base64 encoded stream.

   ********************************* NOTICE ********************************
   The code in this function was taken from source code in the uudecode
   program. It was modified to only read base64 encoding. The input and
   output methods were also modified.

   The uudecode program is:

   Copyright (c) 1983 Regents of the University of California.
   Copyright (C) 1994, 1995 Free Software Foundation, Inc.
   All rights reserved.

   The uudecode program is licensed with the GNU General Public License.
   A notice is required to use this source code. The notice is contained
   in the README file accompanying Pup.

   The source code used came from:
   ftp://ftp.gnu.org/pub/gnu/sharutils/sharutils-4.2.1.tar.gz
   *************************************************************************
   ------------------------------------------------------------------------- */
void read_base64(int index, char *uuestr, inout_struct *inout, 
                 msgbox_struct *msgbox)
{
  char message[50];
  int last_data;
  unsigned char *p;
  static const char b64_tab[256] =
  {
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*000-007*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*010-017*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*020-027*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*030-037*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*040-047*/
    '\177', '\177', '\177', '\76',  '\177', '\177', '\177', '\77',  /*050-057*/
    '\64',  '\65',  '\66',  '\67',  '\70',  '\71',  '\72',  '\73',  /*060-067*/
    '\74',  '\75',  '\177', '\177', '\177', '\100', '\177', '\177', /*070-077*/
    '\177', '\0',   '\1',   '\2',   '\3',   '\4',   '\5',   '\6',   /*100-107*/
    '\7',   '\10',  '\11',  '\12',  '\13',  '\14',  '\15',  '\16',  /*110-117*/
    '\17',  '\20',  '\21',  '\22',  '\23',  '\24',  '\25',  '\26',  /*120-127*/
    '\27',  '\30',  '\31',  '\177', '\177', '\177', '\177', '\177', /*130-137*/
    '\177', '\32',  '\33',  '\34',  '\35',  '\36',  '\37',  '\40',  /*140-147*/
    '\41',  '\42',  '\43',  '\44',  '\45',  '\46',  '\47',  '\50',  /*150-157*/
    '\51',  '\52',  '\53',  '\54',  '\55',  '\56',  '\57',  '\60',  /*160-167*/
    '\61',  '\62',  '\63',  '\177', '\177', '\177', '\177', '\177', /*170-177*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*200-207*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*210-217*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*220-227*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*230-237*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*240-247*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*250-257*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*260-267*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*270-277*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*300-307*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*310-317*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*320-327*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*330-337*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*340-347*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*350-357*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*360-367*/
    '\177', '\177', '\177', '\177', '\177', '\177', '\177', '\177', /*370-377*/
  };
  int i, hi, len, start, nextstart, end, di;

  nextstart = index;
  start = index;
  len = strlen(uuestr);
  hi = 0;

  while (1)
  {
    last_data = 0;
    di = 0;
    /*---------------------------------------------------
    The following is used to read from stdin. I replaced
    it with reading from uuestr.
    ----------------------------------------------------- */
    #ifdef COMMENTED_OUT
    if (fgets (inout->ebuf, sizeof(inout->ebuf), stdin) == NULL)
    {
      fprintf(stderr, "Error: Short file.\n");
      exit(1);
    }
    #endif
    /*----------------------------------------------------*/
    if(len < start + UBUFSIZE)
      end = len;
    else
      end = start + UBUFSIZE;

    for(i = start; i < end; i++)
    {
      if(uuestr[i] == '\n')
      {
        nextstart = i + 1;
        inout->ebuf[hi++] = uuestr[i];
        break;
      }
      inout->ebuf[hi++] = uuestr[i];
    }
    inout->ebuf[hi] = 0;
    hi = 0;
    start = nextstart;
    /*----------------------------------------------------*/
    p = inout->ebuf;

    if (memcmp (inout->ebuf, "====", 4) == 0)
      break;

    if (last_data != 0)
    {
      sprintf(message, "UUE error: data following `=' padding.");
      put_msg(msgbox, message, RED, 0);
      return;
    }
    /* The following implementation of the base64 decoding might look
       a bit clumsy but I only try to follow the POSIX standard:
       ``All line breaks or other characters not found in the table
       [with base64 characters] shall be ignored by decoding
       software.''  */
    while (*p != '\n')
    {
      char c1, c2, c3;

      while ((b64_tab[*p] & '\100') != 0)
        if (*p == '\n' || *p++ == '=')
          break;
      if (*p == '\n')
      /* This leaves the loop.  */
        continue;
      c1 = b64_tab[*p++];
      while ((b64_tab[*p] & '\100') != 0)
        if (*p == '\n' || *p++ == '=')
        {
          put_msg(msgbox, "UUE error: illegal line.", RED, 0);
          return;
        }

      c2 = b64_tab[*p++];
      while (b64_tab[*p] == '\177')
        if (*p++ == '\n')
        {
          put_msg(msgbox, "UUE error: illegal line.", RED, 0);
          return;
        }

      if (*p == '=')
      {
        inout->in_buf[di++] = (c1 << 2 | c2 >> 4);
        last_data = 1;
        break;
      }
      c3 = b64_tab[*p++];
      while (b64_tab[*p] == '\177')
      if (*p++ == '\n')
      {
        put_msg(msgbox, "UUE error: illegal line.", RED, 0);
        return;
      }

      inout->in_buf[di++] = (c1 << 2 | c2 >> 4);
      inout->in_buf[di++] = (c2 << 4 | c3 >> 2);
      if (*p == '=')
      {
        last_data = 1;
        break;
      }
      else
        inout->in_buf[di++] = (c3 << 6 | b64_tab[*p++]);
    }
    /* At this point we have 1 line of decoded data. It is a line and
       not a UBUFSIZE chunk because the encoded data has carriage returns
       (that's not part of the data) that causes the inner loop to
       break out.  If the data you just decoded is NOT compressed,
       you would skip the following call to call_inflate() and instead
       put your output statement here. */
    inout->in_buf_len = di;
    inout->in_buf[di] = 0;
    if(call_inflate(inout, msgbox))
      return;
  }
}
/* -------------------------------------------------------------------------
        uue_inflate() Decode and uncompress a base64 encoded, deflated array.
   ------------------------------------------------------------------------- */
void uue_inflate(FILE *fpp, char *uuestr, msgbox_struct *msgbox)
{
  int index = 0;
  static inout_struct inout;
  int err;

  inout.out = fpp;
  err = inflate_init(&inout, msgbox);
  if(!err)
    index = read_base64_header(uuestr, msgbox);
  if(index)
    read_base64(index, uuestr, &inout, msgbox);
  if(!err)
    inflate_end(&inout, msgbox);
}
/* -------------------------------------------------------------------------
        print_test_page() Generates the Pup test Postscript page.
   ------------------------------------------------------------------------- */
void print_test_page(msgbox_struct *msgbox)
{
  extern FILE *fp;
  int y;

  fprintf(fp, "%%!PS-Adobe-3.0\n");
  fprintf(fp, "%%%%PS by pup %s Author: Matt Simpson matthewsimpson@home.com\n",
          VERSION);
  fprintf(fp, "%%%%Title: pup %s Test Page\n", VERSION);
  fprintf(fp, "%%%%The images in this program follows the format of "
                 "GIMP PostScript\n");
  fprintf(fp, "%%%%file plugin V 1.06 by Peter Kirchgessner.\n");
  fprintf(fp, "%%%%DocumentData: Clean7Bit\n");
  fprintf(fp, "%%%%Pages: 1\n");
  fprintf(fp, "%%%%EndComments\n");
  fprintf(fp, "%%%%Page: 1 1\n");
  fprintf(fp, "/Times-Roman findfont 32 scalefont setfont\n");
  fprintf(fp, "178 699 moveto\n");
  fprintf(fp, "(Linux)show\n");
  fprintf(fp, "178 656 moveto\n");
  fprintf(fp, "(Printer Test Page)show\n");
  fprintf(fp, "/Courier findfont 10 scalefont setfont\n");
  fprintf(fp, "72 597 moveto\n");
  fprintf(fp, "(Congratulations!)show\n");
  fprintf(fp, "72 573 moveto\n");
  fprintf(fp, "(If you can read this information, you have correctly "
               "installed your Linux)show\n"); 
  fprintf(fp, "72 561 moveto\n");
  fprintf(fp, "(compatible printer.)show\n");

  /* Host Info */
  y = 561;
  if(get_uinfo(USYSNAME) != NULL)
  {
    fprintf(fp, "\n%% Host info this page was generated from\n");
    y -= 26;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "/Times-Roman findfont 12 scalefont setfont\n");
    fprintf(fp, "(Host Information)show\n");
    y -= 16;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "/Courier findfont 10 scalefont setfont\n");
    fprintf(fp, "(Hostname:             %s)show\n", get_uinfo(UNODENAME));
    y -= 12;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "(Operating System:     %s", get_uinfo(USYSNAME));
    if(strcasecmp(get_uinfo(USYSNAME), "Linux"))
    {
      fprintf(fp, " (OK, this is really a %s Printer Test Page))show\n",
              get_uinfo(USYSNAME));
    }
    else
      fprintf(fp, ")show\n");
    y -= 12;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "(OS Release:           %s)show\n", get_uinfo(URELEASE));
    y -= 12;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "(OS Version:           %s)show\n", get_uinfo(UVERSION));
    y -= 12;
    fprintf(fp, "72 %d moveto\n", y);
    fprintf(fp, "(Host Type:            %s)show\n", get_uinfo(UMACHINE));
  }

  /* Printer Info */
  fprintf(fp, "\n%% Get printer info\n");
  y -=26;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "/Times-Roman findfont 12 scalefont setfont\n");
  fprintf(fp, "(Printer Information)show\n");
  y -= 16;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "/Courier findfont 10 scalefont setfont\n");
  fprintf(fp, "(Product:              )show\n");
  fprintf(fp, "statusdict /product known { statusdict /product get } "
              "{ ( ) } ifelse show\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(Printer Name:         )show\n");
  fprintf(fp, "currentsystemparams /PrinterName known\n");
  fprintf(fp, 
          "  { currentsystemparams /PrinterName get } { ( ) } ifelse show\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "/str 15 string def\n");
  fprintf(fp, "(Installed Memory:     )show\n");
  fprintf(fp, "currentsystemparams /RamSize known\n");
  fprintf(fp, "  { currentsystemparams /RamSize get str cvs } "
              "{ (unknown) } ifelse show\n");
  fprintf(fp, "( bytes)show\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(PS Language level:    )show\n");
  fprintf(fp, "systemdict /languagelevel known\n");
  fprintf(fp, "  { languagelevel str cvs } { ( ) } ifelse show\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(Interpreter Version:  )show\n");
  fprintf(fp, "systemdict /version known\n");
  fprintf(fp, "  { version str cvs } { ( ) } ifelse show\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(Revision:             )show\n");
  fprintf(fp, "statusdict /revision known\n");
  fprintf(fp, "  { statusdict /revision get str cvs } { ( ) } ifelse show\n");
  y -= 12;
  fprintf(fp, "\n");
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "%%The following donated by Brian Hall brianw.hall@compaq.com\n");
  fprintf(fp, "(Supported font types: )show\n");
  fprintf(fp, "languagelevel 2 ge\n");
  fprintf(fp, 
    "  { (*) {str cvs show ( ) show} 100 string /FontType resourceforall\n");
  fprintf(fp,
    "  } { ( ) } ifelse\n\n");
  y -= 12;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(This printer has printed )show\n");
  fprintf(fp, "currentsystemparams /PageCount known\n");
  fprintf(fp, "  { currentsystemparams /PageCount get str cvs } "
              "{ (unknown) } ifelse show\n");
  fprintf(fp, "( pages to date.)show\n");
  y -= 24;
  fprintf(fp, "72 %d moveto\n", y);
  fprintf(fp, "(The following demostrates some capabilities of your printer."
              ")show\n");

  /* Circle and color chart */ 
  fprintf(fp, "\n%% Circle with radial lines\n");
  fprintf(fp, "0 setgray\n");
  fprintf(fp, "0.001 setlinewidth\n");
  fprintf(fp, "gsave\n");
  fprintf(fp, "158 242 translate\n");
  fprintf(fp, "newpath 0 0 50 0 360 arc stroke\n");
  fprintf(fp, "0 10 350 {\n");
  fprintf(fp, "  pop\n");
  fprintf(fp, "  10 rotate\n");
  fprintf(fp, "  0 0 moveto\n");
  fprintf(fp, "  0 50 lineto } for\n");
  fprintf(fp, "stroke\n");
  fprintf(fp, "grestore\n");
  fprintf(fp, "232 237 moveto\n");
  fprintf(fp, "(Circle with 10 Degree Radial Lines)show\n");
  fprintf(fp, "\n%% Color chart\n");
  fprintf(fp, "274 155 moveto\n");
  fprintf(fp, "/Times-Roman findfont 12 scalefont setfont\n");
  fprintf(fp, "(Color Extremes)show\n");
  fprintf(fp, "/outline { 0 setgray gsave stroke grestore } def\n");
  fprintf(fp, "/box {\n");
  fprintf(fp, "  40 0 rlineto\n");
  fprintf(fp, "  0 10 rlineto\n");
  fprintf(fp, "  -40 0 rlineto\n");
  fprintf(fp, "  closepath } def\n");
  fprintf(fp, "3 setlinewidth\n");
  fprintf(fp, "newpath 111 125 moveto box outline\n");
  fprintf(fp, "1 0 0 setrgbcolor fill\n");
  fprintf(fp, "newpath 161 125 moveto box outline\n");
  fprintf(fp, "0 1 0 setrgbcolor fill\n");
  fprintf(fp, "newpath 211 125 moveto box outline\n");
  fprintf(fp, "0 0 1 setrgbcolor fill\n");
  fprintf(fp, "newpath 261 125 moveto box outline\n");
  fprintf(fp, "0 setgray fill\n");
  fprintf(fp, "newpath 311 125 moveto box outline\n");
  fprintf(fp, "1 0 0 0 setcmykcolor fill\n");
  fprintf(fp, "newpath 361 125 moveto box outline\n");
  fprintf(fp, "0 1 0 0 setcmykcolor fill\n");
  fprintf(fp, "newpath 411 125 moveto box outline\n");
  fprintf(fp, "0 0 1 0 setcmykcolor fill\n");
  fprintf(fp, "newpath 461 125 moveto box outline\n");
  fprintf(fp, "0 0 0 1 setcmykcolor fill\n");
  fprintf(fp, "0 setgray\n");
  fprintf(fp, "/Times-Roman findfont 10 scalefont setfont\n");
  fprintf(fp, "126 110 moveto\n");
  fprintf(fp, "(R)show\n");
  fprintf(fp, "176 110 moveto\n");
  fprintf(fp, "(G)show\n");
  fprintf(fp, "226 110 moveto\n");
  fprintf(fp, "(B)show\n");
  fprintf(fp, "271 110 moveto\n");
  fprintf(fp, "(BLK)show\n");
  fprintf(fp, "326 110 moveto\n");
  fprintf(fp, "(C)show\n");
  fprintf(fp, "376 110 moveto\n");
  fprintf(fp, "(M)show\n");
  fprintf(fp, "426 110 moveto\n");
  fprintf(fp, "(Y)show\n");
  fprintf(fp, "476 110 moveto\n");
  fprintf(fp, "(K)show\n");

  /* Tux image */
  fprintf(fp, "\n%% Begin Tux Image\n");
  fprintf(fp, "%%%%BeginProlog\n");
  fprintf(fp, "5 dict begin\n");
  fprintf(fp, "%%%%EndProlog\n");
  fprintf(fp, "gsave\n");
  fprintf(fp, "%% Translate for offset\n");
  fprintf(fp, "64.000000 650.800000 translate\n");
  fprintf(fp, "%% Translate to begin of first scanline\n");
  fprintf(fp, "0.000000 79.200000 translate\n");
  fprintf(fp, "79.200000 -79.200000 scale\n");
  fprintf(fp, "%% Variable to keep one line of raster data\n");
  fprintf(fp, "/scanline 192 3 mul string def\n");
  fprintf(fp, "%% Image geometry\n");
  fprintf(fp, "192 192 8\n");
  fprintf(fp, "%% Transformation matrix\n");
  fprintf(fp, "[ 192 0 0 192 0 0 ]\n");
  fprintf(fp, "{ currentfile scanline readhexstring pop } false 3\n");
  fprintf(fp, "colorimage\n");
  uue_inflate(fp, tux_image, msgbox);

  /* pup logo */
  fprintf(fp, "\n%% Begin Pup Image\n"); 
  fprintf(fp, "grestore\n");
  fprintf(fp, "gsave\n");
  fprintf(fp, "76.000000 72.000000 translate\n");
  fprintf(fp, "0.000000 10.080000 translate\n");
  fprintf(fp, "10.080000 -10.080000 scale\n");
  fprintf(fp, "/scanline1 32 3 mul string def\n");
  fprintf(fp, "%% Image geometry\n");
  fprintf(fp, "32 32 8\n");
  fprintf(fp, "%% Transformation matrix\n");
  fprintf(fp, "[ 32 0 0 32 0 0 ]\n");
  fprintf(fp, "{ currentfile scanline1 readhexstring pop } false 3\n");
  fprintf(fp, "colorimage\n");
  uue_inflate(fp, dog_image, msgbox);

  fprintf(fp, "grestore\n");
  fprintf(fp, "1 0 0 setrgbcolor\n");
  fprintf(fp, "/Courier-BoldOblique findfont 14 scalefont setfont\n");
  fprintf(fp, "72 76.5 moveto\n");
  fprintf(fp, "(pup)show\n");
  fprintf(fp, "0 setgray\n");
  fprintf(fp, "/Helvetica findfont 6 scalefont setfont\n");
  fprintf(fp, "107 76 moveto\n");
  fprintf(fp, "((c)%s Matt Simpson)show\n", CURYEAR);
  fprintf(fp, "/Helvetica findfont 20 scalefont setfont\n");
  fprintf(fp, "168 77 moveto\n");
  fprintf(fp, "(.)show\n");
  fprintf(fp, "/Helvetica findfont 6 scalefont setfont\n");
  fprintf(fp, "175 76 moveto\n");
  fprintf(fp, "(GNU Public License)show\n");

  fprintf(fp, "showpage\n");
  fprintf(fp, "%%%%Trailer\n");
  fprintf(fp, "%%%%EOF\n");
  /*fprintf(fp, "%c", 0x4);*/ /* ctrl-D */
  flush_dev(msgbox);
}
/* -------------------------------------------------------------------------
        print_ttf_page() Generates a TTF test Postscript page.
   ------------------------------------------------------------------------- */
void print_ttf_page(msgbox_struct *msgbox)
{
  extern FILE *fp;
  uue_inflate(fp, ttfps_str, msgbox);
  flush_dev(msgbox);
}
