/***************************************************************************
  main.cpp
  -------------------
  A brewing recipe calculator for Unix/X11
  -------------------
  Copyright 1999-2004, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
****************************************************************************/

#include <cstdlib>
#include <iostream>

#include <qapplication.h>
#include <qstring.h>

#include "controller.h"
#include "resource.h"

using namespace std;
using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// doversion()
// -----------
// Print out copyright and version info to stdout

void doversion()
{
    cout << PACKAGE << ' ' << VERSION << '\n';
    cout << ID_DESCRIPTION << '\n';
    cout << ID_COPYRIGHT << ' ' << ID_AUTHOR << " <" << ID_AUTHOR_EMAIL << ">\n";
}

//////////////////////////////////////////////////////////////////////////////
// dohelp()
// --------
// Print out help info to stdout

void dohelp()
{
    cout << "Usage: " << PACKAGE << " [options] [file]\n";

    // general options
    cout << "\nGeneral Options\n\n";
    cout << "file\n";
    cout << "    Open the specified file in QBrew\n";
    cout << "-help\n";
    cout << "    Print the command line options.\n";
    cout << "-version\n";
    cout << "    Print the version number of QBrew.\n";
    cout << endl;

    // misc options
    cout << "Miscellaneous Options\n\n";
    cout << "    These are options for the Qt library and the X Windows System.\n";
    cout << "-display display\n";
    cout << "    Sets the X display (default is $DISPLAY).\n";
    cout << "-geometry geometry\n";
    cout << "    Sets the client geometry of the main widget.\n";
    cout << "-fn or -font font\n";
    cout << "    Defines the application font.\n";
    cout << "-bg or -background color\n";
    cout << "    Sets the default background color and an application palette\n";
    cout << "    (light and dark shades are calculated).\n";
    cout << "-fg or -foreground color\n";
    cout << "    Sets the default foreground color.\n";
    cout << "-btn or -button color\n";
    cout << "    Sets the default button color.\n";
    cout << endl;

    // bug reports
    cout << "Report bugs to <" << PACKAGE_BUGREPORT << ">" << endl;
}

//////////////////////////////////////////////////////////////////////////////
// doargs()
// --------
// Process command line arguments that QApplication doesn't take care of

QString doargs(int argc, char **argv)
{
    for (int n=1; n<argc; n++) {
        QString arg = argv[n];
        // detect all common forms of help requests
        if ((arg.lower() == "-h") || (arg == "-help") ||
            (arg == "--help") || (arg == "-?")) {
            dohelp();
            return QString::null;
        } else if ((arg.lower() == "-v") || (arg == "-version") ||
                   (arg == "--version")) {
            doversion();
            return QString::null;
        } else if (arg[0] == '-') {
            // no other valid options
            cout << "Invalid parameter \"" << arg << "\"\n";
            dohelp();
            return QString::null;
        } else {
            // must be a filename
            return arg;
        }
    }
    return (AppResource::ID_DEFAULT_FILE);
}

//////////////////////////////////////////////////////////////////////////////
// main()
// ------
// Entry point of application

int main(int argc, char ** argv)
{
    QApplication app(argc, argv);

    // check for additional command line options before we go further
    QString filename = doargs(app.argc(), app.argv());
    if (filename.isEmpty()) return 0;

    Controller* controller = Controller::instance();
    controller->initialize(filename);
    app.setMainWidget(controller);

    controller->show();
    QObject::connect(qApp, SIGNAL(lastWindowClosed()), qApp, SLOT(quit()));
    return app.exec();
}
