/*
 * CP4D.h
 * $Id: CP4D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


#ifndef __CP4D_H
#define __CP4D_H



// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif

// Own
///////////
#include "CV4D.h"

// forward declarations
/////////////////////////
class CP3D;


/** This class provides a interface to 3D point 
  *
  * @author Michael Meissner
  */
class CP4D {   
public:
  static double epsilon;

  /** Default constructor.
   * The default value of the instantiated point will be (0.0,0.0,0.0,0.0). */
  CP4D() { m_ard[0] = 0.0;
           m_ard[1] = 0.0;
	   m_ard[2] = 0.0;
	   m_ard[3] = 0.0; };

  /** Construct new point.
   * The value of the point will be (rdX, rdY, rdZ, 1). */
  CP4D(double rdX, double rdY, double rdZ) { m_ard[0] = rdX;
                                             m_ard[1] = rdY;
					     m_ard[2] = rdZ;
					     m_ard[3] = 1; };

  /** Construct new point.
   * The value of the point will be (rdX, rdY, rdZ, rdW). */
  CP4D(double rdX, double rdY, double rdZ, double rdW) { m_ard[0] = rdX;
                                                         m_ard[1] = rdY;
							 m_ard[2] = rdZ;
							 m_ard[3] = rdW; };

  /** Copy constructor.
   * The parameters will be simply copied. */
  CP4D(const CP4D& Point) { m_ard[0] = Point[0];
                            m_ard[1] = Point[1];
                            m_ard[2] = Point[2];
                            m_ard[3] = Point[3]; };



  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////

  /** Cast operator to convert CP4D points to CP3D points.
    * Each component is devided by the fourth component. */
  operator CP3D() const;

  /** Assign one point to another.*/
  const CP4D& operator=(const CP4D&);

  /** Compares to points for being equal.
    * The result will be 'true'(1) if the two point are indentically
    * up to <= CP4D::epsilon for each component. Otherwise 'false'(0) 
    * will be returned. */ 
  int operator==(const CP4D&);

  /** Compares to points for not being equal.
    * Same as operator== but inverted.
    * @see operator==() */
  int operator!=(const CP4D&);

  /** Adds a vector to this point. */
  CP4D& operator+=(const CV4D&);

  /** Subtracts a vector from this point. */
  CP4D& operator-=(const CV4D&);

  /** Subtracts two points. The resulting vector is returned. */
  CV4D operator-(const CP4D&) const;

  /** Adds a vector to a point. */
  CP4D operator+(const CV4D&) const;

  /** Subtracts a vector from a point. */
  CP4D operator-(const CV4D&) const;

  /** Negates the point. */
  CP4D operator-() const;

  /** Returns the i-th component of the point.
    * The index goes from 0 to 3, 0 stands for the
    * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator[](int i) { return m_ard[i]; };

  /** Same as above but does not alter anything. */
  double operator[](int i) const { return m_ard[i]; };



  /////////////
  // METHODS //
  /////////////

  /** Converts a point to a vector.
    * It's implemented as 'get'-method to prevent
    * implicit casting by the compiler. */
  CV4D getCV4D() const;

  /** Returns the x-coordinate of the point. */
  double getX() const   { return m_ard[0]; };
  
  /** Returns the y-coordinate of the point. */
  double getY() const   { return m_ard[1]; };
  
  /** Returns the z-coordinate of the point. */
  double getZ() const   { return m_ard[2]; };
  
  /** Returns the w-coordinate of the point. */
  double getW() const   { return m_ard[3]; };

  /** Sets the x-coordinate of the point to 'rdX'. */
  void setX(double rdX) { m_ard[0] = rdX; };

  /** Sets the y-coordinate of the point to 'rdY'. */
  void setY(double rdY) { m_ard[1] = rdY; };

  /** Sets the z-coordinate of the point to 'rdZ'. */
  void setZ(double rdZ) { m_ard[2] = rdZ; };

  /** Sets the w-coordinate of the point to 'rdW'. */
  void setW(double rdW) { m_ard[3] = rdW; };

  /** Set the values of the point.
    * The value of the point will be (rdX, rdY, rdZ, rdW). */
  void setCoord(double rdX, double rdY, double rdZ, double rdW) { m_ard[0] = rdX; 
                                                                  m_ard[1] = rdY;
								  m_ard[2] = rdZ; 
								  m_ard[3] = rdW; 
                                                                  return; };

  /** Prints a point to the standard output. */ 
  void print() const;
  


  /////////////
  // FRIENDS //
  /////////////
 
  /** Same as above. But more useful for streams. */
  friend ostream& operator<<(ostream&, const CP4D&); 

  /** Reads a point from the given stream. */
  friend istream& operator>>(istream&, CP4D&); 

  /*
    CP4D operator*(double& a)
    { CP4D pnt;pnt.m_ard[0]=a*m_ard[0]; pnt.m_ard[1]=a*m_ard[1]; pnt.m_ard[2]=a*m_ard[2]; return pnt;}
    friend CP4D operator*(double a,CP4D& p1)
    { return CP4D(a*p1.m_ard[0], a*p1.m_ard[1], a*p1.m_ard[2]); }
    friend CP4D  AffinComb(CP4D&,double,CP4D&);
    friend double dist(CP4D&, CP4D&);
  */

  /** Returns the affine combination of the points and vectors. */
  friend inline CP4D AffinComb3(double, const CP4D&, 
                         	double, const CP4D&,
			 	double, const CP4D&);
 

private:
  friend class CP3D;

protected:
  double m_ard[4];
};





// Function   : operator=
// Parameters : const CP4D& cPoint
// Purpose    : 
// Comments   :
inline const CP4D& CP4D::operator=(const CP4D& cPoint) {
/*******************************************************************/
  m_ard[0] = cPoint.m_ard[0];
  m_ard[1] = cPoint.m_ard[1];
  m_ard[2] = cPoint.m_ard[2];
  m_ard[3] = cPoint.m_ard[3];

  return *this;
}



// Function   : getCV4D
// Parameters : 
// Purpose    : 
// Comments   :
inline CV4D CP4D::getCV4D() const
/*******************************************************************/
{
  return CV4D(m_ard[0], m_ard[1], m_ard[2], m_ard[3]);
}
  


// Function   : AffinComb3
// Parameters : double r, const CP4D& R, double s, const CP4D& S,
//              double t, const CP4D &T
// Purpose    : 
// Comments   :
inline CP4D AffinComb3(double r, const CP4D& R, double s, const CP4D& S,
                       double t, const CP4D &T)
/*******************************************************************/
{
  return CP4D(r*R[0] + s*S[0] + t*T[0],
              r*R[1] + s*S[1] + t*T[1],
              r*R[2] + s*S[2] + t*T[2],
              r*R[3] + s*S[3] + t*T[3]);
}

#endif
