/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>

#include "XUpdateScanner.h"
#include "OXProperties.h"

extern Properties properties;

namespace rfb {

unsigned int scanlines[32] = {  0, 16,  8, 24,
                                4, 20, 12, 28,
			       10, 26, 18,  2,
			       22,  6, 30, 14,
			        1, 17,  9, 25,
			        7, 23, 15, 31,
			       19,  3, 27, 11,
			       29, 13,  5, 21 };

XUpdateScanner::XUpdateScanner( Display *_dpy,
                                Window _window,
                                Framebuffer *_fb,
                                unsigned int _tileWidth,
                                unsigned int _tileHeight,
                                unsigned int _blockWidth,
                                unsigned int _blockHeight )
  : region(NULL)
  , dpy( _dpy )
  , mouseX(-1)
  , mouseY(-1)
  , window( _window )
  , fb( _fb )
  , tileWidth( 32 )
  , tileHeight( 32 )
  , blockWidth( _blockWidth )
  , blockHeight( _blockHeight )
  , count ( 0 )
  , countx ( 0 )
  , county ( 0 )
  , scanline( NULL )
  , tile( NULL )
{
//  tile = XGetImage( dpy, window, 0, 0, tileWidth, tileHeight, AllPlanes, ZPixmap );
    tile = XShmCreateImage( dpy,
                            DefaultVisual( dpy, 0 ),
                            fb->pixelFormat.bits_per_pixel,
                            ZPixmap,
                            NULL,
                            &shminfo_tile,
                            tileWidth,
                            tileHeight );
                                  
    shminfo_tile.shmid = shmget( IPC_PRIVATE,
                                 tile->bytes_per_line * tile->height,
                                 IPC_CREAT | 0777 );
    shminfo_tile.shmaddr = tile->data = (char *) shmat( shminfo_tile.shmid, 0, 0 );
    shminfo_tile.readOnly = False;
  
    XShmAttach( dpy, &shminfo_tile );

    tilesX = (fb->width + tileWidth - 1) / tileWidth;
    tilesY = (fb->height + tileHeight - 1) / tileHeight;
    tileMap = (char *) malloc( tilesX * tilesY );

    unsigned int i;
    for ( i = 0; i < tilesX * tilesY; i++ ) tileMap[i] = 0;

    scanline = XShmCreateImage( dpy,
                                DefaultVisual( dpy, 0 ),
                                fb->pixelFormat.bits_per_pixel,
                                ZPixmap,
                                NULL,
                                &shminfo_scanline,
                                fb->width,
                                1 );
                                  
    shminfo_scanline.shmid = shmget( IPC_PRIVATE,
                                     scanline->bytes_per_line,
                                     IPC_CREAT | 0777 );
    shminfo_scanline.shmaddr = scanline->data = (char *) shmat( shminfo_scanline.shmid, 0, 0 );
    shminfo_scanline.readOnly = False;
  
    XShmAttach( dpy, &shminfo_scanline );


    scanrow = XShmCreateImage( dpy,
                               DefaultVisual( dpy, 0 ),
                               fb->pixelFormat.bits_per_pixel,
                               ZPixmap,
                               NULL,
                               &shminfo_scanrow,
                               1,
                               fb->height );
                                  
    shminfo_scanrow.shmid = shmget( IPC_PRIVATE,
                                     scanrow->bytes_per_line * fb->height,
                                     IPC_CREAT | 0777 );
    shminfo_scanrow.shmaddr = scanrow->data = (char *) shmat( shminfo_scanrow.shmid, 0, 0 );
    shminfo_scanrow.readOnly = False;
  
    XShmAttach( dpy, &shminfo_scanrow );
    
};


XUpdateScanner::~XUpdateScanner()
{
  if (region) delete region;

  XShmDetach( dpy, &shminfo_scanrow );
  XDestroyImage( scanrow );
  shmdt( shminfo_scanrow.shmaddr );
  shmctl( shminfo_scanrow.shmid, IPC_RMID, 0 );

  XShmDetach( dpy, &shminfo_scanline );
  XDestroyImage( scanline );
  shmdt( shminfo_scanline.shmaddr );
  shmctl( shminfo_scanline.shmid, IPC_RMID, 0 );

  free( tileMap );
  XShmDetach( dpy, &shminfo_tile );
  XDestroyImage( tile );
  shmdt( shminfo_tile.shmaddr );
  shmctl( shminfo_tile.shmid, IPC_RMID, 0 );
}


void XUpdateScanner::checkTile( int x, int y, list< Hint > &hintList )
{
  unsigned int maxWidth = fb->width - x;
  unsigned int maxHeight = fb->height - y;
  if ( maxWidth > tileWidth ) maxWidth = tileWidth;
  if ( maxHeight > tileHeight ) maxHeight = tileHeight;

  if ( ( maxWidth == tileWidth ) && ( maxHeight == tileHeight ) ) {
    XShmGetImage( dpy, window, tile, x, y, AllPlanes );
  } else {
    XGetSubImage( dpy, window, x, y, maxWidth, maxHeight, AllPlanes, ZPixmap, tile, 0, 0 );
  }
  unsigned int line;
  bool changed = false;
  unsigned char *src = (unsigned char*) tile->data;
  unsigned char *dest = fb->data + y * fb->bytesPerLine + x * (fb->pixelFormat.bits_per_pixel >> 3);
  for ( line = 0; line < maxHeight; line++ ) {
    if ( memcmp( dest, src, maxWidth * (fb->pixelFormat.bits_per_pixel >> 3) ) ) {
      changed = true;
      memcpy( dest, src, maxWidth * (fb->pixelFormat.bits_per_pixel >> 3) );
    }
    src += tile->bytes_per_line;
    dest += fb->bytesPerLine;
  }
  if ( changed ) {
    Hint hint;
    hint.type = hintRefresh;
    hint.hint.refresh.x = x;
    hint.hint.refresh.y = y;
    hint.hint.refresh.width = maxWidth;
    hint.hint.refresh.height = maxHeight;
    hintList.push_back( hint );

  }
}



#define POINTER_WIDTH   12
#define POINTER_HEIGHT  18

char pointerMap[] =
"            "
" ..         "
" .+.        "
" .++.       "
" .+++.      "
" .++++.     "
" .+++++.    "
" .++++++.   "
" .+++++++.  "
" .++++++++. "
" .+++++.... "
" .++.++.    "
" .+. .++.   "
" ..  .++.   "
"      .++.  "
"      .++.  "
"       ..   "
"            ";


void XUpdateScanner::paintMousePointer(int x, int y) {

    int px, py, mx, my;
    mx = fb->width - x;
    if ( mx > POINTER_WIDTH ) mx = POINTER_WIDTH;
    my = fb->height - y;
    if ( my > POINTER_HEIGHT ) my = POINTER_HEIGHT;
    
    saveRegion(x,y,mx,my);

    for ( py = 0; py < my; py++ )
        for ( px = 0; px < mx; px++ ) {
            int ofs = (x + px) * (fb->pixelFormat.bits_per_pixel >> 3)
	            + (y + py) * fb->bytesPerLine;
            switch ( pointerMap[ px + py * POINTER_WIDTH ] ) {

              case '.':
  	        switch ( fb->pixelFormat.bits_per_pixel ) {
	          case  8:
	            fb->data[ofs]   = 0;
	            break;
	          case 16:
	            fb->data[ofs]   = 0;
	            fb->data[ofs+1] = 0;
	            break;
	          case 24:
	            fb->data[ofs]   = 0;
	            fb->data[ofs+1] = 0;
	            fb->data[ofs+2] = 0;
	            break;
	          case 32:
	            fb->data[ofs]   = 0;
	            fb->data[ofs+1] = 0;
	            fb->data[ofs+2] = 0;
	            fb->data[ofs+3] = 0;
	            break;
	        }
	        break;

              case '+':
	        switch ( fb->pixelFormat.bits_per_pixel ) {
	          case  8:
	            fb->data[ofs]   = 255;
	            break;
	          case 16:
	            fb->data[ofs]   = 255;
	            fb->data[ofs+1] = 255;
	            break;
	          case 24:
	            fb->data[ofs]   = 255;
	            fb->data[ofs+1] = 255;
	            fb->data[ofs+2] = 255;
	            break;
	          case 32:
	            fb->data[ofs]   = 255;
	            fb->data[ofs+1] = 255;
	            fb->data[ofs+2] = 255;
	            fb->data[ofs+3] = 255;
	            break;
	        }
	        break;

	      default: break;
            }
        }
}


void XUpdateScanner::saveRegion(int x, int y, int w, int h) {
    if (region) delete region;
    unsigned int psize = (fb->pixelFormat.bits_per_pixel >> 3);
    region = new unsigned char[w * h * psize];
    if (!region) return;
    int py;
    int i = 0;
    for ( py = y; py < y+h; py++ ) {
        memcpy( region+i,
                fb->data + py * fb->bytesPerLine + x * psize,
                w * psize );
        i += w * psize;
    }
    regionX = x;
    regionY = y;
    regionW = w;
    regionH = h;
}


void XUpdateScanner::restoreRegion() {
    if (!region) return;
    unsigned int psize = (fb->pixelFormat.bits_per_pixel >> 3);
    int py;
    int i = 0;
    for ( py = regionY; py < regionY+regionH; py++ ) {
        memcpy( fb->data + py * fb->bytesPerLine + regionX * psize,
                region+i,
                regionW * psize );
        i += regionW * psize;
    }
    delete region;
    region = NULL;
}


void XUpdateScanner::searchUpdates( list< Hint > &hintList )
{
  restoreRegion();

  unsigned int i;
  unsigned int x, y;
  unsigned int psize = (fb->pixelFormat.bits_per_pixel >> 3);


  for ( i = 0; i < tilesX * tilesY; i++ ) {
    tileMap[i] = 0;
  }

  if ( count % 2 ) {
      countx %= 32;
      y = scanlines[countx];
      while ( y < fb->height ) {
          XShmGetImage( dpy, window, scanline, 0, y, AllPlanes );
          x = 0;
          while ( x < fb->width ) {
              unsigned char *src = (unsigned char*) scanline->data + x * psize;
              unsigned char *dest = fb->data + y * fb->bytesPerLine + x * psize;
              if ( memcmp( dest, src, 32 * psize ) ) {
	        tileMap[ (x / tileWidth) + (y / tileHeight) * tilesX ] = 1;
              }
              x += 32;
          }
          y += 32;
      }
      countx++;
  }
  else {
      county %= 32;
      x = scanlines[county];
      while ( x < fb->width ) {
          XShmGetImage( dpy, window, scanrow, x, 0, AllPlanes );
          y = 0;
          unsigned char *src = (unsigned char*) scanrow->data;
          unsigned char *fbm = fb->data + x * psize;
          while ( y < fb->height ) {
              for ( i = 0; i < psize; i++ )
                  if ( fbm[i] - src[i] ) {
                      tileMap[ (x / tileWidth) + (y / tileHeight) * tilesX ] = 1;
                      int add = ((y + 32) & (~31)) - 1 - y;
                      i = psize;
                      src += add * scanrow->bytes_per_line;
                      fbm += add * fb->bytesPerLine;
                      y += add;
                  }
              src += scanrow->bytes_per_line;
              fbm += fb->bytesPerLine;
              y ++;
          }
          x += 32;
      }
      county++;
  }
  
  int tilec = 0;
  for ( y = 0; y < tilesY; y++ ) {
    for ( x = 0; x < tilesX; x++ ) {
      if ( tileMap[x + y * tilesX] > 0 ) {
        checkTile( x * tileWidth, y * tileHeight, hintList );
        tilec++;
      }
    }
  }

  cerr << "TILES: " << tilec << endl;
  
  if ( properties.showMousePointer ) {

      Window root_return, child_return;
      int root_x, root_y;
      int win_x, win_y;
      unsigned int mask_return;
      XQueryPointer( dpy, window, &root_return, &child_return,
                     &root_x, &root_y,
		     &win_x, &win_y,
		     &mask_return );

    // draw Pointer Map

      int mx, my;
      mx = fb->width - root_x;
      if ( mx > POINTER_WIDTH ) mx = POINTER_WIDTH;
      my = fb->height - root_y;
      if ( my > POINTER_HEIGHT ) my = POINTER_HEIGHT;
      
      paintMousePointer(root_x, root_y);

      if ( (root_x-mouseX) || (root_y-mouseY) ) {

          Hint hint;
          hint.type = hintRefresh;
          hint.hint.refresh.x = root_x;
          hint.hint.refresh.y = root_y;
          hint.hint.refresh.width = mx;
          hint.hint.refresh.height = my;
          hintList.push_back( hint );

          if ( mouseX < 0 ) mouseX = 0;
          if ( mouseY < 0 ) mouseY = 0;
          mx = fb->width - mouseX;
          if ( mx > POINTER_WIDTH ) mx = POINTER_WIDTH;
          my = fb->height - mouseY;
          if ( my > POINTER_HEIGHT ) my = POINTER_HEIGHT;
          hint.type = hintRefresh;
          hint.hint.refresh.x = mouseX;
          hint.hint.refresh.y = mouseY;
          hint.hint.refresh.width = mx;
          hint.hint.refresh.height = my;
          hintList.push_back( hint );
          
          mouseX = root_x;
          mouseY = root_y;
      }
  }

  count++;
}





} // namespace rfb


