/*
 * kickbot.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */
// implements the kickbot program

#include <rtstring.h>
#include <rtcollect.h>
#include <rtsystem.h>
#include <rtfile.h>
#include <rtini.h>

#include "robvars.h"

// always include this file as the last one
#include <rtmain.h>

namespace rt {

class KickbotArgs{
public:
	KickbotArgs() : compFolder(lrt::File::getCurrentFolder()),
		files(0), printHelp(false) {}

	lrt::File compFolder;
	lrt::Vector<lrt::String> files;
	bool printHelp;
};

KickbotArgs parseArgs(const lrt::Array<lrt::String>& args)
{
	KickbotArgs margs;

	for( int i = 0; i < args.length(); i++ )
	{
		if( args[i][0] == '-' )
		{
			lrt::String arg = args[i].substring(1);

			// "c"ompetition "f"older
			if(!arg.compareIgnoreCase("cf"))
			{
				i++;
				if(i >= args.length())
					lrt::System::exit(1, "Parameter missing for argument -" + arg);
				// ensure that a path was given
				lrt::String temp = args[i];
				if(!temp.endsWith("/") && !temp.endsWith(lrt::File::separator))
					temp += lrt::File::separatorChar;

				margs.compFolder = temp;
			}
			// "batch" mode (non-interactive)
			else if(!arg.compareIgnoreCase("batch"))
			{
				lrt::System::setInteractive(false);
			}
			// "h"elp or "help"
			else if((arg == "h") || (arg == "H") || !arg.compareIgnoreCase("help"))
			{
				margs.printHelp = true;
			}

#ifdef __SYMBIAN32__
			// current folder
			else if(!arg.compareIgnoreCase("cd"))
			{
				i++; // look at next arg
				if(i >= args.length())
					lrt::System::exit(1, "Parameter missing for argument -" + arg);
				lrt::String newCurrentFolder = args[i];
				// check that a path was given
				char c = newCurrentFolder[newCurrentFolder.length() - 1];
				// no path => append slash
				if(!(c == '/') && !(c == lrt::File::separatorChar))
					newCurrentFolder += lrt::File::separatorChar;

				// get competition folder
				lrt::String compFolder = margs.compFolder.getLocalName();

				// set current folder
				lrt::File::setCurrentFolder(lrt::File(newCurrentFolder));

				// set competition folder
				margs.compFolder = lrt::File(compFolder);
			}
#endif
			else
			{
				lrt::System::println("Warning: unknown parameter " + args[i]);
			}
		}
		else // file or file pattern
		{
			int starPos = args[i].indexOf('*'); // has got a star?
			if(starPos < 0) // no, so just add it
				margs.files += lrt::File(margs.compFolder, args[i]).getLocalName(margs.compFolder);
			else { // yes, so add all files matching the pattern
				lrt::Array<lrt::File> f(lrt::StarFilenameFilter::getFiles(
					lrt::File(margs.compFolder, args[i]).getName()));
				for(int fi = 0; fi < f.length(); fi++)
					margs.files += f[fi].getLocalName(margs.compFolder);
			}
		}
	}

	return margs;
}

void printHelp()
{
	lrt::System::println("Removes some robots from a tournament that was run");
	lrt::System::println("using the top mode (-t) of RoboTour. ");
	lrt::System::println("You will have to re-run makehtml after this operation.");
	lrt::System::println();

	lrt::System::println("Usage: kickbot [options] [prog1 [prog2 [...]]]");
	lrt::System::println(lrt::String("Version ") + version);
	lrt::System::println();
	lrt::System::println("Options:");
#ifdef __SYMBIAN32__
	lrt::System::println("  -cd F      Set current folder to F before running robotour.");
#endif
	lrt::System::println("  -cf   F     Sets the competition folder to F (default: current)");
 	lrt::System::println("  -h, -help   Print this help text."); 
	lrt::System::println("  progn       Any robot file name, or file pattern.");
	lrt::System::println("              These file names should be entered relative to the");
	lrt::System::println("              competition folder (see -cf). If no files are given,");
	lrt::System::println("              only the charts page, but no detailed pages is created.");
}

lrt::String getInfoFileName(const lrt::File& compFolder, const lrt::String& botname)
{
	lrt::String ret;
	if(botname.length() > 0) {
		ret = lrt::File(compFolder, botname).getName();
		ret[ret.length() - 1] = 'y';
	}
	else
		ret = lrt::File(compFolder, "y.y").getName();
	return ret;
}



} // namespace




/** Since rtmain.h includes system includes, we may not import any namespace in
  * this file. Only in this file, really. */

int rtMain(const lrt::Array<lrt::String>& args)
{
	lrt::System::setAppName("Kickbot");
	rt::KickbotArgs margs = rt::parseArgs(args);

	if(margs.printHelp)
	{
		rt::printHelp();
		return 0;
	}

	lrt::String chartsFile = lrt::File(margs.compFolder, "charts.xxx").getName();
	lrt::IniFile charts;
	charts.read(chartsFile);

	for(int f = 0; f < margs.files.length(); f++)
	{
		lrt::String& botname = margs.files[f];
		charts.removeSection(botname);

		lrt::IniFile info;
		lrt::String infoName = rt::getInfoFileName(margs.compFolder, botname);
		info.read(infoName);
		if(info.hasSection("info"))
		{
			info.getSection("info")["top"] = "no";
			info.write(infoName);
		}

	}

	charts.write(chartsFile);


	return 0;
}

