/*
 * robhtml.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements HTML functions (creating tournament HTMLs) 

#include "robhtml.h"

#include "robutils.h"

#include <rtsystem.h>
#include <rtini.h>
#include <rtfile.h>
#include <rtmath.h>
#include <rtchar.h>
#include <rtresource.h>
#include <rtstreams.h>

using namespace lrt;

namespace rt {

String HtmlCreator::defaultChartsIniName = "charts.xxx";
String HtmlCreator::defaultChartsHtmlName = "charts.html";
String HtmlCreator::cssName = "charts.css";

HtmlCreator::HtmlCreator(const String& competitionName, const File& competitionFolder,
	const File& outputFolder, bool createLinks, bool createDownloadLinks) : 
	competitionName(competitionName), competitionFolder(competitionFolder),
	outputFolder(outputFolder), createLinks(createLinks), createDownloadLinks(
	createDownloadLinks), myManager(false, Math::MAX_INT, competitionFolder.getName()),
	chartsHtmlName(defaultChartsHtmlName)
{
}

File HtmlCreator::getDetailedHtml(const String& botFileName)
{
	return File(outputFolder, botFileName + ".html");
}

bool HtmlCreator::createChartsHtml(bool recalc, const String& iniName, const String& htmlName)
{
	chartsHtmlName = htmlName;

	// first check if this can succeed (charts ini must be present of course)
	String chartsFileName(File(competitionFolder, iniName).getName());
	if(!File(chartsFileName).exists())
		return false; // no charts ini found

	if(!cssExists())
		if(!writeCss())
			System::println("warning: couldn't write css file");

	Vector<TopTourResult> store(0);

	// read the charts
	if(!recalc) 
	{
		IniFile charts;
		charts.read(chartsFileName);
		myManager.readCharts(charts, store);
		store.sort(TopTourResult::compareByPoints);
	}
	else // calc the charts
	{
		IniFile charts;
		charts.read(chartsFileName);
		myManager.calcCharts(charts, store);
		// is already sorted now
		NoBreakSection nbs; // protect new charts ini
			myManager.checkChartsLength(charts, store);
			myManager.updateCharts(charts, store);
			charts.write(chartsFileName);
		nbs.leave();
		// now finished updating, lets go html 
		store.clear();
		myManager.readCharts(charts, store);
		store.sort(TopTourResult::compareByPoints);
	}

	if(store.length() == 0)
		System::println("warning: empty charts or reading error?");

	// create the HTML
	NoBreakSection nbsProtectHtml; // create complete HTML always

	OutputStream* html = new FileOutputStream(File(outputFolder, htmlName));
	if(html->fail())
	{ delete html; return false; }

	html->write("<html>\n<head>\n <title>RoboTour Charts</title>\n");
	html->write(" <link rel=\"stylesheet\" href=\"" + cssName + "\">\n");
	html->write("</head>\n\n");
	html->write("<body class=\"bodybg\">\n");
	html->write("<h1>RoboTour Charts</h1>\n");
	html->write("<h3>" + toHtml(competitionName) + "</h3>\n");
	html->write("<table border=0 cellspacing=2 cellpadding=2 width=\"100%\">\n");
	html->write("<tr class=\"title\">\n");

	html->write("  <TD>\n" );
	html->write("    &nbsp;\n" );
	html->write("  </TD>\n" );
	html->write("  <TD>\n" );
	html->write("   Name<BR>\n" );
	html->write("   <SMALL>Last Updated</SMALL>\n" );
	html->write("  </TD>\n" );
	html->write("  <TD>\n" );
	html->write("   Author,<BR>Country\n" );
	html->write("  </TD>\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   WINS\n" );
	html->write("   </TD>\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   LOSES\n" );
	html->write("  </TD>\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   TIES\n" );
	html->write("  </TD>\n" );
	html->write("  <TD align=center width=100>\n" );
	html->write("   SCORE\n" );
	html->write("  </TD>\n" );
	html->write(" </TR>\n" );

	for(int b = 0; b < store.length(); b++)
	{
		TopTourResult& res = store[b];
		String filename    = File(competitionFolder, res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');
		String programname = toHtml(res.headers["name"].value);
		String author      = toHtml(res.headers["author"].value);
		String country     = toHtml(res.headers["country"].value);
		String wins        = toHtml(String(res.wins));
		String looses      = toHtml(String(res.looses));
		String ties        = toHtml(String(res.ties));
		String points      = toHtml(String(res.points));
		String date        = toHtml(res.date.toString());
		String dethtml	   = getDetailedHtml(res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');

		html->write(" <TR class=\"topcur\">\n" );
		html->write("  <TD>\n" );
		html->write(String("    ") + (b+1) + ".\n");
		html->write("  </TD>\n" );
		html->write("  <TD>\n" );
		if(createLinks)
		  html->write("   <A HREF=\"" + dethtml + "\">" + programname + "</A>\n");
		else
		  html->write("   " + programname + "\n");

		if(createDownloadLinks)
		  html->write("   <BR><SMALL>" + date + "&nbsp;&nbsp;<A HREF=\"" + filename + "\">Download</A></SMALL>\n");
		else
		  html->write("   <BR><SMALL>" + date + "&nbsp;&nbsp;</SMALL>\n");
		html->write("  </TD>\n" );
		html->write("  <TD>\n" );
		html->write("   " +author+ ",<BR>" +country+ "\n");
		html->write("  </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " +wins+ "\n");
		html->write("   </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " +looses+ "\n");
		html->write("  </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " +ties+ "\n");
		html->write("  </TD>\n" );
		html->write("  <TD align=center width=100>\n" );
		html->write("   " +points+ "\n");
		html->write("  </TD>\n" );
		html->write(" </TR>\n" );
	}

	html->write("</TABLE>\n");
	html->write("</BODY>\n");
	html->write("</HTML>");

	html->close();
	delete html;

	return true;
}

bool HtmlCreator::createDetailedHtml(const String& botFileName)
{
	if(!cssExists())
		if(!writeCss())
			System::println("warning: couldn't write css file");

	String infoFile = myManager.getInfoFileName(botFileName);
	TopInfo info(infoFile);
	info.read();

	// link to the bot file
	String botLink = File(competitionFolder, botFileName).
		getLocalName(outputFolder).replace(File::separatorChar, '/');
	// link to the main charts html page
	String chartsLink = File(outputFolder, chartsHtmlName).
		getLocalName(outputFolder).replace(File::separatorChar, '/');

	// create the HTML
	NoBreakSection nbsProtectHtml; // create complete HTML always

	OutputStream* html = new FileOutputStream(getDetailedHtml(botFileName));
	if(html->fail())
	{ delete html; return false; }

	html->write("<html>\n<head>\n <title>RoboTour Detailed Simulation Results</title>\n");
	html->write(" <link rel=\"stylesheet\" href=\"" + cssName + "\">\n");
	html->write("</head>\n\n");
	html->write("<body class=\"bodybg\">\n");

	html->write("<H1>Detailed Simulation Results for " + toHtml(info.headers["name"].value) + "</H1>\n\n");

	html->write("<H3>" + toHtml(competitionName) + "</H3>\n\n" );

	html->write("<TABLE WIDTH=\"100%\">\n");
	html->write(" <TR>" );
	html->write("  <TD class=\"detleft\" WIDTH=\"70%\">\n");
	html->write("   <BIG><B>" + toHtml(info.headers["name"].value) + "</B></BIG><BR>\n");
	html->write("   " + toHtml(info.headers["author"].value) + "<BR>\n");
	html->write("   " + toHtml(info.headers["country"].value) + "<BR>\n");
	html->write("  </TD>\n" );
	html->write("  <TD class=\"detright\">" );
	html->write("   <SMALL>last changed:<BR>\n" );
	html->write("   " + toHtml(info.date.toString()) + "<BR><BR>\n");
	if(createDownloadLinks)
	  html->write("   <A HREF=\"" + botLink + "\">Download " + toHtml(botLink) + "</A></SMALL>\n");
	else
	  html->write("   File name: " + toHtml(botLink) + "</SMALL>\n");

	html->write("  </TD>" );
	html->write(" </TR>" );
	html->write("</TABLE>\n\n" );

	// big table
	html->write("<TABLE BORDER=\"0\" CELLSPACING=\"2\" CELLPADDING=\"2\" WIDTH=\"100%\">\n");
	html->write(" <TR class=\"title\">\n" );
	html->write("  <TD>\n" );
	html->write("   Opponent<BR>\n" );
	html->write("   <SMALL>Last Updated</SMALL>\n" );
	html->write("  </TD>\n" );
	html->write("  <TD>\n" );
	html->write("   Author,<BR>Country\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   WINS\n" );
	html->write("   </TD>\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   LOSES\n" );
	html->write("  </TD>\n" );
	html->write("  <TD align=center width=50>\n" );
	html->write("   TIES\n" );
	html->write("  </TD>\n" );
	html->write(" </TR>\n" );

	Vector<TopTourResult> store(0);
	readDetailed(botFileName, store);
	store.sort(TopTourResult::compareByDate);

	for(int o = 0; o < store.length(); o++)
	{
		TopTourResult& res = store[o];
		String filename    = File(competitionFolder, res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');
		String programname = toHtml(res.headers["name"].value);
		String author      = toHtml(res.headers["author"].value);
		String country     = toHtml(res.headers["country"].value);
		String wins        = toHtml(String(res.wins));
		String looses      = toHtml(String(res.looses));
		String ties        = toHtml(String(res.ties));
		String points      = toHtml(String(res.points));
		String date        = toHtml(res.date.toString());
		String dethtml	   = getDetailedHtml(res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');

		String cssClass;
		if(res.top == "yes")
		{
			if(res.id == res.infoId) // is bot played version (id) == bot current version (infoId)?
				cssClass = "topcur";
			else
				cssClass = "topold";
		}
		else
		{
			if(res.id == res.infoId)
				cssClass = "badcur";
			else
				cssClass = "badold";
		}

		html->write(" <TR class=\"" + cssClass + "\">\n");
		html->write("  <TD>\n" );
		if(createLinks)
			html->write("   <A HREF=\"" + dethtml + "\">" + programname + "</A><BR>\n");
		else
			html->write("   " + programname + "<BR>\n");
		if(createDownloadLinks)
			html->write("   <SMALL>"+ date + "&nbsp;&nbsp;<A HREF=\""+ filename +"\">Download</A></SMALL>\n");
		else
			html->write("   <SMALL>"+ date + "</SMALL>\n");
		html->write("  </TD>\n" );
		html->write("  <TD>\n" );
		html->write("   " + author + ",<BR>" + country + "\n");
		html->write("  </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " + wins + "\n");
		html->write("   </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " + looses + "\n");
		html->write("  </TD>\n" );
		html->write("  <TD align=center width=50>\n" );
		html->write("   " + ties + "\n");
		html->write("  </TD>\n" );
		html->write(" </TR>\n" );
	}

	// print legend
	html->write("</TABLE>\n");
	html->write("<A HREF=\"" + chartsLink + "\">Back to the charts</A><BR>\n\n");
	html->write("<h3>Legend:</h3>\n");
	html->write("<TABLE cellpadding=2 cellspacing=0 border=0>\n");
	html->write("<TR><TD class=\"topcur\">\n");
	html->write("  Opponent is in the charts\n");
	html->write("</TD></TR>\n");
	html->write("<TR><TD class=\"topold\">\n");
	html->write("  Former version of a charts bot\n");
	html->write("</TD></TR>\n");
	html->write("<TR><TD class=\"badcur\">\n");
	html->write("  Opponent isn't good enough for the charts\n");
	html->write("</TD></TR>\n");
	html->write("<TR><TD class=\"badold\">\n");
	html->write("  Former version of a bot, which is not in the charts\n");
	html->write("</TD></TR>\n");
	html->write("</TABLE>\n");
	html->write("</BODY>\n");
	html->write("</HTML>\n");

	html->close();
	delete html;

	return true;
}

bool HtmlCreator::readDetailed(const String& botFileName, Vector<TopTourResult>& store)
{
	store.clear();

	String botFile = File(competitionFolder, botFileName).getName();
	String resFile = myManager.getResultFileName(botFile);
	IniFile results;
	if(!results.read(resFile))
		System::println("warning: result file " + resFile + " not found.");


	IniFile::Iterator iter = results.begin();
	while(iter.hasElement())
	{
		const IniSection& sec = iter.get().getValue();
		String infoFile = myManager.getInfoFileName(iter.get().getKey());
		TopInfo info(infoFile);
		info.read();
		TopTourResult res(info.headers, iter.get().getKey());
		res.wins   = sec["wins"].intValue(0);
		res.looses = sec["looses"].intValue(0);
		res.ties   = sec["ties"].intValue(0);
		res.id     = sec["id"].intValue(0);
		res.date = info.date;
		res.top = info.top;
		res.infoId = info.id.intValue(0);
		store += res;
		++iter;
	}

	return true;
}

String HtmlCreator::toHtml(const String& text)
{
	String ret;

	int tlen = text.length();
	const char* data = text.cStr();
	for(int t = 0; t < tlen; t++)
	{
		char c = data[t];
		if(Char::isLetter(c) || Char::isDigit(c) ||
			(c==' ') || (c=='.') || (c==',')) // char is safe
			ret += c;
		else 
		{
			ret += "&#";
			ret += String(((int)((unsigned char)c)));
			ret += ';';
		}
	}

	return ret;
}

bool HtmlCreator::writeCss()
{
	InputStream* cssRes = Resource::getResource(cssName);
	if((cssRes == 0) || (cssRes->fail())) // not found
		return false;

	NoBreakSection nbsProtectCss; // write complete CSS always

	OutputStream* cssOut = new FileOutputStream(File(outputFolder, cssName));
	if(cssOut->fail()) // can't write
		return false;
	// copy stream
	int c;
	while((c = cssRes->read()) >= 0)
		cssOut->write(c);
	delete cssRes;
	delete cssOut;

	return true;
}

bool HtmlCreator::cssExists()
{
	return File(outputFolder, cssName).exists();
}


} // namespace

